/*
 * ProcessWindowController.m
 *
 * Copyright (C) 2008 MikuInstaller Project. All rights reserved.
 * http://mikuinstaller.sourceforge.jp/
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#import <CoreFoundation/CoreFoundation.h>
#import "ProcessWindowController.h"

static NSString * const ProcessToolbarIdentifier = @"Process Toolbar";
static NSString * const AbortToolbarItemIdentifier = @"ToolbarItem Abort";
static NSString * const ReportToolbarItemIdentifier = @"ToolbarItem Report";

static NSString *
localize(NSString *key)
{
	return NSLocalizedStringFromTable(key, @"ProcessManager", nil);
}

@interface ProcessWindowTableViewDelegate : NSObject {
	id dataSourceOwner;
	SEL dataSourceSelector;
	NSMutableDictionary *rowHeightCache;
}
- (id)initWithOwner:(id)owner selector:(SEL)selector;
- (NSArray *)dataSources;
@end

@implementation ProcessWindowTableViewDelegate

- (id)initWithOwner:(id)owner selector:(SEL)selector;
{
	if ((self = [super init])) {
		dataSourceOwner = [owner retain];
		dataSourceSelector = selector;
		rowHeightCache = [[NSMutableDictionary alloc]
				  initWithCapacity:255];
	}
	return self;
}

- (void)dealloc
{
	[dataSourceOwner release];
	[rowHeightCache release];
	[super dealloc];
}

- (NSArray *)dataSources
{
	return [dataSourceOwner performSelector:dataSourceSelector];
}

- (NSInteger)numberOfRowsInTableView:(NSTableView *)tableView
{
	return [[self dataSources] count];
}

- (id)tableView:(NSTableView *)tableView
objectValueForTableColumn:(NSTableColumn *)tableColumn
	    row:(int)row
{
	NSArray *sources = [self dataSources];

	if (!(sources && row >= 0 && row < [sources count]))
		return nil;

	return [[sources objectAtIndex:row]
		objectForKey:[tableColumn identifier]];
}

- (void)tableViewColumnDidResize:(NSNotification *)notification
{
	NSTableView *tableView = [notification object];
	NSArray *sources = [self dataSources];

	[rowHeightCache removeAllObjects];
	[tableView noteHeightOfRowsWithIndexesChanged:
		   [NSIndexSet indexSetWithIndexesInRange:
			       NSMakeRange(0,[sources count])]];
}

- (CGFloat)tableView:(NSTableView *)tableView heightOfRow:(NSInteger)row
{
	NSNumber *key, *value;
	NSString *text;
	NSTableColumn *column;
	NSTextFieldCell *textFieldCell;

	key = [NSNumber numberWithInteger:row];
	value = [rowHeightCache objectForKey:key];
	if (value)
		return [value floatValue];

	column = [[tableView tableColumns] lastObject];
	textFieldCell = [column dataCellForRow:row];
	text = [self tableView:tableView
     objectValueForTableColumn:column
			   row:row];
	[textFieldCell setStringValue:text];

	NSRect bounds = NSMakeRect(0, 0, [column width], 10000.0);
	NSSize size = [textFieldCell cellSizeForBounds:bounds];

	[rowHeightCache setObject:[NSNumber numberWithFloat:size.height]
			   forKey:key];
	return size.height;
}

@end

@implementation ProcessWindowController

- (id)initWithProcessManager:(ProcessManager *)manager
{
	self = [self initWithWindowNibName:@"ProcessWindow"];
	if (self) {
		processManager = manager; /* avoid reference loop */
	}
	return self;
}

- (void)windowDidLoad
{
	NSToolbar *toolbar;
	ProcessWindowTableViewDelegate *delegate;

	toolbar = [[[NSToolbar alloc]
		    initWithIdentifier:ProcessToolbarIdentifier]
		   autorelease];
	[toolbar setAllowsUserCustomization:YES];
	[toolbar setAutosavesConfiguration:NO];
	[toolbar setDisplayMode:NSToolbarDisplayModeIconAndLabel];
	[toolbar setDelegate:self];
	[[self window] setToolbar:toolbar];

	delegate = [[ProcessWindowTableViewDelegate alloc]
		    initWithOwner:processManager
		    selector:@selector(currentProcesses)];
	[processTableView setDataSource:delegate];
	[processTableView setDelegate:delegate];
	[processTableView setAutosaveTableColumns:NO];

	delegate = [[ProcessWindowTableViewDelegate alloc]
		    initWithOwner:processManager
		    selector:@selector(logMessages)];
	[logTableView setDataSource:delegate];
	[logTableView setDelegate:delegate];
	[logTableView setAutosaveTableColumns:NO];

	[[self window] center];

	NSString *myName = [[NSBundle mainBundle]
			    objectForInfoDictionaryKey:
			    (NSString *)kCFBundleNameKey];
	NSString *title = [[self window] title];
	[[self window]
	 setTitle:[title stringByAppendingFormat:@" - %@", myName]];
}

- (NSArray *)toolbarAllowedItemIdentifiers:(NSToolbar *)toolbar
{
	return [NSArray arrayWithObjects:
			AbortToolbarItemIdentifier,
			NSToolbarFlexibleSpaceItemIdentifier,
			ReportToolbarItemIdentifier,
			nil];
}

- (NSArray *)toolbarDefaultItemIdentifiers:(NSToolbar *)toolbar
{
	return [self toolbarAllowedItemIdentifiers:toolbar];
}

- (NSToolbarItem *)toolbar:(NSToolbar *)toolbar
     itemForItemIdentifier:(NSString *)itemIdentifier
 willBeInsertedIntoToolbar:(BOOL)willBeInserted
{
	NSString *label;
	NSToolbarItem *item = nil;

	if ([itemIdentifier isEqual:AbortToolbarItemIdentifier]) {
		item = [[[NSToolbarItem alloc]
			 initWithItemIdentifier:itemIdentifier]
			autorelease];
		label = localize(@"Abort");
		[item setLabel:label];
		[item setPaletteLabel:label];
		[item setToolTip:localize(@"Abort selected process.")];
		[item setImage: [NSImage imageNamed:@"StopScript.tiff"]];
		[item setTarget:self];
		[item setAction:@selector(abortSelectedProcess:)];
	}
	else if ([itemIdentifier isEqual:ReportToolbarItemIdentifier]) {
		item = [[[NSToolbarItem alloc]
			 initWithItemIdentifier:itemIdentifier]
			autorelease];
		label = localize(@"Make a Report");
		[item setLabel:label];
		[item setPaletteLabel:label];
		[item setToolTip:localize(@"Make a trouble report.")];
		[item setImage:[NSImage imageNamed:@"mailTypeASCII.tiff"]];
		[item setTarget:self];
		[item setAction:@selector(createReport:)];
	}
	return item;
}

- (BOOL)validateToolbarItem:(NSToolbarItem *)toolbarItem
{
	NSString *identifier = [toolbarItem itemIdentifier];

	if ([identifier isEqual:AbortToolbarItemIdentifier]) {
		return ([processTableView selectedRow] >= 0);
	}
	else if ([identifier isEqual:ReportToolbarItemIdentifier]) {
		return YES;
	}
	return NO;
}

- (void)abortSelectedProcess:(id)sender
{
	NSArray *currentProcesses = [processManager currentProcesses];
	NSInteger row = [processTableView selectedRow];
	NSParameterAssert(row >= 0 && row < [currentProcesses count]);

	[[currentProcesses objectAtIndex:row]
	 terminateProcessBy:@"abortSelectedProcess"
	 quiet:YES];
}

- (void)createReport:(id)sender
{
	NSSavePanel *sheet = [NSSavePanel savePanel];
	NSString *filename;
	NSArray *desktopPath;

	[sheet setRequiredFileType:@"txt"];

	filename = [[NSBundle mainBundle]
		    objectForInfoDictionaryKey:
		    (NSString*)kCFBundleNameKey];
	filename = [filename stringByAppendingString:@" Log"];

	desktopPath = NSSearchPathForDirectoriesInDomains(NSDesktopDirectory,
							  NSUserDomainMask,
							  YES);
	[sheet
	 beginSheetForDirectory:[desktopPath objectAtIndex:0]
	 file:filename
	 modalForWindow:[self window]
	 modalDelegate:self
	 didEndSelector:@selector(createReportPanelDidEnd:returnCode:contextInfo:)
	 contextInfo:NULL];
}

- (void)createReportPanelDidEnd:(NSSavePanel *)sheet
		     returnCode:(int)returnCode
		    contextInfo:(void *)contextInfo
{
	[sheet orderOut:self];

	if (returnCode != NSOKButton)
		return;

	NSError *error;
	BOOL result = [processManager saveReportToPath:[sheet filename]
						 error:&error];
	if (result)
		return;

	NSBeginCriticalAlertSheet(localize(@"Make a Report"),
				  nil,            /* default */
				  nil,            /* alternate */
				  nil,            /* other */
				  [self window],  /* docWindow */
				  nil,            /* delegate */
				  NULL,           /* didEndSelector */
				  NULL,           /* didDismissSelector */
				  NULL,           /* contextInfo */
				  @"%@",          /* format */
				  [error localizedDescription]);
}

- (void)currentProcessesDidChange
{
	[processTableView reloadData];
}

- (void)logMessagesDidChange
{
	[logTableView reloadData];
}

@end
