/*
 * PreferencesController.h
 *
 * Copyright (C) 2008 MikuInstaller Project. All rights reserved.
 * http://mikuinstaller.sourceforge.jp/
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#import "PreferencesController.h"

@interface NSObject (PreferencesProtocol)
- (NSDictionary *)environmentVariables;
- (void)relaunch:(NSDictionary *)env;
@end

@implementation PreferencesKeyValuePair

- (id)initWithValue:(NSString *)aValue
		key:(NSString *)aKey
	  container:(id)object
{
	if ((self = [super init])) {
		container = object;  /* avoid reference loop */
		key = [aKey retain];
		value = [aValue retain];
	}
	return self;
}

- (id)initWithContainer:(id)object
{
	if ((self = [super init])) {
		container = object;  /* avoid reference loop */
		[self setKey:nil];
		[self setValue:nil];
	}
	return self;
}

- (void)dealloc
{
	[key release];
	[value release];
	[super dealloc];
}

- (NSString *)value
{
	return value;
}

- (void)setValue:(NSString *)newValue
{
	if (!newValue)
		newValue = [NSString string];
	[value release];
	value = [newValue retain];
}

- (NSString *)key
{
	return key;
}

- (void)setKey:(NSString *)newKeyBase
{
	if (!newKeyBase)
		newKeyBase = @"NoName";

	NSString *newKey = newKeyBase;
	if (container) {
		NSInteger try = 0;
		NSArray *content = [container valueForKey:@"content"];
		NSEnumerator *e = [content objectEnumerator];
		id entry;

		while ((entry = [e nextObject])) {
			if (entry != self
			    && [newKey isEqual:[entry valueForKey:@"key"]])
				newKey = [NSString stringWithFormat:@"%@%ld",
						   newKeyBase, ++try];
		}
	}

	[key release];
	key = [newKey retain];
}

@end

@implementation PreferencesController

- (id)init
{
	return [super initWithWindowNibName:@"Preferences"];
}

- (void)windowDidLoad
{
	[[self window] center];
	NSSortDescriptor *sort =
		[[[NSSortDescriptor alloc] initWithKey:@"key" ascending:YES]
		 autorelease];
	[arrayController setSortDescriptors:[NSArray arrayWithObject:sort]];
}

- (void)load
{
	NSDictionary *env = [applicationController environmentVariables];
	NSMutableArray *content = [NSMutableArray
				   arrayWithCapacity:[env count]];
	NSEnumerator *e = [env keyEnumerator];
	PreferencesKeyValuePair *entry;
	NSString *key;

	while ((key = [e nextObject])) {
		entry = [[[PreferencesKeyValuePair alloc]
			  initWithValue:[env objectForKey:key]
			  key:key
			  container:arrayController]
			 autorelease];
		[content addObject:entry];
	}

	[arrayController setContent:content];
}

- (NSDictionary *)dump
{
	NSArray *src = [arrayController content];
	NSMutableDictionary *env = [NSMutableDictionary
				    dictionaryWithCapacity:[src count]];
	NSEnumerator *e = [src objectEnumerator];
	id entry;

	while ((entry = [e nextObject])) {
		[env setObject:[entry valueForKey:@"value"]
			forKey:[entry valueForKey:@"key"]];
	}

	return env;
}

- (void)showWindow:(id)sender
{
	[super showWindow:sender];
	[self load];
}

- (void)saveAndRelaunch:(id)sender
{
	NSAlert *alert =
		[NSAlert
		 alertWithMessageText:
		 NSLocalizedString(@"Are your sure you want to"
				   " apply the changes and restart"
				   " the application?", nil)
		 defaultButton:nil
		 alternateButton:NSLocalizedString(@"Cancel", nil)
		 otherButton:nil
		 informativeTextWithFormat:
		 NSLocalizedString(@"All current processes are terminated"
				   " forcely before restarting.", nil)];

	[alert
	 beginSheetModalForWindow:[self window]
	 modalDelegate:self
	 didEndSelector:@selector(RelaunchConfirm:returnCode:contextInfo:)
	 contextInfo:NULL];
}

- (void)RelaunchConfirm:(NSAlert *)alert
	     returnCode:(int)returnCode
	    contextInfo:(void *)contextInfo
{
	if (returnCode == NSAlertDefaultReturn) {
		[applicationController relaunch:[self dump]];
		[self close];
	}
}

- (void)newEntry:(id)sender
{
	NSMutableDictionary *newItem = [[[PreferencesKeyValuePair alloc]
					 initWithContainer:arrayController]
					autorelease];
	[arrayController addObject:newItem];
	NSUInteger row = [[arrayController arrangedObjects]
			  indexOfObject:newItem];
	[arrayController setSelectionIndex:row];
	[tableView editColumn:0 row:row withEvent:nil select:YES];
}

@end
