/*
 * Controller.m
 *
 * Copyright (C) 2008 MikuInstaller Project. All rights reserved.
 * http://mikuinstaller.sourceforge.jp/
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#import "Controller.h"
#import "ApplicationInfo.h"

#ifdef DEBUG
#define TRACE(args) NSLog args
#else
#define TRACE(args)
#endif

static NSString * const HelperApplicationName = @"NegiHelper";
static NSString * const MIKUBUNDLE = @"MIKUBUNDLE";
static NSString * const DefaultShell = @"/bin/sh";

@implementation Controller

static void
substituteMenuTitle(id menu, NSString *appName)
{
	NSMutableString *buf = [NSMutableString stringWithCapacity:256];

	[buf setString:[menu title]];
	[buf replaceOccurrencesOfString:@"@APPNAME@"
			     withString:appName
				options:NSLiteralSearch | NSBackwardsSearch
				  range:NSMakeRange(0, [buf length])];
	[menu setTitle:buf];

	if ([menu isKindOfClass:[NSMenu class]]) {
		NSInteger i, len;
		len = [menu numberOfItems];
		for (i = 0; i < len; i++)
			substituteMenuTitle([menu itemAtIndex:i], appName);
	} else {
		NSMenu *submenu = [menu submenu];
		if (submenu)
			substituteMenuTitle(submenu, appName);
	}
}

- (id)init
{
	if ((self = [super init])) {
		environmentVariables =
			[[NSMutableDictionary alloc] initWithCapacity:10];
		scriptCache =
			[[NSMutableDictionary alloc] initWithCapacity:10];
	}
	return self;
}

- (void)dealloc
{
	[[NSNotificationCenter defaultCenter] removeObserver:self];
	[environmentVariables release];
	[scriptCache release];
	[super dealloc];
}

- (NSDictionary *)environmentVariables
{
	return environmentVariables;
}

- (void)setEnvironmentVariables:(NSDictionary *)env
{
	[environmentVariables removeAllObjects];
	[scriptCache removeAllObjects];
	[environmentVariables addEntriesFromDictionary:env];

	if (![environmentVariables valueForKey:MIKUBUNDLE]) {
		NSBundle *wineBundle = [ApplicationInfo wineBundle];
		if (wineBundle) {
			[environmentVariables setObject:[wineBundle bundlePath]
						 forKey:MIKUBUNDLE];
		}
	}
}

- (void)awakeFromNib
{
	NSBundle *mainBundle = [NSBundle mainBundle];
	NSString *filename;

	[super init];

	[[NSNotificationCenter defaultCenter] 
	 addObserver:self
	 selector:@selector(currentProcessesDidChange:)
	 name:ProcessManagerDidChangeProcessesNotification
	 object:processManager];

	launched = NO;

	/* load configurations */
	filename = [mainBundle pathForResource:@"environment" ofType:@"plist"];
	if (filename) {
		[self setEnvironmentVariables:
		      [NSDictionary dictionaryWithContentsOfFile:filename]];
	}

	/* replace "@STARTAPP@" in application menu with bundle name. */
	substituteMenuTitle([NSApp mainMenu],
			    [[NSBundle mainBundle]
			     objectForInfoDictionaryKey:
			     (NSString *)kCFBundleNameKey]);
}

- (BOOL)isWindowVisible
{
	NSArray *windows = [NSApp windows];
	NSEnumerator *e = [windows objectEnumerator];
	NSWindow *window;

	while ((window = [e nextObject])) {
		if ([window isVisible])
			return YES;
	}
	return NO;
}

- (void)currentProcessesDidChange:(NSNotification *)notification
{
	if ([[processManager currentProcesses] count] == 0
	    && ![self isWindowVisible])
		[NSApp terminate:self];
}

- (void)appleScriptError:(NSDictionary *)error prefix:(id)prefix
{
	[processManager log:@"%@: AppleScript error: %@ at %@",
			prefix,
			[error objectForKey:NSAppleScriptErrorMessage],
			[error objectForKey:NSAppleScriptErrorRange]];
}

- (id)loadScript:(NSString *)name
{
	NSString *key, *src, *filename;
	NSDictionary *error;
	id result = nil;

	result = [scriptCache objectForKey:name];
	if (result)
		return result;

	key = [NSString stringWithFormat:@"$%@_APPLESCRIPT", name];
	src = [environmentVariables objectForKey:key];
	if (src) {
		NSAppleScript *script =
			[[[NSAppleScript alloc] initWithSource:src]
			 autorelease];
		if (script && ![script isCompiled]) {
			if (![script compileAndReturnError:&error])
				script = nil;
		}
		if (!script)
			[self appleScriptError:error prefix:filename];
		result = script;
	}

	if (!result) {
		key = [NSString stringWithFormat:@"$%@_SH", name];
		result = [environmentVariables objectForKey:key];
	}
	if (result)
		[scriptCache setObject:result forKey:name];

	return result;
}

- (void)startShellCommand:(NSString *)command
		     name:(NSString *)name
		arguments:(NSArray *)arguments
{
	NSMutableDictionary *env =
		[NSMutableDictionary dictionaryWithCapacity:10];
	NSEnumerator *e = [environmentVariables keyEnumerator];
	NSString *key;

	/* reject internal variables from environmentVariables */
	while ((key = [e nextObject])) {
		if (![key hasPrefix:@"$"])
			[env setObject:[environmentVariables objectForKey:key]
				forKey:key];
	}

	NSString *shell = [environmentVariables objectForKey:@"$SHELL"];
	if (!shell)
		shell = DefaultShell;

	NSArray *args = [NSArray arrayWithObjects:@"-c", command, name, nil];
	args = [args arrayByAddingObjectsFromArray:arguments];

	[processManager startProcess:shell
			   arguments:args
			 environment:env];
}

- (void)invoke:(NSString *)actionName arguments:(NSArray *)args
{
	id script;
	NSDictionary *error;

	TRACE((@"action: %@", actionName));

	script = [self loadScript:actionName];
	if (script && [script isKindOfClass:[NSAppleScript class]]) {
		TRACE((@"execute applescript %@", actionName));
		if (![(NSAppleScript *)script executeAndReturnError:&error])
			[self appleScriptError:error prefix:actionName];
		TRACE((@"execute applescript %@ finished", actionName));
	}
	else if (script) {
		[self startShellCommand:script name:actionName arguments:args];
	}
}

- (void)application:(NSApplication *)sender openFiles:(NSArray *)filenames
{
	NSEnumerator *e = [filenames objectEnumerator];
	NSString *filename;

	TRACE((@"openFiles: launched? %d\n", launched));

	while ((filename = [e nextObject])) {
		[self invoke:@"OPEN"
		   arguments:[NSArray arrayWithObject:filename]];
	}
	launched = YES;
}

- (void)relaunch:(NSDictionary *)env
{
	[self setEnvironmentVariables:env];
	[processManager terminateAll];
	[self invoke:@"LAUNCH" arguments:nil];
	[processManager showStatusWindow:self];
}

- (void)applicationDidFinishLaunching:(NSNotification *)notification
{
	TRACE((@"DidFinishLaunching: launched? %d\n", launched));

	if ([@"YES" isEqual:[environmentVariables objectForKey:@"$DEBUG"]]) {
		[preferencesWindow showWindow:self];
		return;
	}

	if (!launched)
		[self invoke:@"LAUNCH" arguments:nil];

	if ([[processManager currentProcesses] count] == 0)
		[NSApp terminate:self];
}

- (void)applicationDidBecomeActive:(NSNotification *)notification
{
	if ([self isWindowVisible])
		return;

	[self invoke:@"ACTIVE" arguments:nil];
}

- (void)applicationWillTerminate:(NSNotification *)notification
{
	[processManager terminateAll];
}

- (void)orderFrontAboutPanel:(id)sender
{
	NSDictionary *options;
	NSString *appName, *version;

	appName = [[NSBundle mainBundle]
		   objectForInfoDictionaryKey:(NSString *)kCFBundleNameKey];
	if (!appName)
		appName = [[NSProcessInfo processInfo] processName];
	if (![appName isEqual:HelperApplicationName])
		appName = [NSString stringWithFormat:@"%@ featuring %@",
				    HelperApplicationName, appName];

	version = [[NSBundle mainBundle]
		   objectForInfoDictionaryKey:
		   @"CFBundleGetInfoString"];

	NSString *creditsRtfPath =
		[[ApplicationInfo wineBundle]
		 pathForResource:@"Credits" ofType:@"rtf"];
	NSAttributedString *credits =
		creditsRtfPath
		? [[[NSAttributedString alloc]
		    initWithPath:creditsRtfPath
		    documentAttributes:nil]
		   autorelease]
		: nil;

	options = [NSDictionary dictionaryWithObjectsAndKeys:
				appName, @"ApplicationName",
				version, @"ApplicationVersion",
				credits, @"Credits",
				nil];

	[NSApp orderFrontStandardAboutPanelWithOptions:options];
}

@end
