/*
 * ProcessWindowController.h
 *
 * Copyright (C) 2008 MikuInstaller Project. All rights reserved.
 * http://mikuinstaller.sourceforge.jp/
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#import "ModalProcessController.h"

@implementation ModalProcessController

- (id)initWithProcessManager:(ProcessManager *)manager
{
	self = [self initWithWindowNibName:@"ModalProcess"];
	if (self) {
		processManager = [manager retain];
		currentProcess = nil;
	}
	return self;
}

- (void)dealloc
{
	[processManager release];
	[currentProcess release];
	[[NSNotificationCenter defaultCenter] removeObserver:self];
	[super dealloc];
}

- (void)windowDidLoad
{
	[[self window] center];
}

- (void)setupWindowForProcess:(Process *)process
		      message:(NSString *)msg
{
	NSParameterAssert(!currentProcess && process);
	[self window];
	[messageTextField setStringValue:msg];
	[progressIndicator startAnimation:self];
	//[[self window] makeKeyAndOrderFront:nil];
	currentProcess = [process retain];
}

- (int)runModalForProcess:(Process *)process
		  message:(NSString *)msg
{
	int result = -1;

	[self setupWindowForProcess:process message:msg];

	/*
	 * On Tiger, we cannot receive ProcessDidTerminateNotification in
	 * modal run loop. Use ProcessDidCloseOutputNotification instead.
	 */
	[[NSNotificationCenter defaultCenter]
	 addObserver:self
	 selector:@selector(processDidTerminate:)
	 name:ProcessDidCloseOutputNotification
	 object:currentProcess];

	@try {
		[NSApp runModalForWindow:[self window]];
		/* we assume that task will be finished as soon as
		 * its output is closed. */
		[[currentProcess task] waitUntilExit];
		result = [[currentProcess task] terminationStatus];
	} @finally {
		[self close];
		[progressIndicator stopAnimation:self];
		[[NSNotificationCenter defaultCenter]
		 removeObserver:self
		 name:ProcessDidCloseOutputNotification
		 object:currentProcess];
		[currentProcess release];
		currentProcess = nil;
	}
	return result;
}

- (void)processDidTerminate:(NSNotification *)notification
{
	[NSApp abortModal];
}

- (void)beginSheelForProcess:(Process *)process
		     message:(NSString *)msg
	      modalForWindow:(NSWindow *)docWindow
	       modalDelegate:(id)delegate
	      didEndSelector:(SEL)selector
		 contextInfo:(void *)contextInfo
{
	[self setupWindowForProcess:process message:msg];

	[[NSNotificationCenter defaultCenter]
	 addObserver:self
	 selector:@selector(processDidTerminateInSheet:)
	 name:ProcessDidTerminateNotification
	 object:currentProcess];
	[self retain];

	[NSApp beginSheet:[self window]
	   modalForWindow:docWindow
	    modalDelegate:delegate
	   didEndSelector:selector
	      contextInfo:contextInfo];
}

- (void)processDidTerminateInSheet:(NSNotification *)notification
{
	Process *proc = [notification object];
	int status = [[proc task] terminationStatus];

	[[NSNotificationCenter defaultCenter]
	 removeObserver:self
	 name:ProcessDidTerminateNotification
	 object:currentProcess];
	[currentProcess release];
	currentProcess = nil;

	@try {
		[NSApp endSheet:[self window] returnCode:status];
	} @finally {
		[self close];
		[self release];
	}
}

- (IBAction)abortProcess:(id)sender
{
	[currentProcess terminateProcessBy:@"abortProcess"
				     quiet:NO];
}

- (IBAction)displayProcessDetail:(id)sender
{
	[[processManager processWindow] showWindow:sender];
}

@end
