/*
 * Controller.m
 *
 * Copyright (C) 2008 MikuInstaller Project. All rights reserved.
 * http://mikuinstaller.sourceforge.jp/
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#import "Controller.h"
#import "ModalProcessController.h"
#import "ConfirmDialogController.h"
#import "PreferencesController.h"
#import "ApplicationInfo.h"

static NSString * const ShellCommand = @"/bin/sh";
static NSString * const DefaultWinePrefixName = @"default";

@implementation Controller

+ (void)initialize
{
	[self exposeBinding:@"applicationPath"];
	[self exposeBinding:@"wineBundlePath"];
	[self exposeBinding:@"wineBundleVersion"];
	[self exposeBinding:@"winePrefixBase"];
	[self exposeBinding:@"winePrefix"];

	[[NSUserDefaults standardUserDefaults]
	 registerDefaults:
	 [NSDictionary
	  dictionaryWithObjectsAndKeys:
	  [NSNumber numberWithBool:YES], @"setupStartMenuAfterExec",
	  [NSNumber numberWithBool:YES], @"openProcessStatusBeforeExecWine",
	  nil]];
}

- (void)dealloc
{
	[[NSNotificationCenter defaultCenter] removeObserver:self];
	[launchedWithFiles release];
	[super dealloc];
}

- (NSString *)applicationPath
{
	NSArray *path;
	NSString *filename;

	path = NSSearchPathForDirectoriesInDomains(
		NSAllApplicationsDirectory,
		NSUserDomainMask,
		YES);
	filename = [path objectAtIndex:0];
	return [filename stringByAppendingPathComponent:ApplicationName];
}

- (NSString *)wineBundlePath
{
	return [[ApplicationInfo wineBundle] bundlePath];
}

- (NSString *)wineBundleVersion
{
	NSBundle *bundle = [ApplicationInfo wineBundle];
	return [bundle objectForInfoDictionaryKey:@"CFBundleGetInfoString"];
}

- (NSString *)winePrefixBase
{
	return [ApplicationInfo pathForApplicationSupport:@"prefix"];
}

- (NSString *)winePrefix
{
	NSString *path =
		[[[NSUserDefaultsController sharedUserDefaultsController]
		  values]
		 valueForKey:@"winePrefix"];

	if (!path) {
		path = [[self winePrefixBase]
			stringByAppendingPathComponent:DefaultWinePrefixName];
	}
	return path;
}

- (void)setWinePrefix:(NSString *)path
{
	return [[[NSUserDefaultsController sharedUserDefaultsController] values]
		setValue:path forKey:@"winePrefix"];
}

#define SETENV(env, key, obj) \
	(([(env) objectForKey:(key)]) \
	 ? (void)0 : (void)[(env) setObject:(obj) forKey:(key)])

- (Process *)startBackend:(NSString *)name
		arguments:(NSArray *)arguments
	      environment:(NSDictionary *)environment
{
	NSMutableDictionary *env =
		[NSMutableDictionary dictionaryWithCapacity:10];
	if (environment)
		[env addEntriesFromDictionary:environment];

	/* NOTE: Once a NSBundle object is created, the object may be
	 * cached. To prevent to create unnecessary NSBundle objects,
	 * we add variables to given environment, not add environment
	 * to env. */
	SETENV(env, @"MIKUBUNDLE", [[ApplicationInfo wineBundle] bundlePath]);
	SETENV(env, @"WINEPREFIX", [self winePrefix]);
	SETENV(env, @"MIKUROOT", [ApplicationInfo
				  pathForApplicationSupport:nil]);
	SETENV(env, @"MIKUAPPDIR", [self applicationPath]);

	NSString *scriptPath = [[NSBundle mainBundle]
				pathForResource:@"backend" ofType:@"sh"];
	NSArray *args = [NSArray arrayWithObjects:scriptPath, name, nil];
	if (arguments)
		args = [args arrayByAddingObjectsFromArray:arguments];

	return [processManager startProcess:ShellCommand
				  arguments:args
				environment:env];
}

- (void)beginSheetWithStartBackend:(NSString *)name
			 arguments:(NSArray *)arguments
		       environment:(NSDictionary *)environment
			   message:(NSString *)message
		    modalForWindow:(NSWindow *)docWindow
		     modalDelegate:(id)delegate
		    didEndSelector:(SEL)didEndSelector
		       contextInfo:(void *)contextInfo
{
	Process *proc = [self startBackend:name
				 arguments:arguments
			       environment:environment];
	if (proc) {
		ModalProcessController *modalWindow =
			[[[ModalProcessController alloc]
			  initWithProcessManager:processManager]
			 autorelease];
		[modalWindow beginSheelForProcess:proc
					  message:message
				   modalForWindow:docWindow
				    modalDelegate:delegate
				   didEndSelector:didEndSelector
				      contextInfo:contextInfo];
	}
}

- (int)runModalWithStartBackend:(NSString *)name
		      arguments:(NSArray *)arguments
		    environment:(NSDictionary *)environment
			message:(NSString *)message
{
	Process *proc = [self startBackend:name
				 arguments:arguments
			       environment:environment];
	if (!proc)
		return -1;

	ModalProcessController *modalWindow =
		[[[ModalProcessController alloc]
		  initWithProcessManager:processManager]
		 autorelease];
	return [modalWindow runModalForProcess:proc
				       message:message];
}

- (BOOL)confirm:(NSString *)nibName
	newPath:(NSString *)newPath
     newVersion:(NSString *)newVersion
	oldPath:(NSString *)oldPath
     oldVersion:(NSString *)oldVersion
{
	ConfirmDialogController *dialog =
		[[[ConfirmDialogController alloc]
		  initWithWindowNibName:nibName]
		 autorelease];
	return [dialog runModalWithNewPath:newPath
				newVersion:newVersion
				   oldPath:oldPath
				oldVersion:oldVersion];
}

- (void)executeWine:(NSArray *)filenames
{
	if (![[NSFileManager defaultManager]
	      fileExistsAtPath:[self winePrefix]]) {
		int status =
			[self
			 runModalWithStartBackend:@"prefixcreate"
			 arguments:nil
			 environment:nil
			 message:
			 [NSString
			  stringWithFormat:
			  NSLocalizedString(@"Initializing WINEPREFIX %@...",
					    nil),
			  [[self winePrefix] lastPathComponent]]];
		if (status != 0)
			return;
	}

	if ([[NSUserDefaults standardUserDefaults]
	     boolForKey:@"openProcessStatusBeforeExecWine"])
		[processManager showStatusWindow:self];

	NSEnumerator *e = [filenames objectEnumerator];
	NSString *filename;

	while ((filename = [e nextObject])) {
		Process *proc =
			[self startBackend:@"open"
				 arguments:[NSArray arrayWithObject:filename]
			       environment:nil];

		[[NSNotificationCenter defaultCenter]
		 addObserver:self
		 selector:@selector(processDidTerminate:)
		 name:ProcessDidTerminateNotification
		 object:proc];
	}
}

- (void)processDidTerminate:(NSNotification *)notification
{
	Process *proc = [notification object];

	[[NSNotificationCenter defaultCenter]
	 removeObserver:self
	 name:ProcessDidTerminateNotification
	 object:proc];

	if ([[proc task] terminationStatus] != 0)
		return;
	if (![[NSUserDefaults standardUserDefaults]
	      boolForKey:@"setupStartMenuAfterExec"])
		return;

	NSString *winePrefix = [proc objectForKey:@"WINEPREFIX"];
	if (!winePrefix)
		return;

	[self
	 runModalWithStartBackend:@"addapp"
	 arguments:nil
	 environment:[NSDictionary dictionaryWithObject:winePrefix
						 forKey:@"WINEPREFIX"]
	 message:[NSString
		  stringWithFormat:
		  NSLocalizedString(@"Making application bundles of"
				    " WINEPREFIX %@...",
				    nil),
		  [winePrefix lastPathComponent]]];
}

- (void)applicationWillFinishLaunching:(NSNotification *)notification
{
	/*
	 * check and setup Wine.bundle.
	 * After this check, one of the following condition holds:
	 * - We can find MikuInstaller.app in Applications folder by
	 *   bundle identifier search, and its Wine.bundle is the
	 *   latest one, or
	 * - There is the latest Wine.bundle in
	 *   ~/Library/Application Support/MikuInstaller.
	 */
	NSString *BundleVersionKey = (NSString *)kCFBundleVersionKey;
	BOOL needUpdateProcess = NO;

	NSBundle *myBundle = [NSBundle mainBundle];

	/* search installed MikuInstaller.app. */
	NSBundle *appBundle = [ApplicationInfo applicationBundle];
	if (appBundle && [[appBundle bundlePath] hasPrefix:@"/Volumes/"])
		appBundle = nil;

	NSBundle *appWineBundle = nil;
	if (appBundle) {
		NSString *filename = [appBundle pathForResource:@"Wine"
							 ofType:@"bundle"];
		appWineBundle = filename
			? [NSBundle bundleWithPath:filename]
			: nil;
	}

	/* search individually-installed Wine.bundle.
	/* Don't use NSBundle for individual Wine.bundle now.
	 * NSBundle object may be cached. */
	NSString *userWineBundlePath =
		[ApplicationInfo pathForApplicationSupport:@"Wine.bundle"];
	NSString *userWineBundleInfoPlist =
		[[userWineBundlePath
		  stringByAppendingPathComponent:@"Contents"]
		 stringByAppendingPathComponent:@"Info.plist"];
	NSDictionary *userWineBundleInfo =
		[NSDictionary dictionaryWithContentsOfFile:
			      userWineBundleInfoPlist];
	NSString *userWineVersion =
		[userWineBundleInfo objectForKey:BundleVersionKey];

	NSString *myVersion =
		[myBundle objectForInfoDictionaryKey:BundleVersionKey];
	NSString *appWineVersion =
		[appWineBundle objectForInfoDictionaryKey:BundleVersionKey];

	/* check and install Wine.bundle. */
	if (userWineVersion && !appBundle
	    && [userWineVersion compare:myVersion] < 0) {
		/*
		 * Wine.bundle is installed individually, and the bundle
		 * is older than me. This means user have selected to
		 * install Wine.bundle without MikuInstaller.
		 */
		if ([self confirm:@"UpdateConfirm"
			  newPath:[myBundle bundlePath]
		       newVersion:myVersion
			  oldPath:userWineBundlePath
		       oldVersion:userWineVersion]) {
			[self
			 runModalWithStartBackend:@"uninstall"
			 arguments:nil
			 environment:[NSDictionary
				      dictionaryWithObject:userWineBundlePath
				      forKey:@"MIKUBUNDLE"]
			 message:
			 NSLocalizedString(@"Uninstalling Wine.bundle...",
					   nil)];
			needUpdateProcess = YES;
			userWineVersion = nil;
		}
	}
	if (userWineVersion && appBundle
	    && [userWineVersion compare:appWineVersion] <= 0) {
		/*
		 * Both MikuInstaller.app and individual Wine.bundle
		 * is installed, but individual Wine.bundle is older
		 * than MikuInstaller.app's one. You should remove
		 * individual Wine.bundle.
		 */
		if ([self confirm:@"RemoveConfirm"
			  newPath:[appBundle bundlePath]
		       newVersion:appWineVersion
			  oldPath:userWineBundlePath
		       oldVersion:userWineVersion]) {
			[self
			 runModalWithStartBackend:@"uninstall"
			 arguments:nil
			 environment:[NSDictionary
				      dictionaryWithObject:userWineBundlePath
				      forKey:@"MIKUBUNDLE"]
			 message:
			 NSLocalizedString(@"Uninstalling Wine.bundle...",
					   nil)];
			needUpdateProcess = YES;
			userWineVersion = nil;
		}
	}
	if (!userWineVersion && !appWineBundle) {
		/*
		 * Neither Wine.bundle nor MikuInstaller.app is installed,
		 * or individual Wine.bundle has been uninstalled by
		 * previous check. Install Wine.bundle individually.
		 */
		[self
		 runModalWithStartBackend:@"install"
		 arguments:[NSArray arrayWithObject:
				    [myBundle pathForResource:@"Wine"
						       ofType:@"bundle"]]
		 environment:[NSDictionary
			      dictionaryWithObject:userWineBundlePath
			      forKey:@"MIKUBUNDLE"]
		 message:
		 NSLocalizedString(@"Installing Wine.bundle...", nil)];
	}

	/* check whether Wine.bundle is installed correctly. */
	NSBundle *wineBundle = [ApplicationInfo wineBundle];

	if (!wineBundle) {
		NSLog(@"FATAL: no Wine.bundle.");
		[NSApp terminate:self];
		return;
	}

	NSString *wineBundleVersion =
		[wineBundle objectForInfoDictionaryKey:BundleVersionKey];

	if (![myVersion isEqual:wineBundleVersion]) {
		if (![self confirm:@"ContinueConfirm"
			   newPath:[myBundle bundlePath]
			newVersion:myVersion
			   oldPath:[wineBundle bundlePath]
			oldVersion:wineBundleVersion]) {
			[NSApp terminate:self];
			return;
		}
	}

	if (needUpdateProcess) {
		[self
		 runModalWithStartBackend:@"prefixupdate"
		 arguments:nil
		 environment:nil
		 message:
		 NSLocalizedString(@"Updating WINEPREFIXes and application"
				   " bundles...", nil)];
	}
}

- (void)application:(NSApplication *)sender openFiles:(NSArray *)filenames
{
	if (launchedWithFiles)
		[self executeWine:filenames];
	else
		launchedWithFiles = [filenames retain];
}

- (void)applicationDidFinishLaunching:(NSNotification *)notification
{
	if (launchedWithFiles)
		[self executeWine:launchedWithFiles];
	[launchedWithFiles release];
	launchedWithFiles = [[NSArray array] retain];
}

- (void)applicationWillTerminate:(NSNotification *)notification
{
	[processManager terminateAll];
}

- (void)openDocument:(id)sender
{
	NSOpenPanel *panel = [NSOpenPanel openPanel];
	NSInteger result;

	[panel setAllowsMultipleSelection: NO];
	[panel setCanChooseDirectories: NO];
	//[openPanel setAccessoryView: openArgumentsView];

	result = [panel runModalForTypes:nil];
	if (result == NSOKButton)
		[self executeWine:[NSArray arrayWithObject:[panel filename]]];
}

- (void)orderFrontAboutPanel:(id)sender
{
	NSString *creditsRtfPath =
		[[ApplicationInfo wineBundle]
		 pathForResource:@"Credits" ofType:@"rtf"];
	NSAttributedString *credits =
		creditsRtfPath
		? [[[NSAttributedString alloc]
		    initWithPath:creditsRtfPath
		    documentAttributes:nil]
		   autorelease]
		: nil;

	NSDictionary *options = [NSDictionary dictionaryWithObjectsAndKeys:
					      credits, @"Credits",
					      nil];

	[NSApp orderFrontStandardAboutPanelWithOptions:options];
}

@end
