//******************************************************************************
//
// MIDITrail / MTFontTexture
//
// フォントテクスチャクラス
//
// Copyright (C) 2010-2022 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "YNBaseLib.h"
#import "MTFontTexture.h"


//******************************************************************************
// コンストラクタ
//******************************************************************************
MTFontTexture::MTFontTexture(void)
{
	m_pFontName = nil;
	m_FontSize = 0;
	m_TexHeight = 0;
	m_TexWidth = 0;
}

//******************************************************************************
// デストラクタ
//******************************************************************************
MTFontTexture::~MTFontTexture(void)
{
	Clear();
}

//******************************************************************************
// クリア
//******************************************************************************
void MTFontTexture::Clear()
{
	[m_pFontName release];
	m_pFontName = nil;
	m_Texture.Release();
}

//******************************************************************************
// フォント設定
//******************************************************************************
int MTFontTexture::SetFont(
		NSString* pFontName,
		float fontSize,
		OGLCOLOR color,
		bool isForceFixedPitch
	)
{
	int result = 0;
	
	[pFontName retain];
	[m_pFontName release];
	m_pFontName = pFontName;
	m_FontSize = fontSize;
	m_Color = color;
	m_isForceFiexdPitch = isForceFixedPitch;
	
EXIT:;
	return result;
}

//******************************************************************************
// テクスチャ生成
//******************************************************************************
int MTFontTexture::CreateTexture(
		OGLDevice* pOGLDevice,
		NSString* pStr
	)
{
	int result = 0;
	NSColor* pTextColor = nil;
	NSFont* pFont = nil;
	NSMutableDictionary* pFontAttributes = nil;
	NSAttributedString* pAttributedString = nil;
	NSImage* pImage = nil;
	NSBitmapImageRep* pBitmap = nil;
	NSSize frameSize;
	NSInteger bmpWidth = 0;
	NSInteger bmpHeight = 0;
	NSString* pFormat = nil;
	NSString* pErrMsg = nil;
	
	//フォント
	pFont = [NSFont fontWithName:m_pFontName size:m_FontSize];
	if (pFont == nil) {
		pFormat = @"MIDITrail could not open the font '%@'.\nPlease launch Font Book. Is that font enabled?";
		pErrMsg = [NSString stringWithFormat:pFormat, m_pFontName];
		result = YN_SET_ERR(pErrMsg, 0, 0);
		goto EXIT;
	}
	
	//テキスト色
	pTextColor = [NSColor colorWithCalibratedRed:m_Color.r
										   green:m_Color.g
											blue:m_Color.b
										   alpha:m_Color.a];
	
	//フォント属性
	pFontAttributes = [NSMutableDictionary dictionary];
	[pFontAttributes setObject:pFont forKey:NSFontAttributeName];
	[pFontAttributes setObject:pTextColor forKey:NSForegroundColorAttributeName];
	
	//属性付き文字列
	pAttributedString = [[NSAttributedString alloc] initWithString:pStr attributes:pFontAttributes];
	
	//イメージ生成
	frameSize = [pAttributedString size];
	pImage = [[NSImage alloc] initWithSize:frameSize];
	
	//描画開始
	[pImage lockFocus];
	{
		//描画結果からビットマップ生成
		//  テクスチャ画像の最大サイズを超える場合はクリップする
		bmpWidth = (NSInteger)frameSize.width;
		bmpHeight = (NSInteger)frameSize.height;
		if (bmpWidth > (NSInteger)OGL_MAX_TEXTURE_SIZE) {
			bmpWidth = (NSInteger)OGL_MAX_TEXTURE_SIZE;
			//NSLog(@"WARNING: The texture image was clipped. width:%f", frameSize.width);
		}
		if (bmpHeight > (NSInteger)OGL_MAX_TEXTURE_SIZE) {
			bmpHeight = (NSInteger)OGL_MAX_TEXTURE_SIZE;
			//NSLog(@"WARNING: The texture image was clipped. height:%f", frameSize.height);
		}
		pBitmap = [[NSBitmapImageRep alloc] initWithBitmapDataPlanes:NULL		//プレーン：メモリ自動確保
														  pixelsWide:bmpWidth	//画像横サイズ（ピクセル）
														  pixelsHigh:bmpHeight	//画像縦サイズ（ピクセル）
													   bitsPerSample:8			//1ピクセルあたりの要素別ビット数
													 samplesPerPixel:4			//1ピクセルあたりの要素数：RGBA
															hasAlpha:YES		//透明度有無
															isPlanar:NO			//要素ごとのプレーンが並ぶ場合はYES
													  colorSpaceName:NSCalibratedRGBColorSpace	//色空間名
														bitmapFormat:0			//ビットマップフォーマット：alpha last, premultiplied, integer
														 bytesPerRow:bmpWidth*4	//1ラインごとのバイトサイズ
														bitsPerPixel:32			//1ピクセルあたりのビット数
				   ];
		
		//カレントコンテキストにビットマップを指定
		[NSGraphicsContext setCurrentContext:[NSGraphicsContext graphicsContextWithBitmapImageRep:pBitmap]];
		
		//アンチエイリアスを有効化
		[[NSGraphicsContext currentContext] setShouldAntialias:YES];
		
		//描画色をセット
		[pTextColor set];
		
		//描画
		[pAttributedString drawAtPoint:NSMakePoint(0, 0)];
	}
	//描画終了
	[pImage unlockFocus];
	
	//テクスチャ生成
	result = m_Texture.LoadBitmap(pOGLDevice, pBitmap);
	if (result != 0) goto EXIT;
	
	m_TexWidth = m_Texture.GetWidth();
	m_TexHeight = m_Texture.GetHeight();
	
EXIT:;
	//TODO: 破棄する必要があるか？
	//[pTextColor release];
	//[pAttributedString release];
	//[pImage release];
	//[pBitmap release];
	return result;
}

//******************************************************************************
// テクスチャポインタ取得
//******************************************************************************
OGLTexture* MTFontTexture::GetTexture()
{
	return &m_Texture;
}

//******************************************************************************
// テクスチャサイズ取得
//******************************************************************************
void MTFontTexture::GetTextureSize(
		unsigned int* pHeight,
		unsigned int* pWidth
	)
{
	*pHeight = m_TexHeight;
	*pWidth = m_TexWidth;
}


