//******************************************************************************
//
// Simple MIDI Library / SMFileReader
//
// WMIDIt@Cǂݍ݃NX
//
// Copyright (C) 2010-2013 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#include "StdAfx.h"
#include "YNBaseLib.h"
#include "SMFileReader.h"
#include "SMCommon.h"
#include "tchar.h"
#include "shlwapi.h"

using namespace YNBaseLib;

namespace SMIDILib {


//******************************************************************************
// RXgN^
//******************************************************************************
SMFileReader::SMFileReader(void)
{
	m_LogPath[0] = '\0';
	m_pLogFile = NULL;
	m_IsLogOut = false;
}

//******************************************************************************
// fXgN^
//******************************************************************************
SMFileReader::~SMFileReader(void)
{
}

//******************************************************************************
// Oo̓pXݒ
//******************************************************************************
int SMFileReader::SetLogPath(
		const TCHAR* pLogPath
	)
{
	int result = 0;
	errno_t eresult = 0;

	m_IsLogOut = false;

	if (pLogPath == NULL) {
		m_LogPath[0] = '\0';
	}
	else {
		eresult = _tcscpy_s(m_LogPath, MAX_PATH, pLogPath);
		if (eresult != 0) {
			result = YN_SET_ERR("Program error.", 0, 0);
			goto EXIT;
		}
	}

	if (_tcslen(m_LogPath) > 0) {
		m_IsLogOut = true;
	}

EXIT:;
	return result;
}

//******************************************************************************
// Standard MIDI File ̃[h
//******************************************************************************
int SMFileReader::Load(
		const TCHAR *pSMFPath,
		SMSeqData* pSeqData
	)
{
	int result = 0;
	unsigned long i = 0;
	HMMIO hFile = NULL;
	SMFChunkTypeSection chunkTypeSection;
	SMFChunkDataSection chunkDataSection;
	SMFChunkTypeSection chunkTypeSectionOfTrack;
	SMTrack* pTrack = NULL;

	if ((pSMFPath == NULL) || (pSeqData == NULL)) {
		result = YN_SET_ERR("Program error.", 0, 0);
		goto EXIT;
	}

	pSeqData->Clear();

	//Ot@CJ
	result = _OpenLogFile();
	if (result != 0 ) goto EXIT;

	//t@CJ
	hFile = mmioOpen((LPTSTR)pSMFPath, NULL, MMIO_READ);
	if (hFile == NULL) {
		result = YN_SET_ERR("File open error.", GetLastError(), 0);
		goto EXIT;
	}

	//wb_ǂݍ
	result = _ReadChunkHeader(hFile, &chunkTypeSection, &chunkDataSection);
	if (result != 0 ) goto EXIT;

	if ((chunkDataSection.format != 0) && (chunkDataSection.format != 1)) {
		//tH[}bg0,1ȊO͖Ή
		result = YN_SET_ERR("Unsupported SMF format.", chunkDataSection.format, 0);
		goto EXIT;
	}
	if ( chunkDataSection.ntracks == 0) {
		//f[^ُ
		result = YN_SET_ERR("Invalid data found.", 0, 0);
		goto EXIT;
	}
	if ( chunkDataSection.timeDivision == 0) {
		//f[^ُ
		result = YN_SET_ERR("Invalid data found.", 0, 0);
		goto EXIT;
	}
	if ((chunkDataSection.timeDivision & 0x80000000) != 0) {
		//\̏ꍇ̓f^^CԂƂ݂Ȃdl
		//ʓIłȂ̂ō̂ƂT|[gȂ
		result = YN_SET_ERR("Unsupported SMF format.", chunkDataSection.timeDivision, 0);
		goto EXIT;
	}

	pSeqData->SetSMFFormat(chunkDataSection.format);
	pSeqData->SetTimeDivision(chunkDataSection.timeDivision);

	for (i = 0; i < chunkDataSection.ntracks; i++) {
		//gbNwb_ǂݍ
		result = _ReadTrackHeader(hFile, i, &chunkTypeSectionOfTrack);
		if (result != 0 ) goto EXIT;

		//gbNCxgǂݍ
		result = _ReadTrackEvents(hFile, chunkTypeSectionOfTrack.chunkSize, &pTrack);
		if (result != 0 ) goto EXIT;

		result = pSeqData->AddTrack(pTrack);
		if (result != 0 ) goto EXIT;
		pTrack = NULL;
	}

	//gbN
	result = pSeqData->CloseTrack();
	if (result != 0 ) goto EXIT;

	//t@Co^
	pSeqData->SetFileName(PathFindFileName(pSMFPath));

EXIT:;
	if (hFile != NULL) {
		mmioClose(hFile, 0);
		hFile = NULL;
	}
	_CloseLogFile();
	return result;
}

//******************************************************************************
// SMFwb_ǂݍ
//******************************************************************************
int SMFileReader::_ReadChunkHeader(
		HMMIO hFile,
		SMFChunkTypeSection* pChunkTypeSection,
		SMFChunkDataSection* pChunkDataSection
	)
{
	int result = 0;
	long apiresult = 0;
	long offset = 0;

	//ʎqƃwb_f[^TCY̓ǂݍ
	apiresult = mmioRead(hFile, (HPSTR)pChunkTypeSection, sizeof(SMFChunkTypeSection));
	if (apiresult != sizeof(SMFChunkTypeSection)) {
		result = YN_SET_ERR("File read error.", GetLastError(), apiresult);
		goto EXIT;
	}

	//GfBAϊ
	_ReverseEndian(&(pChunkTypeSection->chunkSize), sizeof(unsigned long));

	//`FbN
	if (memcmp(pChunkTypeSection->chunkType, "MThd", 4) != 0) {
		result = YN_SET_ERR("Invalid data found.", 0, 0);
		goto EXIT;
	}
	if (pChunkTypeSection->chunkSize < sizeof(SMFChunkDataSection)) {
		result = YN_SET_ERR("Invalid data found.", pChunkTypeSection->chunkSize, 0);
		goto EXIT;
	}

	//wb_f[^̓ǂݍ
	apiresult = mmioRead(hFile, (HPSTR)pChunkDataSection, sizeof(SMFChunkDataSection));
	if (apiresult != sizeof(SMFChunkDataSection)) {
		result = YN_SET_ERR("File read error.", GetLastError(), apiresult);
		goto EXIT;
	}

	//GfBAϊ
	_ReverseEndian(&(pChunkDataSection->format), sizeof(unsigned short));
	_ReverseEndian(&(pChunkDataSection->ntracks), sizeof(unsigned short));
	_ReverseEndian(&(pChunkDataSection->timeDivision), sizeof(unsigned short));

	//w肳ꂽf[^TCY܂ŃXLbviÔ߁j
	if (pChunkTypeSection->chunkSize > sizeof(SMFChunkDataSection)) {
		offset = pChunkTypeSection->chunkSize - sizeof(SMFChunkDataSection);
		apiresult = mmioSeek(hFile, offset, SEEK_CUR);
		if (apiresult == -1) {
			result = YN_SET_ERR("File read error.", GetLastError(), apiresult);
			goto EXIT;
		}
	}

	result = _WriteLogChunkHeader(pChunkTypeSection, pChunkDataSection);
	if (result != 0) goto EXIT;

EXIT:;
	return result;
}

//******************************************************************************
// SMFgbNwb_̓ǂݍ
//******************************************************************************
int SMFileReader::_ReadTrackHeader(
		HMMIO hFile,
		unsigned long trackNo,
		SMFChunkTypeSection* pChunkTypeSection
	)
{
	int result = 0;
	long apiresult = 0;

	//ʎqƃwb_f[^TCY̓ǂݍ
	apiresult = mmioRead(hFile, (HPSTR)pChunkTypeSection, sizeof(SMFChunkTypeSection));
	if (apiresult != sizeof(SMFChunkTypeSection)) {
		result = YN_SET_ERR("File read error.", GetLastError(), apiresult);
		goto EXIT;
	}

	//GfBAϊ
	_ReverseEndian(&(pChunkTypeSection->chunkSize), sizeof(unsigned long));

	//`FbN
	if (memcmp(pChunkTypeSection->chunkType, "MTrk", 4) != 0) {
		result = YN_SET_ERR("Invalid data found.", 0, 0);
		goto EXIT;
	}

	result = _WriteLogTrackHeader(trackNo, pChunkTypeSection);
	if (result != 0) goto EXIT;

EXIT:;
	return result;
}

//******************************************************************************
// SMFgbNCxg̓ǂݍ
//******************************************************************************
int SMFileReader::_ReadTrackEvents(
		HMMIO hFile,
		unsigned long chunkSize,
		SMTrack** pPtrTrack
	)
{
	int result = 0;
	unsigned long readSize = 0;
	unsigned long deltaTime = 0;
	unsigned long offset = 0;
	unsigned char portNo = 0;
	bool isEndOfTrack = false;
	SMEvent event;
	SMTrack* pTrack = NULL;

	try {
		pTrack = new SMTrack();
	}
	catch (std::bad_alloc) {
		result = YN_SET_ERR("Could not allocate memory.", 0, 0);
		goto EXIT;
	}

	//o͐|[g̏l̓gbNPʂ0ԂƂ
	portNo = 0;

	m_PrevStatus = 0;
	while (readSize < chunkSize) {

		//f^^Cǂݍ
		result = _ReadDeltaTime(hFile, &deltaTime, &offset);
		if (result != 0) goto EXIT;
		readSize += offset;

		//Cxgǂݍ
		result = _ReadEvent(hFile, &event, &isEndOfTrack, &offset);
		if (result != 0) goto EXIT;
		readSize += offset;

		//o̓|[g̐؂ւmF
		if (event.GetType() == SMEvent::EventMeta) {
			if (event.GetMetaType() == 0x21) {
				SMEventMeta meta;
				meta.Attach(&event);
				portNo = meta.GetPortNo();
			}
		}

		//CxgXgɒǉ
		result = pTrack->AddDataSet(deltaTime, &event, portNo);
		if (result != 0) goto EXIT;

		//gbNI[
		if (isEndOfTrack) {
			if (readSize != chunkSize) {
				//f[^s
				result = YN_SET_ERR("Invalid data found.", readSize, chunkSize);
				goto EXIT;
			}
			break;
		}
	}

	*pPtrTrack = pTrack;
	pTrack = NULL;

EXIT:;
	delete pTrack;
	return result;
}

//******************************************************************************
// SMFf^^C̓ǂݍ
//******************************************************************************
int SMFileReader::_ReadDeltaTime(
		HMMIO hFile,
		unsigned long* pDeltaTime,
		unsigned long* pOffset
	)
{
	int result = 0;

	result = _ReadVariableDataSize(hFile, pDeltaTime, pOffset);
	if (result != 0) goto EXIT;

	result = _WriteLogDeltaTime(*pDeltaTime);
	if (result != 0) goto EXIT;

EXIT:;
	return result;
}

//******************************************************************************
// SMFϒf[^TCY̓ǂݍ
//******************************************************************************
int SMFileReader::_ReadVariableDataSize(
		HMMIO hFile,
		unsigned long* pVariableDataSize,
		unsigned long* pOffset
	)
{
	int result = 0;
	int i = 0;
	long apiresult = 0;
	unsigned char tmp = 0;

	*pVariableDataSize = 0;
	*pOffset = 0;

	for (i = 0; i < 4; i++){
		apiresult = mmioRead(hFile, (HPSTR)&tmp, sizeof(unsigned char));
		if (apiresult != sizeof(unsigned char)) {
			result = YN_SET_ERR("File read error.", GetLastError(), apiresult);
			goto EXIT;
		}

		*pOffset += sizeof(unsigned char);
		*pVariableDataSize = (*pVariableDataSize << 7) | (tmp & 0x7F);

		if ((tmp & 0x80) == 0) break;
	}

EXIT:;
	return result;
}

//******************************************************************************
// Cxg̓ǂݍ
//******************************************************************************
int SMFileReader::_ReadEvent(
		HMMIO hFile,
		SMEvent* pEvent,
		bool* pIsEndOfTrack,
		unsigned long* pOffset
	)
{
	int result = 0;
	long apiresult = 0;
	unsigned char tmp = 0;
	unsigned char status = 0;
	unsigned long offsetTmp = 0;
	*pIsEndOfTrack = false;
	*pOffset = 0;

	//Xe[^Xǂݍ
	apiresult = mmioRead(hFile, (HPSTR)&tmp, sizeof(unsigned char));
	if (apiresult != sizeof(unsigned char)) {
		result = YN_SET_ERR("File read error.", GetLastError(), apiresult);
		goto EXIT;
	}
	*pOffset += sizeof(unsigned char);

	//jOXe[^X̏ȗ`FbN
	//OMIDICxg݂Ă1byteŏʃrbg0Ȃȗ
	if ((m_PrevStatus != 0) && ((tmp & 0x80) == 0)) { 
		//ȗꂽ̂őOMIDICxg̃Xe[^Xp
		status = m_PrevStatus;

		//ǂݍ݈ʒu߂
		apiresult = mmioSeek(hFile, -1, SEEK_CUR);
		if (apiresult == -1) {
			result = YN_SET_ERR("File read error.", GetLastError(), apiresult);
			goto EXIT;
		}
		*pOffset -= 1;
	}
	else {
		status = tmp;
	}

	switch (status & 0xF0) {
		case 0x80:  //m[gIt
		case 0x90:  //m[gI
		case 0xA0:  //|tHjbNL[vbV[
		case 0xB0:  //Rg[`FW
		case 0xC0:  //vO`FW
		case 0xD0:  //`lvbV[
		case 0xE0:  //sb`xh
			//MIDICxg
			result = _ReadEventMIDI(hFile, status, pEvent, &offsetTmp);
			if (result != 0) goto EXIT;
			//jOXe[^Xȗ̂ߑOXe[^XƂċL
			m_PrevStatus = status;
			break;
		case 0xF0:
			if ((status == 0xF0) || (status == 0xF7)) {
				//SysExCxg
				result = _ReadEventSysEx(hFile, status, pEvent, &offsetTmp);
				if (result != 0) goto EXIT;
			}
			else if (status == 0xFF) {
				//^Cxg
				result = _ReadEventMeta(hFile, status, pEvent, pIsEndOfTrack, &offsetTmp);
				if (result != 0) goto EXIT;
			}
			else {
				//f[^s
				result = YN_SET_ERR("Invalid data found.", status, 0);
				goto EXIT;
			}
			break;
		default:
			//f[^s
			result = YN_SET_ERR("Invalid data found.", status, 0);
			goto EXIT;
	}
	*pOffset += offsetTmp;

EXIT:;
	return result;
}

//******************************************************************************
// MIDICxg̓ǂݍ
//******************************************************************************
int SMFileReader::_ReadEventMIDI(
		HMMIO hFile,
		unsigned char status,
		SMEvent* pEvent,
		unsigned long* pOffset
	)
{
	int result = 0;
	int apiresult = 0;
	unsigned char data[2];
	unsigned long size = 0;

	*pOffset = 0;

	//DATA1ǂݍ
	apiresult = mmioRead(hFile, (HPSTR)&(data[0]), sizeof(unsigned char));
	if (apiresult != sizeof(unsigned char)) {
		result = YN_SET_ERR("File read error.", GetLastError(), apiresult);
		goto EXIT;
	}
	*pOffset += sizeof(unsigned char);

	switch (status & 0xF0) {
		case 0x80:  //m[gIt
		case 0x90:  //m[gI
		case 0xA0:  //|tHjbNL[vbV[
		case 0xB0:  //Rg[`FW
		case 0xE0:  //sb`xh
			//DATA2ǂݍ
			apiresult = mmioRead(hFile, (HPSTR)&(data[1]), sizeof(unsigned char));
			if (apiresult != sizeof(unsigned char)) {
				result = YN_SET_ERR("File read error.", GetLastError(), apiresult);
				goto EXIT;
			}
			*pOffset += sizeof(unsigned char);
			size = 2;
			break;
		case 0xC0:  //vO`FW
		case 0xD0:  //`lvbV[
			//DATA2Ȃ
			size = 1;
			break;
		default:
			//f[^s
			result = YN_SET_ERR("Invalid data found.", status, 0);
			goto EXIT;
	}

	result = pEvent->SetMIDIData(status, data, size);
	if (result != 0) goto EXIT;

	result = _WriteLogEventMIDI(status, data, size);
	if (result != 0) goto EXIT;

EXIT:;
	return result;
}

//******************************************************************************
// SysExCxg̓ǂݍ
//******************************************************************************
int SMFileReader::_ReadEventSysEx(
		HMMIO hFile,
		unsigned char status,
		SMEvent* pEvent,
		unsigned long* pOffset
	)
{
	int result = 0;
	int apiresult = 0;
	unsigned long size = 0;
	unsigned char* pData = NULL;
	unsigned long offsetTmp = 0;
	*pOffset = 0;

	//ϒf[^TCYǂݍ
	result = _ReadVariableDataSize(hFile, &size, &offsetTmp);
	if (result != 0) goto EXIT;
	*pOffset += offsetTmp;

	try {
		pData = new unsigned char[size];
	}
	catch (std::bad_alloc) {
		result = YN_SET_ERR("Could not allocate memory.", 0, 0);
		goto EXIT;
	}

	//ϒf[^ǂݍ
	apiresult = mmioRead(hFile, (HPSTR)(pData), size);
	if (apiresult != size) {
		result = YN_SET_ERR("File read error.", GetLastError(), apiresult);
		goto EXIT;
	}
	*pOffset += size;

	result = pEvent->SetSysExData(status, pData, size);
	if (result != 0) goto EXIT;

	result = _WriteLogEventSysEx(status, pData, size);
	if (result != 0) goto EXIT;

EXIT:;
	delete [] pData;
	return result;
}

//******************************************************************************
// ^Cxg̓ǂݍ
//******************************************************************************
int SMFileReader::_ReadEventMeta(
		HMMIO hFile,
		unsigned char status,
		SMEvent* pEvent,
		bool* pIsEndOfTrack,
		unsigned long* pOffset
	)
{
	int result = 0;
	int apiresult = 0;
	unsigned long size = 0;
	unsigned char type = 0;
	unsigned char* pData = NULL;
	unsigned long offsetTmp = 0;
	*pIsEndOfTrack = false;
	*pOffset = 0;

	//ʂǂݍ
	apiresult = mmioRead(hFile, (HPSTR)&type, sizeof(unsigned char));
	if (apiresult != sizeof(unsigned char)) {
		result = YN_SET_ERR("File read error.", GetLastError(), apiresult);
		goto EXIT;
	}
	*pOffset += sizeof(unsigned char);

	//^Cxg
	switch (type) {
		            //  sizeiv:ϒf[^TCYj
		case 0x00:  //  2  V[PXԍ
		case 0x01:  //  v  eLXg
		case 0x02:  //  v  쌠\
		case 0x03:  //  v  V[PX^gbN
		case 0x04:  //  v  y햼
		case 0x05:  //  v  ̎
		case 0x06:  //  v  }[J[
		case 0x07:  //  v  L[|Cg
		case 0x08:  //  v  vO^F
		case 0x09:  //  v  foCX ^
		case 0x20:  //  1  MIDI`lvtBbNX
		case 0x21:  //  1  | [gw
		case 0x2F:  //  0  gbNI[
		case 0x51:  //  3  e|ݒ
		case 0x54:  //  5  SMPTE ItZbg
		case 0x58:  //  4  q̐ݒ
		case 0x59:  //  2  ̐ݒ
		case 0x7F:  //  v  V[PT胁^Cxg
			break;
		default:
			//m̎ʂłG[ɂ͂Ȃ
			// result = YN_SET_ERR("Invalid data found.", type, 0);
			// goto EXIT;
			break;
	}

	if (status == 0x2F) {
		*pIsEndOfTrack = true;
	}

	//ϒf[^TCYǂݍ
	result = _ReadVariableDataSize(hFile, &size, &offsetTmp);
	if (result != 0) goto EXIT;
	*pOffset += offsetTmp;

	//ϒf[^ǂݍ
	if (size > 0) {
		try {
			pData = new unsigned char[size];
		}
		catch (std::bad_alloc) {
			result = YN_SET_ERR("Could not allocate memory.", 0, 0);
			goto EXIT;
		}
		apiresult = mmioRead(hFile, (HPSTR)pData, size);
		if (apiresult != size) {
			result = YN_SET_ERR("File read error.", GetLastError(), apiresult);
			goto EXIT;
		}
		*pOffset += size;
	}

	result = pEvent->SetMetaData(status, type, pData, size);
	if (result != 0) goto EXIT;

	result = _WriteLogEventMeta(status, type, pData, size);
	if (result != 0) goto EXIT;

EXIT:;
	delete [] pData;
	return result;
}

//******************************************************************************
// GfBAϊ
//******************************************************************************
void SMFileReader::_ReverseEndian(
		void* pData,
		unsigned long size
	)
{
	unsigned char tmp;
	unsigned char* pHead = (unsigned char*)pData;
	unsigned char* pTail = pHead + size - 1;

	while (pHead < pTail) {
		tmp = *pHead;
		*pHead = *pTail;
		*pTail = tmp;
		pHead += 1;
		pTail -= 1;
	}

	return;
}

//******************************************************************************
// Ot@CI[v
//******************************************************************************
int SMFileReader::_OpenLogFile()
{
	int result = 0;
	errno_t eresult = 0;

	if (_tcslen(m_LogPath) == 0) goto EXIT;

	eresult = _tfopen_s(&m_pLogFile, m_LogPath, _T("w"));
	if (eresult != 0) {
		result = YN_SET_ERR("Log file open error.", 0, 0);
		goto EXIT;
	}

EXIT:;
	return result;
}

//******************************************************************************
// Ot@CN[Y
//******************************************************************************
int SMFileReader::_CloseLogFile()
{
	int result = 0;
	int eresult = 0;

	if (!m_IsLogOut) goto EXIT;

	eresult = fclose(m_pLogFile);
	if (eresult != 0) {
		result = YN_SET_ERR("Log file close error.", 0, 0);
		goto EXIT;
	}

	m_pLogFile = NULL;

EXIT:;
	return result;
}

//******************************************************************************
// Oo
//******************************************************************************
int SMFileReader::_WriteLog(char* pText)
{
	int result = 0;
	size_t size = 0;
	size_t eresult = 0;

	if (!m_IsLogOut) goto EXIT;

	size = strlen(pText);

	eresult = fwrite(pText, size, 1, m_pLogFile);
	if (eresult != size) {
		result = YN_SET_ERR("Log file write error.", size, eresult);
		goto EXIT;
	}

EXIT:;
	return result;
}

//******************************************************************************
// OóFt@Cwb_
//******************************************************************************
int SMFileReader::_WriteLogChunkHeader(
		SMFChunkTypeSection* pChunkTypeSection,
		SMFChunkDataSection* pChunkDataSection
	)
{
	int result = 0;
	char msg[256];

	if (!m_IsLogOut) goto EXIT;

	_WriteLog("--------------------\n");
	_WriteLog("File Header\n");
	_WriteLog("--------------------\n");
	_WriteLog("Chunk Type : MThd\n");
	sprintf_s(msg, 256, "Length     : %d\n", pChunkTypeSection->chunkSize);
	_WriteLog(msg);
	sprintf_s(msg, 256, "Format     : %d\n", pChunkDataSection->format);
	_WriteLog(msg);
	sprintf_s(msg, 256, "nTracks    : %d\n", pChunkDataSection->ntracks);
	_WriteLog(msg);
	sprintf_s(msg, 256, "Devision   : %d\n", pChunkDataSection->timeDivision);
	_WriteLog(msg);

EXIT:;
	return result;
}

//******************************************************************************
// OóFgbNwb_
//******************************************************************************
int SMFileReader::_WriteLogTrackHeader(
		unsigned long trackNo,
		SMFChunkTypeSection* pChunkTypeSection
	)
{
	int result = 0;
	char msg[256];

	if (!m_IsLogOut) goto EXIT;

	_WriteLog("--------------------\n");
	sprintf_s(msg, 256, "Track No.%d\n", trackNo);
	_WriteLog(msg);
	_WriteLog("--------------------\n");
	_WriteLog("Chunk Type : MTrk\n");
	sprintf_s(msg, 256, "Length     : %d\n", pChunkTypeSection->chunkSize);
	_WriteLog(msg);
	_WriteLog("Delta Time | Event\n");

EXIT:;
	return result;
}

//******************************************************************************
// OóFf^^C
//******************************************************************************
int SMFileReader::_WriteLogDeltaTime(
		unsigned long deltaTime
	)
{
	int result = 0;
	char msg[256];

	if (!m_IsLogOut) goto EXIT;

	sprintf_s(msg, 256, "% 10d | ", deltaTime);
	_WriteLog(msg);

EXIT:;
	return result;
}

//******************************************************************************
// OóFMIDICxg
//******************************************************************************
int SMFileReader::_WriteLogEventMIDI(
		unsigned char status,
		unsigned char* pData,
		unsigned long size
	)
{
	int result = 0;
	char* cmd = "";
	char msg[256];

	if (!m_IsLogOut) goto EXIT;

	switch (status & 0xF0) {
		case 0x80: cmd = "Note Off";				break;
		case 0x90: cmd = "Note On";					break;
		case 0xA0: cmd = "Polyphonic Key Pressure";	break;
		case 0xB0: cmd = "Control Change";			break;
		case 0xC0: cmd = "Program Change";			break;
		case 0xD0: cmd = "Channel Pressure";		break;
		case 0xE0: cmd = "PitchBend";				break;
		default:   cmd = "unknown";					break;
	}

	sprintf_s(msg, 256, "MIDI: ch.%d cmd=<%s>", (status & 0x0F), cmd);
	_WriteLog(msg);

	if (size == 2) {
		sprintf_s(msg, 256, " data=[ %02X %02X %02X ]\n", status, pData[0], pData[1]);
	}
	else {
		sprintf_s(msg, 256, " data=[ %02X %02X ]\n", status, pData[0]);
	}
	_WriteLog(msg);

EXIT:;
	return result;
}

//******************************************************************************
// OóFSysExCxg
//******************************************************************************
int SMFileReader::_WriteLogEventSysEx(
		unsigned char status,
		unsigned char* pData,
		unsigned long size
	)
{
	int result = 0;
	char msg[256];
	unsigned long i = 0;

	if (!m_IsLogOut) goto EXIT;

	sprintf_s(msg, 256, "SysEx: status=%02X size=%d data=[", status, size);
	_WriteLog(msg);

	for (i = 0; i < size; i++) {
		sprintf_s(msg, 256, " %02X", pData[i]);
		_WriteLog(msg);
	}
	_WriteLog(" ]\n");

EXIT:;
	return result;
}

//******************************************************************************
// OóF^Cxg
//******************************************************************************
int SMFileReader::_WriteLogEventMeta(
		unsigned char status,
		unsigned char type,
		unsigned char* pData,
		unsigned long size
	)
{
	int result = 0;
	char* cmd = "";
	char msg[256];
	unsigned long i = 0;

	if (!m_IsLogOut) goto EXIT;

	switch (type) {
		case 0x00: cmd = "Sequence Number";					break;
		case 0x01: cmd = "Text Event";						break;
		case 0x02: cmd = "Copyright Notice";				break;
		case 0x03: cmd = "Sequence/Track Name";				break;
		case 0x04: cmd = "Instrument Name";					break;
		case 0x05: cmd = "Lyric";							break;
		case 0x06: cmd = "Marker";							break;
		case 0x07: cmd = "Cue Point";						break;
		case 0x08: cmd = "Program Name";					break;
		case 0x09: cmd = "Device Name";						break;
		case 0x21: cmd = "Port Number (Undocumented)";		break;
		case 0x2F: cmd = "End of Track";					break;
		case 0x51: cmd = "Set Tempo";						break;
		case 0x54: cmd = "SMPTE Offset";					break;
		case 0x58: cmd = "Time Signature";					break;
		case 0x59: cmd = "Key Signature";					break;
		case 0x7F: cmd = "Sequencer-Specific Meta-Event";	break;
		default:   cmd = "<unknown>";						break;
	}

	sprintf_s(msg, 256, "Meta: status=%02X type=%02X<%s> size=%d data=[", status, type, cmd, size);
	_WriteLog(msg);

	for (i = 0; i < size; i++) {
		sprintf_s(msg, 256, " %02X", pData[i]);
		_WriteLog(msg);
	}
	_WriteLog(" ]\n");

EXIT:;
	return result;
}

} // end of namespace

