//******************************************************************************
//
// MIDITrail / MTFileOpenPanel
//
// ファイル選択パネルクラス
//
// Copyright (C) 2010 WADA Masashi. All Rights Reserved.
//
//******************************************************************************

#import "MTFileOpenPanel.h"
#import "YNBaseLib.h"


//******************************************************************************
// コンストラクタ
//******************************************************************************
MTFileOpenPanel::MTFileOpenPanel(void)
{
	m_pPanel = nil;
	m_pAccessoryView = nil;
	m_pLabel = nil;
	m_pEncodingPopUp = nil;
}

//******************************************************************************
// デストラクタ
//******************************************************************************
MTFileOpenPanel::~MTFileOpenPanel(void)
{
	[m_pPanel release];
	[m_pAccessoryView release];
	[m_pLabel release];
	[m_pEncodingPopUp release];
}

//******************************************************************************
// 初期化
//******************************************************************************
int MTFileOpenPanel::Initialize()
{
	int result = 0;
	
	//エンコーディングリスト初期化
	result = m_EncodingList.Initialize();
	if (result != 0) goto EXIT;
	
	//パネル生成
	m_pPanel = [NSOpenPanel openPanel];
	if (m_pPanel == nil) {
		result = YN_SET_ERR(@"Program error.", 0, 0);
		goto EXIT;
	}
	[m_pPanel retain];
	
	//アクセサリビュー生成
	result = _CreateAccessoryView();
	if (result != 0) goto EXIT;
	
EXIT:;
	return result;
}

//******************************************************************************
// エンコーディング選択
//******************************************************************************
int MTFileOpenPanel::SetSelectedEncodingName(
		const NSString* pEncodingName
	)
{
	int result = 0;
	unsigned long index = 0;
	unsigned long selectedIndex = 0;
	unsigned long targetId = 0;
	BOOL isFound = NO;
	
	//エンコーディングリストから指定されたエンコーディング名を検索
	for (index = 0; index < m_EncodingList.GetSize(); index++) {
		if ([pEncodingName isEqualToString:m_EncodingList.GetEncodingName(index)]) {
			selectedIndex = index;
			isFound = YES;
			break;
		}
	}
	
	//指定されたエンコーディング名が見つからなかった場合
	//必ず存在するはずのエンコーディングを選択状態とする
	if (!isFound) {
		targetId = GetDefaultEncodingId();
		for (index = 0; index < m_EncodingList.GetSize(); index++) {
			if (m_EncodingList.GetEncodingId(index) == targetId) {
				selectedIndex = index;
				isFound = YES;
				break;
			}
		}
	}
	
	//それでも見つからない場合はリスト先頭を選択状態とする
	if (!isFound) {
		selectedIndex = 0;
	}
	
	//エンコーディング選択ポップアップボタンの選択状態を設定
	[m_pEncodingPopUp selectItemAtIndex:selectedIndex];	
	
	return result;
}

//******************************************************************************
// ファイル選択パネル表示
//******************************************************************************
NSInteger MTFileOpenPanel::showModalWindow()
{
	NSInteger btn = 0;
	
	//ファイル選択パネル表示
	btn = [m_pPanel runModal];
	
	//NSLog(@"tag %d", [[m_pEncodingPopUp selectedItem] tag]);
	
	//ファイル選択時の処理
	if (btn == NSOKButton) {
		//ここでは何もしない
	}
	
EXIT:;
	return btn;
}

//******************************************************************************
// 選択ファイルパス取得
//******************************************************************************
const NSString* MTFileOpenPanel::GetSelectedFilePath()
{
	return [m_pPanel filename];
}

//******************************************************************************
// 選択エンコーディング名取得
//******************************************************************************
const NSString* MTFileOpenPanel::GetSelectedEncodingName()
{
	return [[m_pEncodingPopUp selectedItem] title];
}

//******************************************************************************
// 選択エンコーディングID取得
//******************************************************************************
unsigned long MTFileOpenPanel::GetSelectedEncodingId()
{
	return [[m_pEncodingPopUp selectedItem] tag];
}

//******************************************************************************
// アクセサリビュー生成
//******************************************************************************
int MTFileOpenPanel::_CreateAccessoryView()
{
	int result = 0;
	NSMenu* pMenu = nil;
	NSMenuItem* pItem = nil;
	unsigned long index = 0;
	
	//  アクセサリビュー
	//  +---------------------------------+
	//  | +--------+ +------------------+ |
	//  | | ラベル  | | ポップアップボタン  | |
	//  | +--------+ +------------------+ |
	//  +---------------------------------+
	
	//アクセサリビュー生成
	m_pAccessoryView = [[NSView alloc] initWithFrame:NSMakeRect(0, 0, 406, 30)];
	
	//ラベル生成
	m_pLabel = [[NSTextField alloc] initWithFrame:NSMakeRect(2,8,120,17)];
	[m_pLabel setEditable:NO];
	[m_pLabel setBezeled:NO];
	[m_pLabel setBackgroundColor:[NSColor controlColor]];
	[m_pLabel setStringValue:@"Default encoding"];
	
	//ポップアップボタン生成
	m_pEncodingPopUp = [[NSPopUpButton alloc] initWithFrame:NSMakeRect(124, 2, 280, 26) pullsDown:NO];
	
	//メニュー生成
	pMenu = [[[NSMenu alloc] initWithTitle:@"Encoding"] autorelease];
	
	//エンコーディング一覧をメニューに登録
	for (index = 0; index < m_EncodingList.GetSize(); index++) {
		//項目生成
		pItem = [[NSMenuItem alloc] initWithTitle:(m_EncodingList.GetEncodingName(index))
										   action:nil
									keyEquivalent:@""];
		[pItem setTag:(m_EncodingList.GetEncodingId(index))];
		//メニューに項目を登録
		[pMenu addItem:pItem];
		[pItem release];
	}
	
	//ポップアップボタンにメニューを登録
	[m_pEncodingPopUp setMenu:pMenu];
	
	//エンコーディング選択ポップアップボタンの選択状態を設定
	[m_pEncodingPopUp selectItemAtIndex:0];	
	
	//アクセサリビューにラベルとポップアップボタンを登録
	[m_pAccessoryView addSubview:m_pLabel];
	[m_pAccessoryView addSubview:m_pEncodingPopUp];
	
	//ファイル選択パネルにアクセサリビューを登録
	[m_pPanel setAccessoryView:m_pAccessoryView];
	
EXIT:;
	return result;
}

//******************************************************************************
// デフォルトエンコーディング名取得
//******************************************************************************
const NSString* MTFileOpenPanel::GetDefaultEncodingName()
{
	unsigned long index = 0;
	unsigned long defaultEncodingId = 0;
	NSString* pDefaultEncodingName = @"";
	
	//デフォルトエンコーディングID取得
	defaultEncodingId = GetDefaultEncodingId();
	
	//エンコーディングIDを一覧から検索
	for (index = 0; index < m_EncodingList.GetSize(); index++) {
		if (defaultEncodingId == m_EncodingList.GetEncodingId(index)) {
			pDefaultEncodingName = m_EncodingList.GetEncodingName(index);
			break;
		}
	}
	
	//ロケールが日本なら ShiftJIS  @"Japanese (Windows, DOS)"
	//それ以外は Windows-1252    @"Western (Windows Latin 1)"
	
	return pDefaultEncodingName;
}

//******************************************************************************
// デフォルトエンコーディングID取得
//******************************************************************************
unsigned long MTFileOpenPanel::GetDefaultEncodingId()
{
	unsigned long encodingId = 0;
	NSString* pLocale = nil;
	
	//エンコーディング選択状態
	pLocale = [[NSLocale currentLocale] localeIdentifier];
	
	if ([pLocale isEqualToString:@"ja_JP"]) {
		//ロケールが日本なら ShiftJIS
		encodingId = NSShiftJISStringEncoding;
	}
	else {
		//それ以外は Windows-1252 （ISO-8859-1の方が適切？）
		encodingId = NSWindowsCP1252StringEncoding;
	}
	
	return encodingId;
}


