#include "disk.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <dirent.h>
#include <sys/types.h>
#include <fcntl.h>
#include <limits.h>
#include <sys/stat.h>
#include <pwd.h>
#include <grp.h>
#include <dirent.h>

cFile::cFile(char* path, char* name) throw(const char*)
{
   mPath = STRDUP(path);
   mName = STRDUP(name);

   if(lstat(path, &mStatBuf) < 0)
      throw "lstat failure on cFile::cFile()";
      
   if(S_ISLNK(mStatBuf.st_mode)) {
      if(stat(path, &mStatBufLink) < 0) 
         throw "stat failure on cFile::Allocate()";
   }
}

cFile::~cFile()
{
   FREE(mPath);
   FREE(mName);
}


ALLOCATED char* cFile::Extension()
{
   for(char* p = mName + strlen(mName); p != mName; p--) {
      if(*p == '.') {
         return STRDUP(p+1);
      }
   }

   return STRDUP("");
}

ALLOCATED char* cFile::NameWithoutExtension()
{
   char* max = mName + strlen(mName);
   for(char* p = mName; p < max; p++) {
      if(*p == '.') {
         char* result = (char*)MALLOC(p - mName + 1);
         memcpy(result, mName, p - mName);
         result[p - mName] = 0;
         return result;
      }
   }

   return STRDUP(mName);
}

ALLOCATED char* cFile::DirName()
{
   for(char* p = mPath + strlen(mPath); p != mPath; p--) {
      if(*p == '/') {
         char* result = (char*)MALLOC(p - mPath + 1);
         memcpy(result, mPath, p - mPath);
         result[p - mPath] = 0;
         return result;
      }
   }

   return STRDUP(mPath);
}


struct stat* cFile::StatBuf()
{
   if(S_ISLNK(mStatBuf.st_mode))
      return &mStatBufLink;
   else
      return &mStatBuf;
}

bool cFile::IsDir()
{
   return S_ISDIR(StatBuf()->st_mode);
}

bool cFile::IsExe() {
   return StatBuf()->st_mode&S_IXUSR || StatBuf()->st_mode&S_IXGRP
          || StatBuf()->st_mode & S_IXOTH;
}

char* cFile::User() {
   struct passwd* ud = getpwuid(StatBuf()->st_uid);
      
   if(ud)
      return ud->pw_name;
   else
      return NULL;
}

char* cFile::Group()
{
   struct group* gd = getgrgid(StatBuf()->st_gid);
   if(gd)
      return gd->gr_name;
   else
      return NULL;
}

char* cFile::LinkTo()
{
	char tmp[PATH_MAX + 1];
	int bytes = readlink(mPath, tmp, PATH_MAX);

	if(bytes >= 0) {
		//tmp[bytes] = 0;

      return STRDUP(tmp);
	}

   return STRDUP("");
}


///////////////////////////////////////////////////////////////////////////////
// cMDir
///////////////////////////////////////////////////////////////////////////////
cDir::cDir(char* path) throw(const char*)
{
   strcpy(mPath, path);
   const int len = strlen(mPath);
   if(mPath[len-1] != '/') {
      mPath[len] = '/';
      mPath[len+1] = 0;
   }

   read();
}

cDir::~cDir()
{
   for(vector<cFile*>::iterator i = mFiles.begin(); i != mFiles.end(); i++) {
      delete (*i);
   }
}

ALLOCATED char* cDir::CWD()
{
	int l = 50;
	char* result = (char*)MALLOC(l);
   
	while(!getcwd(result, l)) {
		l += 50;
		result = (char*)REALLOC(result, l);
   	}

   return result;
}

void cDir::read() throw(const char*)
{   
   DIR* dir = opendir(mPath);
   if(dir == NULL) throw "opendir() false on cDir::Read()";

   for(vector<cFile*>::iterator i = mFiles.begin(); i != mFiles.end(); i++) delete *i;
   mFiles.clear();
   
   struct dirent* entry;
   while(entry = readdir(dir)) {
      char path[PATH_MAX];
      strcpy(path, mPath);
      strcat(path, entry->d_name);

      cFile* file = NULL;
      
      try {
         file = new cFile(path, entry->d_name);
      }
      catch(const char* msg) {
      }
      
      if(file) mFiles.push_back(file);
   }
   closedir(dir);
}

bool cDir::Move(char* path) 
{
   char old_path[PATH_MAX];
   strcpy(old_path, path);

   strcpy(mPath, path);
   const int len = strlen(mPath);
   if(mPath[len-1] != '/') {
      mPath[len] = '/';
      mPath[len+1] = 0;
   }

   try {
      read();
   }
   catch(const char* msg) {
      strcpy(mPath, old_path);
      read();

      return false;
   }

   return true;
}

bool cDir::Reread()
{
   return Move(mPath);
}

ALLOCATED char* cDir::ParentDir()
{
   if(strcmp(mPath, "/") == 0) return STRDUP("");

   for(char* p=mPath + strlen(mPath)-2; p != mPath-1; p--) {
      if(*p == '/') {
         char* result = (char*)MALLOC(p-mPath + 2);
         memcpy(result, mPath, p-mPath);
         result[p-mPath] = '/';
         result[p-mPath+1] = 0;
         
         return result;
      }
   }

   return STRDUP("");
}

cFile* cDir::FindFileWithName(char* name)
{
   for(vector<cFile*>::iterator i=mFiles.begin(); i!=mFiles.end(); i++) {
      if(strcmp((*i)->Name(), name) == 0) {
         return *i;
      }
   }

   return NULL;
}

char* cDir::Name()
{
   if(strcmp(mPath, "/") == 0) return STRDUP("");

   const int len = strlen(mPath);
   for(char* p=mPath + len-2; p != mPath-1; p--) {
      if(*p == '/') {
         char* result = (char*)MALLOC(mPath + len - p-1);
         memcpy(result, p+1, mPath + len - p -2);
         result[mPath + len - p-2] = 0;
         
         return result;
      }
   }

   return STRDUP("");
}

