/*
 * Copyright(c) 2005- C/pHeR.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at epl-v10.html.
 */
package jp.sorceforge.mergedoc.jstyle;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.preference.ColorSelector;
import org.eclipse.jface.preference.FieldEditor;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.PreferenceConverter;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;

/**
 * A field editor for a boolean and color type preference.
 */
public class BooleanColorFieldEditor extends FieldEditor {

    /**
     * The previously selected, or "before", value.
     */
    private boolean wasSelected;

    /**
     * The checkbox control, or <code>null</code> if none.
     */
    private Button checkBox;

	/**
	 * The color selector, or <code>null</code> if none.
	 */
	private ColorSelector colorSelector;

    /**
     * Creates a new boolean field editor 
     */
    protected BooleanColorFieldEditor() {
    }

    /**
     * Creates this field editor's main control containing all of its
     * basic controls.
     *
     * @param parent the parent control
     */
    protected void createControl(Composite parent) {
        GridLayout layout = new GridLayout();
        layout.numColumns = getNumberOfControls();
        layout.marginWidth = 0;
        layout.marginHeight = 20;
        layout.horizontalSpacing = HORIZONTAL_GAP;
        parent.setLayout(layout);
        doFillIntoGrid(parent, layout.numColumns);
    }

    /**
     * Creates a boolean field editor in the given style.
     * 
     * @param name the name of the preference this field editor works on
     * @param labelText the label text of the field editor
     * @param parent the parent of the field editor's control
     */
    public BooleanColorFieldEditor(String name, String labelText, Composite parent) {
        init(name, labelText);
        createControl(parent);
    }

    /* (non-Javadoc)
     * Method declared on FieldEditor.
     */
    protected void adjustForNumColumns(int numColumns) {
    }

    /* (non-Javadoc)
     * Method declared on FieldEditor.
     */
    protected void doFillIntoGrid(Composite parent, int numColumns) {

    	checkBox = getBooleanChangeControl(parent);
        checkBox.setText(getLabelText());

		Button colorButton = getColorChangeControl(parent);
		GridData gd = new GridData();
		int widthHint = convertHorizontalDLUsToPixels(colorButton,
				IDialogConstants.BUTTON_WIDTH);
		gd.widthHint = Math.max(widthHint, colorButton.computeSize(SWT.DEFAULT,
				SWT.DEFAULT, true).x);
		colorButton.setLayoutData(gd);
    }

    /**
     * Returns the name of the boolean preference this field editor operates on.
     *
     * @return the name of the boolean preference
     */
    private String getBooleanPreferenceName() {
    	return getPreferenceName() + JStyleConstants.SUFFIX_SHOW;
    }
    
    /**
     * Returns the name of the color preference this field editor operates on.
     *
     * @return the name of the color preference
     */
    private String getColorPreferenceName() {
    	return getPreferenceName() + JStyleConstants.SUFFIX_COLOR;
    }
    
    /* (non-Javadoc)
     * Method declared on FieldEditor.
     * Loads the value from the preference store and sets it to
     * the check box.
     */
    protected void doLoad() {
        if (checkBox != null) {
            boolean value = getPreferenceStore().getBoolean(getBooleanPreferenceName());
            checkBox.setSelection(value);
            wasSelected = value;
            colorSelector.setEnabled(wasSelected);
        }
		if (colorSelector != null) {
			colorSelector.setColorValue(PreferenceConverter.getColor(
					getPreferenceStore(), getColorPreferenceName()));
		}
    }

    /* (non-Javadoc)
     * Method declared on FieldEditor.
     * Loads the default value from the preference store and sets it to
     * the check box.
     */
    protected void doLoadDefault() {
        if (checkBox != null) {
            boolean value = getPreferenceStore().getDefaultBoolean(getBooleanPreferenceName());
            checkBox.setSelection(value);
            wasSelected = value;
            colorSelector.setEnabled(wasSelected);
        }
		if (colorSelector != null) {
			colorSelector.setColorValue(PreferenceConverter.getDefaultColor(
					getPreferenceStore(), getColorPreferenceName()));
		}
    }

    /* (non-Javadoc)
     * Method declared on FieldEditor.
     */
    protected void doStore() {
        getPreferenceStore().setValue(getBooleanPreferenceName(),
                checkBox.getSelection());
		PreferenceConverter.setValue(getPreferenceStore(), getColorPreferenceName(),
				colorSelector.getColorValue());
    }

    /**
     * Returns this field editor's current value.
     *
     * @return the value
     */
    public boolean getBooleanValue() {
        return checkBox.getSelection();
    }

    /**
     * Returns the change button for this field editor.
     * @param parent The Composite to create the receiver in.
     *
     * @return the change button
     */
    protected Button getBooleanChangeControl(Composite parent) {
        if (checkBox == null) {
            checkBox = new Button(parent, SWT.CHECK | SWT.LEFT);
            checkBox.setFont(parent.getFont());
            checkBox.addSelectionListener(new SelectionAdapter() {
                public void widgetSelected(SelectionEvent e) {
                    boolean isSelected = checkBox.getSelection();
                    valueChanged(wasSelected, isSelected);
                    wasSelected = isSelected;
                    colorSelector.setEnabled(wasSelected);
                }
            });
            checkBox.addDisposeListener(new DisposeListener() {
                public void widgetDisposed(DisposeEvent event) {
                    checkBox = null;
                }
            });
        } else {
            checkParent(checkBox, parent);
        }
        return checkBox;
    }

	/**
	 * Returns the change button for this field editor.
	 * 
	 * @param parent
	 *            The control to create the button in if required.
	 * @return the change button
	 */
	protected Button getColorChangeControl(Composite parent) {
		if (colorSelector == null) {
			colorSelector = new ColorSelector(parent);
			colorSelector.addListener(new IPropertyChangeListener() {
				// forward the property change of the color selector
				public void propertyChange(PropertyChangeEvent event) {
					BooleanColorFieldEditor.this.fireValueChanged(
							event.getProperty(),
							event.getOldValue(),
							event.getNewValue());
					setPresentsDefaultValue(false);
				}
			});

		} else {
			checkParent(colorSelector.getButton(), parent);
		}
		return colorSelector.getButton();
	}

    /* (non-Javadoc)
     * Method declared on FieldEditor.
     */
    public int getNumberOfControls() {
         return 2;
    }

    /* (non-Javadoc)
     * Method declared on FieldEditor.
     */
    public void setFocus() {
        if (checkBox != null) {
            checkBox.setFocus();
        }
    }

    /* (non-Javadoc)
     * Method declared on FieldEditor.
     */
    public void setLabelText(String text) {
        super.setLabelText(text);
        Label label = getLabelControl();
        if (label == null && checkBox != null) {
            checkBox.setText(text);
        }
    }

    /**
     * Informs this field editor's listener, if it has one, about a change
     * to the value (<code>VALUE</code> property) provided that the old and
     * new values are different.
     *
     * @param oldValue the old value
     * @param newValue the new value
     */
    protected void valueChanged(boolean oldValue, boolean newValue) {
        setPresentsDefaultValue(false);
        if (oldValue != newValue)
            fireStateChanged(VALUE, oldValue, newValue);
    }

    /*
     * @see FieldEditor.setEnabled
     */
    public void setEnabled(boolean enabled, Composite parent) {
        //Only call super if there is a label already
        super.setEnabled(enabled, parent);
        getBooleanChangeControl(parent).setEnabled(enabled);
        getColorChangeControl(parent).setEnabled(enabled);
    }

    /**
     * Stores this field editor's value back into the preference store.
     */
    public void store() {
    	IPreferenceStore preferenceStore = getPreferenceStore();
        if (preferenceStore != null) {
            doStore();
        }
    }
}
