/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.aspect.resource;

import java.io.File;
import java.io.FileNotFoundException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.Map.Entry;

import jp.sourceforge.mergedoc.pleiades.aspect.Pleiades;
import jp.sourceforge.mergedoc.pleiades.aspect.advice.AspectMapping;
import jp.sourceforge.mergedoc.pleiades.aspect.advice.JointPoint;
import jp.sourceforge.mergedoc.pleiades.aspect.advice.PointCut;
import jp.sourceforge.mergedoc.pleiades.log.Logger;
import jp.sourceforge.mergedoc.pleiades.util.FileSystem;
import jp.sourceforge.mergedoc.pleiades.util.UnMnemonicProperties;

/**
 * |󎫏NXłB
 * <p>
 * @author C/pHeR
 */
public class TranslationDictionary implements IDictionary {

	/** K[ */
	private static final Logger log = Logger.getLogger(TranslationDictionary.class);

	/** ftHg|vpeB[Et@C */
	public static final String DEFAULT_PROPERTIES_FILE_NAME = "translation.properties";

	/** |vpeB[ǉfBNg[ */
	private static final String ADDITIONS_DIRECTORY = "additions";
	
	/** |LbVEvpeB[Et@C */
	private static final File cacheFile = new File(Pleiades.configurationPath, "translation-cached.properties");

	/** ftHg|vpeB[Et@C */
	public static final String CONVERTER_PROPERTIES_FILE_NAME = "translation-converter.properties";
	
	/** {pj[jbNɕϊȂꍇ̖|vpeB[̒lvtBbNX */
	private static final String NOMNEMONIC = "%NOMNEMONIC%";
	
	/** ̃NX̃VOgECX^X */
	private static final TranslationDictionary singleton;
	static {
		if (log.isDebugEnabled()) {
			singleton = new TranslationLoggingDictionary();
		} else {
			singleton = new TranslationDictionary();
		}
	}
	
	/**
	 * |󎫏CX^X擾܂B
	 * <p>
	 * @return |󎫏CX^X
	 */
	public static TranslationDictionary getInstance() {
		return singleton;
	}

	/** |vpeB[ێ}bv */
	private final Map<String, String> map = new HashMap<String, String>();
	
	/** {pj[jbNɕϊȂ}bv */
	private final Set<String> noMnemonicSet = new HashSet<String>();

	/**
	 * ftHg\z܂B
	 */
	protected TranslationDictionary() {
		load();
	}

	/**
	 * t@C[h܂B
	 */
	protected void load() {
		
		// -clean łȂ  LbVꍇ̓LbVgp
		if (!Pleiades.getInstance().getPleiadesOption().isClean() && cacheFile.exists()) {
			
			map.putAll(loadProperties(cacheFile));
			log.info("|LbVEvpeB[[h܂B" + map.size());
		}
		// -clean 
		else {
			
			File propFile = FileSystem.getResourceFile(DEFAULT_PROPERTIES_FILE_NAME);
			if (!propFile.exists()) {
				Exception e = new FileNotFoundException(propFile.getPath());
				log.fatal("|vpeB[܂B", e);
				return;
			}
			map.putAll(loadProperties(propFile));
			
			File additions = FileSystem.getResourceFile(ADDITIONS_DIRECTORY);
			if (additions.exists()) {

				File[] files = additions.listFiles();
				Arrays.sort(files); // 

				for (File file : files) {
					if (file.isFile() && file.getName().endsWith(".properties")) {
						map.putAll(loadProperties(file));
					}
				}
			}
			log.info("|vpeB[[h܂B" + map.size());

			// ϊ
			convert();
			
		}
	}

	/**
	 * ϊ܂B
	 */
	protected void convert() {
		
		File convFile = FileSystem.getResourceFile(CONVERTER_PROPERTIES_FILE_NAME);
		if (convFile.exists()) {

			Map<String, String> convMap = new HashMap<String, String>();
			convMap.putAll(loadProperties(convFile));
			
			for (Entry<String, String> entry : map.entrySet()) {

				String key = entry.getKey();
				String value = entry.getValue();
				String resultValue = value;

				// uiF{Ɂق܂Ɂj
				for (Entry<String, String> e : convMap.entrySet()) {
					String convKey = e.getKey();
					String convValue = e.getValue();
					resultValue = resultValue.replaceAll(convKey, convValue);
				}
				if (!value.equals(resultValue)) {
					map.put(key, resultValue);
				}
			}
		}
	}
	
	/**
	 * |vpeB[LbVƂĉi܂B
	 */
	public void store() {
		
		Properties prop = new Properties();
		prop.putAll(map);
		FileSystem.storeProperties(prop, cacheFile);
		log.info("|LbVEvpeB[Et@Cۊǂ܂B" + prop.size());
	}

	/**
	 * w肵vpeB[Et@C[h܂B
	 * <p>
	 * @param file vpeB[Et@C
	 * @return }bv
	 */
	@SuppressWarnings("unchecked")
	private Map<? extends String, ? extends String> loadProperties(File file) {

		Properties prop = FileSystem.loadProperties(file);
		return (Map<? extends String, ? extends String>) prop;
	}

	/**
	 * @see IDictionary#lookup(String, JointPoint)
	 */
	public String lookup(String en, JointPoint jointPoint) {
		

		// (&A) ̂悤ȓ{p̃j[jbNɊ܂܂Ăꍇ
		// |ς݂ł邽߁AȂ
		if (hasJaMnemonic(en)) {
			return en;
		}

		// pꃊ\[X񂩂j[jbN䕶 & 
		String enNoMnemonic = en.replaceFirst(
				"\\&(" + UnMnemonicProperties.MNEMONIC_CHARS + ")", "$1");
		
		// |vpeB[{擾
		String jaNoMnemonic = getJaValue(enNoMnemonic, jointPoint);
		
		// j[jbN{pɕϊB
		// ̕ϊ͖ꂪȂꍇsB
		String ja = editMnemonicEnToJa(en, enNoMnemonic, jaNoMnemonic);

		return ja;
	}

	/**
	 * @see IDictionary#lookupIgnoreMnemonic(String, JointPoint)
	 */
	public String lookupIgnoreMnemonic(String en, JointPoint jointPoint) {

		// (&A) ̂悤ȓ{p̃j[jbNɊ܂܂Ăꍇ
		// |ς݂ł邽߁AȂ
		if (hasJaMnemonic(en)) {
			return en;
		}
		
		// |vpeB[{擾
		String ja = getJaValue(en, jointPoint);
		
		return ja;
	}

	/**
	 * (&A) ̂悤ȓ{p̃j[jbNɊ܂܂Ă邩肵܂B
	 * <p>
	 * @param value \[X
	 * @return ܂܂Ăꍇ true
	 */
	protected boolean hasJaMnemonic(String value) {
		return value.matches(
				"(?s)^.*?\\(\\&" + UnMnemonicProperties.MNEMONIC_CHARS + "\\).*$");
	}
	
	/**
	 * {擾܂B
	 * <p>
	 * @param enNoMnemonic pꃊ\[Xij[jbNj
	 * @param jointPoint WCgE|Cg
	 * @return {ꃊ\[Xij[jbNjB|łȂꍇ enNoMnemonicB
	 */
	protected String getJaValue(String enNoMnemonic, JointPoint jointPoint) {
		
		// |svȏꍇ́Â܂ܕԂ
		if (isNoTranslation(enNoMnemonic, jointPoint)) {
			return enNoMnemonic;
		}
		
		// |vpeB[{擾
		String result = map.get(enNoMnemonic);

		// |vpeB[擾łȂꍇ
		if (result == null) {
			
			// trim čēx擾
			result = getJaValueByTrimmed(enNoMnemonic);
		}
		
		// DEBUG
		//if (enNoMnemonic.equals("all")) {
		//	String s = System.currentTimeMillis() + " [" + enNoMnemonic + "]  [" + result + "]";
		//	log.debug("fobO|ǐՃX^bNg[X", new Exception(s));
		//	return s;
		//}
		
		// Ȃꍇ͂̂܂ܕԂ
		if (result == null) {
			result = enNoMnemonic;
		}
		
		// NpɃLbV
		if (Pleiades.getInstance().getPleiadesOption().isClean()) {
			map.put(enNoMnemonic, result);
		}
		
		return result;
	}

	/**
	 * trim ē{ꃊ\[X擾܂B
	 * <p>
	 * @param enNoMnemonic pꃊ\[Xij[jbNj
	 * @return {ꃊ\[Xij[jbNjB|łȂꍇ nullB
	 */
	protected String getJaValueByTrimmed(String enNoMnemonic) {
		
		// 擪Xy[Xޔ
		StringBuilder leading = new StringBuilder();
		char[] cArray = enNoMnemonic.toCharArray();
		for (int i = 0; i < cArray.length; i++) {
			char c = cArray[i];
			if (c == ' ' || c == '\t' || c == '\r' || c == '\n') {
				leading.append(c);
			} else {
				break;
			}
		}
		if (leading != null && leading.length() == enNoMnemonic.length()) {
			return null;
		}
		
		// Xy[XA:A... ޔ
		StringBuilder trailing = new StringBuilder();
		for (int i = cArray.length - 1; i > 0; i--) {
			char c = cArray[i];
			if (c == ' ' || c == '\t' || c == '\r' || c == '\n' || c == ':') {
				trailing.insert(0, c);
			} else if (c == '.' && i > 2 && cArray[i - 1] == '.' && cArray[i - 2] == '.') {
				trailing.insert(0, "...");
				i -= 2;
			} else {
				break;
			}
		}
		
		// OXy[X
		String enTrimmed = enNoMnemonic.substring(leading.length(), enNoMnemonic.length() - trailing.length());
		
		// ēxA|vpeB[{擾
		String result = map.get(enTrimmed);
		if (result != null && !result.equals(enTrimmed)) {
			log.debug("OXy[X[" + enTrimmed + "] [" + enNoMnemonic +
					"] |󌋉[" + leading + result + trailing + "]");
		}
		
		// |vpeB[擾łȂꍇ
		if (result == null) {

			// K\vpeB[{擾
			result = getJaValueByRegex(enTrimmed);

			// 󂪌Ȃꍇ̃Oo
			if (result == null) {
				NotFoundProperties.getInstance().println(enNoMnemonic);
			}
		}

		// OXy[X𕜌
		if (result != null) {
			result = leading + result + trailing;
		}
		
		return result;
	}

	/**
	 * K\{ꃊ\[X擾܂B
	 * <p>
	 * @param enTrimmed pꃊ\[Xij[jbNj
	 * @return {ꃊ\[Xij[jbNjB|łȂꍇ nullB
	 */
	protected String getJaValueByRegex(String enTrimmed) {
		return RegexDictionary.getInstance().lookup(enTrimmed);
	}
	
	/**
	 * |󂪕sv肵܂B
	 * ĂяõpbP[WNXɔ肳܂B
	 * <p>
	 * @param enNoMnemonic pꃊ\[Xij[jbNj
	 * @param jointPoint WCgE|Cg
	 * @return |󂪕svȏꍇ true
	 */
	protected boolean isNoTranslation(String enNoMnemonic, JointPoint jointPoint) {

		final int TRACE_MAX = 30;
		StackTraceElement[] stes = null;

		// xml [exclludeTrace] Ăяog[Xɂ鏜OiċAIɑkj
		if (jointPoint != null) {

			PointCut pointCut = AspectMapping.getInstance().getPointCut(jointPoint);
			if (pointCut != null) {
				
				List<JointPoint> excludeTrace = pointCut.getExcludeTrace();
				if (excludeTrace.size() > 0) {

					stes = Thread.currentThread().getStackTrace();
					for (JointPoint jp : excludeTrace) {

						for (int i = 0; i < TRACE_MAX && i < stes.length; i++) {

							StackTraceElement ste = stes[i];
							String className = jp.getClassName();
							if (className.equals(ste.getClassName())) {

								String methodName = jp.getMethodName();
								if (methodName == null || methodName.equals(ste.getMethodName())) {
									return true;
								}
							}
						}
					}
				}
			}
		}

		// properties [%EXCULUDE%] Ăяoɂ鏜OiƃpbP[WPʁj
		Set<String> noTransPathEntries = TranslationExcludeProperties
				.getInstance().getPathEntries(enNoMnemonic);

		if (noTransPathEntries != null) {
			if (stes == null) {
				stes = Thread.currentThread().getStackTrace();
			}
			for (int i = 0; i < TRACE_MAX && i < stes.length; i++) {

				StackTraceElement ste = stes[i];
				String className = ste.getClassName();

				for (String noTransPath : noTransPathEntries) {
					if (className.startsWith(noTransPath)) {
						return true;
					}
				}
			}
		}
		return false;
	}

	/**
	 * j[jbNpp{pɕҏW܂B<br>
	 * j&x  (&X)
	 * <p>
	 * @param en				pꃊ\[X  ij[jbNLj
	 * @param enNoMnemonic		pꃊ\[X  ij[jbNj
	 * @param jaNoMnemonic		{ꃊ\[Xij[jbNj
	 * @return					{ꃊ\[Xij[jbNLj
	 */
	protected String editMnemonicEnToJa(String en,
			String enNoMnemonic, String jaNoMnemonic) {

		if (Pleiades.getInstance().getPleiadesOption().isNoMnemonic()) {
			return jaNoMnemonic;
		}
		if (en.equals(enNoMnemonic)) {
			return jaNoMnemonic;
		}
		if (jaNoMnemonic.startsWith(NOMNEMONIC)) {
			String noMnemonicResult = jaNoMnemonic.replaceFirst("^" + NOMNEMONIC, "");
			noMnemonicSet.add(noMnemonicResult);
			return noMnemonicResult;
		}
		if (noMnemonicSet.contains(en)) {
			return en;
		}

		String mnemonicChar = en.replaceFirst(
				"(?s)^.*?\\&(" + UnMnemonicProperties.MNEMONIC_CHARS + ").*$",
				"$1");

		if (mnemonicChar.length() != 1) {
			log.error("Mnemonic invalid length: " + mnemonicChar.length());
			log.error(" enValue:            " + en);
			log.error(" enValueNonMnemonic: " + enNoMnemonic);
			log.error(" mnemonicChar:       " + mnemonicChar);
			return en;
		}

		// hoge @Ctrl+X		-> hoge(&H) @Ctrl+X
		// '@Override'		-> '@Override'(&O)
		String mnemonicJa = "(&" + mnemonicChar.toUpperCase() + ")";
		String ja = jaNoMnemonic.replaceFirst("(?s)^(.{2,}?)(" +
				"@\\p{ASCII}+|" +			// j @Ctrl+Shift+X
				"\\s*|" +					// j 
				":\\s*|" +					// j :
				":\\s+\\{[0-9]\\}\\s*|" +	// j :{}
				"\\.\\.\\.(</a>|)\\s*|" +	// j ...
				"(:|)\\s+\\(.+\\)\\s*" +	// j :()
				")$", "$1" + mnemonicJa + "$2");
		
		return ja;
	}
}
