/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.aspect;

import java.io.IOException;
import java.lang.instrument.IllegalClassFormatException;
import java.security.ProtectionDomain;

import javassist.CannotCompileException;
import javassist.CtClass;
import javassist.CtMethod;
import javassist.NotFoundException;
import jp.sourceforge.mergedoc.pleiades.log.Logger;

/**
 * ԌvO@\t LauncherTransformer łB
 * <p>
 * @author C/pHeR
 */
public class LauncherLoggingTransformer extends LauncherTransformer implements ITimingLogger {

	/** K[ */
	private static final Logger log = Logger.getLogger(LauncherLoggingTransformer.class);
	
	/** ̃NX */
	private static final String CLASS_NAME = LauncherLoggingTransformer.class.getName();
	
	/** transform (ۃNX) \bh̏v */
	private volatile long transformAbstractTime;
	
	/** transform (ۃNX) \bȟĂяo */
	private volatile long transformAbstractCall;
	
	/** createCtClass (ۃNX) \bh̏v */
	private volatile long createCtClassTime;
	
	/** createCtClass (ۃNX) \bȟĂяo */
	private volatile long createCtClassCall;
	
	/** transform \bh̏v */
	private volatile long transformTime;
	
	/** transform \bȟĂяo */
	private volatile long transformCall;
	
	/** weaveUptimeMethod \bh̏v */
	private volatile long weaveUptimeMethodTime;
	
	/** weaveUptimeMethod \bȟĂяo */
	private volatile long weaveUptimeMethodCall;
	
	/** weaveStartTranslationAspect \bh̏v */
	private volatile long weaveStartTranslationAspectTime;
	
	/** weaveStartTranslationAspect \bȟĂяo */
	private volatile long weaveStartTranslationAspectCall;

	/**
	 * NgXtH[}[\z܂B
	 */
	public LauncherLoggingTransformer() {
		Pleiades.getInstance().registTimingLogger(this);
	}

	/**
	 * eNX̓\bhɎԌvOo͂ǉ\bhłB
	 */
	@Override
	public byte[] transform(
			ClassLoader loader,
			String internalName,
			Class<?> classBeingRedefined,
			ProtectionDomain protectionDomain,
			byte[] bytecode) throws IllegalClassFormatException {
		
		long startTime = System.currentTimeMillis();
		byte[] result = super.transform(loader, internalName, classBeingRedefined, protectionDomain, bytecode);
		synchronized (this) {
			transformAbstractTime += System.currentTimeMillis() - startTime;
			transformAbstractCall++;
		}
		return result;
	}
	
	/**
	 * oCgR[hϊs܂B
	 */
	@Override
	protected byte[] transform(
			ClassLoader loader,
			String className,
			ProtectionDomain protectionDomain,
			byte[] bytecode)
		throws CannotCompileException, NotFoundException, IOException {

		long startTime = System.currentTimeMillis();
		
		byte[] result = null;
		if (isEclipseLauncherClass(className)) {
			
			// |gXtH[}[E\bh̖ߍ
			result = weaveStartTranslationAspect(protectionDomain, bytecode);

		} else if (isWorkbenchClass(className)) {

			// ̃gXtH[}[G[WFg폜
			Pleiades.getInstance().getInstrumentation().removeTransformer(this);
			
			// [Nx`NX̕ϊ
			result = weaveUptimeMethod(protectionDomain, bytecode);
		}
		
		synchronized (this) {
			transformTime += System.currentTimeMillis() - startTime;
			transformCall++;
		}
		return result;
	}

	/**
	 * [Nx`ENX肵܂B
	 * <p>
	 * @param className NX
	 * @return [Nx`ENX̏ꍇ true
	 */
	protected boolean isWorkbenchClass(String className) {
		return className.equals("org.eclipse.ui.internal.Workbench");
	}
	
	/**
	 * eNX̓\bhɎԌvOo͂ǉ\bhłB
	 */
	@Override
	protected byte[] weaveStartTranslationAspect(ProtectionDomain protectionDomain, byte[] bytecode)
		throws CannotCompileException, NotFoundException, IOException {
		
		long startTime = System.currentTimeMillis();
		byte[] result = super.weaveStartTranslationAspect(protectionDomain, bytecode);
		synchronized (this) {
			weaveStartTranslationAspectTime += System.currentTimeMillis() - startTime;
			weaveStartTranslationAspectCall++;
		}
		return result;
	}
	
	/**
	 * NԌv\bh𖄂ߍ݂܂B
	 * <p>
	 * @param protectionDomain veNVEhC
	 * @param bytecode oCgR[h
	 * @return ߍ݌̃oCgR[h
	 */
	protected byte[] weaveUptimeMethod(ProtectionDomain protectionDomain, byte[] bytecode)
		throws CannotCompileException, NotFoundException, IOException {

		long startTime = System.currentTimeMillis();
		
		CtClass ctClass = createCtClass(bytecode, protectionDomain);
		CtMethod runStartupWithProgress =
			ctClass.getMethod("runStartupWithProgress", "(ILjava/lang/Runnable;)V");
		runStartupWithProgress.insertAfter(CLASS_NAME + ".logUptimeAll();");
		byte[] result =  ctClass.toBytecode();
		
		synchronized (this) {
			weaveUptimeMethodTime += System.currentTimeMillis() - startTime;
			weaveUptimeMethodCall++;
		}
		return result;
	}

	/**
	 * eNX̓\bhɎԌvOo͂ǉ\bhłB
	 */
	@Override
	protected CtClass createCtClass(
			byte[] bytecode,
			ProtectionDomain protectionDomain) throws IOException, NotFoundException {
		
		long startTime = System.currentTimeMillis();
		CtClass result = super.createCtClass(bytecode, protectionDomain);
		synchronized (this) {
			createCtClassTime += System.currentTimeMillis() - startTime;
			createCtClassCall++;
		}
		return result;
	}

	/**
	 * ׂĂ̓o^ς݃gXtH[}[̎ԌvOo͂܂B
	 */
	public static void logUptimeAll() {
		
		for (ITimingLogger logger : Pleiades.getInstance().getTimeingLoggers()) {
			logger.logUptime();
		}

		long startTime = Long.valueOf(System.getProperty("eclipse.startTime"));
		double time = (System.currentTimeMillis() - startTime) / 1000d;
		log.debug(String.format(
				"Eclipse ̋N܂BN: %.3f b", time));
	}

	/**
	 * ԌvOo͂܂B
	 * @see ITimingLogger#logUptime()
	 */
	public void logUptime() {
		
		String msg = "%-31s %6.3f bA%8d ";
		
		log.debug(String.format(msg, "transform (Abstract)",
				transformAbstractTime / 1000d, transformAbstractCall));
		log.debug(String.format(msg, "  transform",
				transformTime / 1000d, transformCall));
		log.debug(String.format(msg, "    createCtClass (Abstract)",
				createCtClassTime / 1000d, createCtClassCall));
		log.debug(String.format(msg, "    weaveStartTranslationAspect",
				weaveStartTranslationAspectTime / 1000d, weaveStartTranslationAspectCall));
		log.debug(String.format(msg, "    weaveUptimeMethod",
				weaveUptimeMethodTime / 1000d, weaveUptimeMethodCall));
	}
}
