#ifndef __MERCURY_DFA__
#define __MERCURY_DFA__

#include <map>
#include <set>

namespace mercury
{
	typedef size_t            state_t;
	typedef std::set<state_t> state_set_t;

	const state_t STATE_ERROR = static_cast<state_t>(-1);


	template<typename _Input>
	class dfa
	{
	public:
		typedef std::pair<state_t, _Input>       state_input_t;
		typedef std::map<state_input_t, state_t> transition_t;
		typedef std::map<state_t      , state_t> transition_default_t;

	public:
		dfa(const state_t &start = STATE_ERROR)
		{
			m_start = start;
		}
		dfa(const state_t &start, const state_set_t &accepts, const transition_t &transition, const transition_default_t &transition_default)
		{
			m_start              = start;
			m_accepts            = accepts;
			m_transition         = transition;
			m_transition_default = transition_default;
		}

		state_t transit(const state_t &state, const _Input &input) const
		{
			{
				const state_input_t tuple(state, input);
				transition_const_iterator_t p = m_transition.find(tuple);
				if(p != m_transition.end())
				{
					return p->second;
				}
			}

			{
				transition_default_const_iterator_t p = m_transition_default.find(state);
				if(p != m_transition_default.end())
				{
					return p->second;
				}
			}

			return STATE_ERROR;
		}

		void connect(const state_t &from, const _Input &input, const state_t &to)
		{
			const state_input_t tuple(from, input);
			m_transition[tuple] = to;
		}
		void connect_default(const state_t &from, const state_t &to)
		{
			m_transition_default[from] = to;
		}

		void merge_transition(const dfa &rhs)
		{
			m_transition.insert(rhs.m_transition.begin(), rhs.m_transition.end());
		}

		bool compact(void)
		{
			return false;
		}

		state_t get_start(void) const           { return m_start; }
		void    set_start(const state_t &start) { m_start = start; }

		state_set_t get_accepts(void) const                 { return m_accepts; }
		void        set_accepts(const state_set_t &accepts) { m_accepts = accepts; }
		void        add_accept (const state_t     &accept ) { m_accepts.insert(accept); }

		bool is_accept_state(const state_t &state) const { return (m_accepts.find(state) != m_accepts.end()); }

		template<typename _InputIterator, typename _OutputIterator>
		bool match(const _InputIterator &text_begin, const _InputIterator &text_end, _OutputIterator &match_end, size_t &match_length) const
		{
			bool    matched = false;
			size_t  length  = 0;
			state_t state   = get_start();

			_InputIterator p = text_begin;

			for(;;)
			{
				if(is_accept_state(state))
				{
					matched      = true;
					match_end    = p;
					match_length = length;
				}
				if(p == text_end) { break; }

				state = transit(state, *p);
				if(state == STATE_ERROR)
				{
					break;
				}
				length++;
			}
			return matched;
		}

	private:
		typedef typename transition_t        ::const_iterator transition_const_iterator_t;
		typedef typename transition_default_t::const_iterator transition_default_const_iterator_t;

		state_t              m_start;
		state_set_t          m_accepts;
		transition_t         m_transition;
		transition_default_t m_transition_default;
	};
}

#endif
