/*   Copyright 2008  KPS Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.co.kpscorp.meema.engine.plugin;

import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import jp.co.kpscorp.meema.engine.BeanMethod;
import jp.co.kpscorp.meema.engine.LineOverException;
import jp.co.kpscorp.meema.util.Service;
import jp.co.kpscorp.meema.util.Util;
import junit.framework.TestCase;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * BeanのプロパティからJunitのテストコードを作成するPlugin
 * 
 * @version 2004/07/29
 * @author Katsusuke
 * @see
 */
public class AssertClassMaker implements Plugin {

	private Map<String, String> assertFilter = new HashMap<String, String>();

	private Map<String, String> objNameMap;

	private Map<String, String> activeObjNameMap;

	private Map<String, String> classNameMap;

	private String filePath;

	private static Log log = LogFactory.getLog(AssertClassMaker.class);

	private String sufix;

	private String preFix = "o";

	private StringBuffer yosokuchi;

	private StringBuffer assertRow;

	private int lineLimit = 64000;

	private String limtMsg = "最大ラインに達しましたので処理を終了します";

	private Map<String, String[]> regexMap = new HashMap<String, String[]>();

	private Service service = new Service();

	private String packageName;

	private String className;

	private String srcPath;

	private List<StringBuffer> code;

	/**
	 * 初期処理
	 * 
	 * @param arg
	 *            サフィックス
	 * @see jp.co.kpscorp.commontools.Plugin#setUp(java.lang.Object[])
	 */
	public void setUp(Object[] arg) {
		objNameMap = new HashMap<String, String>();
		activeObjNameMap = new HashMap<String, String>();
		classNameMap = new HashMap<String, String>();
		code = new ArrayList<StringBuffer>();
		if (arg.length > 0 && arg[0] instanceof String) {
			sufix = (String) arg[0];
		}
		if (arg.length > 1 && arg[1] instanceof String) {
			packageName = (String) arg[1];
		}
	}

	/**
	 * @param bm
	 * @throws Exception
	 * @see jp.co.kpsCorp.commonTools.BeanPropertyGetter.Plugin#objectBegin(jp.co.kpsCorp.commonTools.BeanPropertyGetter.BeanMethod)
	 */
	public void objectBegin(BeanMethod bm) throws Exception {
		if (bm.getDepth() == 0) {
			setUp(bm);
			return;
		}
		// Check対象で無いなら処理しない（ただしiteratorのnext()は呼ぶ
		if (!bm.isCheckOn() && !bm.getPropertyName().equals("next")) {
			return;
		}
		// String className = service.getSimpleClassName(bm.getReturnObject());
		String className = bm.getReturnType().getSimpleName();
		bm.setReturnObjectName(getUniqueObjectName(className));
		StringBuffer rowBuf = new StringBuffer("\t\t\t");
		// 初めて登場するオブジェクト名なら新規変数定義をする。
		if (objNameMap.get(bm.getReturnObjectName()) == null) {
			rowBuf.append(className);
			rowBuf.append(" ");
			objNameMap.put(bm.getReturnObjectName(), bm.getReturnObjectName());
		}
		// 活動中のオブジェクト名に登録
		activeObjNameMap
				.put(bm.getReturnObjectName(), bm.getReturnObjectName());
		// className objName= (className) objName.getterName();
		rowBuf.append(bm.getReturnObjectName() + " = " + bm.getMethodString()
				+ ";");
		code.add(rowBuf);
		// *
		Class<? extends Object> c = bm.getReturnType();
		addClassName(c);
	}

	private void addClassName(Class<? extends Object> c) {
		if (c.isPrimitive()) {
			return;
		}
		if (!c.isArray()) {
			classNameMap.put(Util.getExAopName(c), null);
		} else {
			String s = c.getName();
			if (s.length() > 2) {
				classNameMap.put(s.substring(2, s.length() - 1), null);
			}
		}
	}

	private String getUniqueObjectName(String className) {
		// String name = service.getObjectName(o);
		// String name = className.substring(0, 1).toLowerCase() +
		// className.substring(1);
		String name = preFix + className;
		if (name.indexOf("[]") > -1) {
			name = name.substring(0, name.indexOf("[]")) + "s";
		}
		int cnt = 2;
		while (activeObjNameMap.get(name) != null) {
			name = name + cnt;
			cnt++;
		}
		activeObjNameMap.put(name, name);
		return name;
	}

	/**
	 * @param bm
	 * @throws Exception
	 * @see jp.co.kpsCorp.commonTools.BeanPropertyGetter.Plugin#objectEnd(jp.co.kpsCorp.commonTools.BeanPropertyGetter.BeanMethod)
	 */
	public void objectEnd(BeanMethod bm) throws Exception {
		// Check対象で無いなら処理しない
		if (!bm.isCheckOn() && !bm.getPropertyName().equals("next")) {
			return;
		}
		if (bm.getDepth() == 0) {
			tearDown(bm);
			return;
		}
		// 活動中のオブジェクト名を削除
		activeObjNameMap.remove(bm.getReturnObjectName());
	}

	/**
	 * @param bm
	 * @throws Exception
	 * @see jp.co.kpsCorp.commonTools.BeanPropertyGetter.Plugin#execElement(jp.co.kpsCorp.commonTools.BeanPropertyGetter.BeanMethod)
	 */
	public void execElement(BeanMethod bm) throws Exception {
		if (!bm.isReturningException()) {
			yosokuchi = makeYosokuchi(bm);
			String[] regexs = regexMap.get(bm.getPropertyName());
			if (regexs != null) {
				yosokuchi = new StringBuffer(doRegex(yosokuchi.toString(),
						regexs));
			}
			assertRow = new StringBuffer();
			assertRow.append(assertRow).append("\t\t\tassertEquals(").append(
					yosokuchi).append(",");
			if (bm.getReturnObject() instanceof Float
					|| bm.getReturnObject() instanceof Double) {
				ifFloatOrDuble(assertRow, bm);
			} else {
				assertRow.append(bm.getMethodString());
				if (regexs != null) {
					assertRow.append(makeRegexStr(regexs));
				}
			}
			assertRow.append(");");
			if (!Util.isFiltering(assertRow.toString(), assertFilter)) {
				code.add(assertRow);
			}
		}
		// UtilまたはArrayをassert文内で使用してた場合classNameMapに追加
		setClassName(Util.class);
		setClassName(Array.class);
	}

	private void setClassName(Class c) {
		if (assertRow.toString().indexOf("(" + c.getSimpleName() + ")") > -1
				|| assertRow.toString().indexOf(" " + c.getSimpleName() + " ") > -1
				|| assertRow.toString().indexOf(c.getSimpleName() + ".") > -1) {
			classNameMap.put(c.getName(), null);
		}
	}

	public void addRegex(String prop, String[] regex) {
		regexMap.put(prop, regex);
	}

	private String doRegex(String s, String[] regexs) {
		for (int i = 0; i < regexs.length; i = i + 2) {
			if (regexs.length > i + 1) {
				s = s.replaceAll(regexs[i], regexs[i + 1]);
			}
		}
		return s;
	}

	private String makeRegexStr(String[] regexs) {
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < regexs.length; i = i + 2) {
			if (regexs.length > i + 1) {
				sb.append(".replaceAll(\"" + regexs[i] + "\",\""
						+ regexs[i + 1] + "\")");
			}
		}
		return sb.toString();
	}

	private void ifFloatOrDuble(StringBuffer row, BeanMethod bm) {
		if (yosokuchi.toString().equals("true")) {
			if (bm.getReturnObject() instanceof Float) {
				row.append("new Float(");
			} else {
				row.append("new Double(");
			}
			row.append(bm.getMethodString()).append(").isNaN()");
		} else {
			if (bm.getMethod().getReturnType().isPrimitive()) {
				row.append(bm.getMethodString()).append(",0");
			} else {
				// ",0"があるとエラーになる
				row.append(bm.getMethodString());
			}
		}
	}

	/**
	 * 予測値を作成
	 * 
	 * @param bm
	 * @return
	 */
	private StringBuffer makeYosokuchi(BeanMethod bm) {
		StringBuffer yosokuchi = new StringBuffer();
		Object retObj = bm.getReturnObject();
		if (retObj instanceof String) {
			yosokuchi.append("\"").append(Util.escapeString((String) retObj))
					.append("\"");
		} else if (retObj == null) {
			yosokuchi.append("null");
		} else if (primitiveYosokuchi(bm)) {
			String r = retObj.toString();
			if (r.equals("NaN")) {
				r = "true";
			}
			yosokuchi.append(r);
			if (retObj instanceof Long) {
				yosokuchi.append("L");
			}
		} else {
			yosokuchi.append("new ").append(retObj.getClass().getSimpleName())
					.append("(\"").append(retObj.toString()).append("\")");
			classNameMap.put(Util.getExAopName(retObj.getClass()), null);
		}
		return yosokuchi;
	}

	private boolean primitiveYosokuchi(BeanMethod bm) {
		return bm.getMethod().getReturnType().isPrimitive();
	}

	private void setUp(BeanMethod bm) throws Exception {
		objNameMap.clear();
		String rclassName = bm.getReturnType().getSimpleName();
		String objName = preFix + rclassName;
		objNameMap.put(objName, objName);
		activeObjNameMap.put(objName, objName);
		bm.setReturnObjectName(objName);
		// private void checkClassname(className objName) {
		makeHeader(bm, rclassName, objName);
		this.className = "Test"
				+ bm.getReturnObject().getClass().getSimpleName() + sufix;
		addClassName(bm.getReturnObject().getClass());
	}

	private void makeHeader(BeanMethod bm, String className, String objName)
			throws Exception {
		StringBuffer row = new StringBuffer("\t\tpublic void ");
		row.append(Util.getTestMethodName(bm.getReturnObject(), sufix));
		row.append("(" + className + " " + objName + ")  throws Exception {");
		code.add(row);
		row = new StringBuffer(
				"\t\t\tLog log = LogFactory.getLog(this.getClass());");
		code.add(row);
		row = new StringBuffer("\t\t\tlog.info(\"");
		row.append(Util.getTestMethodName(bm.getReturnObject(), sufix));
		row.append(" for \"  + ");
		row.append(objName);
		row.append(".toString() + \" start!\");");
		code.add(row);
	}

	private void tearDown(BeanMethod bm) throws Exception {
		StringBuffer row = new StringBuffer("\t\t\tlog.info(\"check end! \");");
		code.add(row);
		row = new StringBuffer("\t\t}");
		code.add(row);

		makeSrcFile();
	}

	private void makeSrcFile() throws IOException, LineOverException {
		String javaFileName = new File(getSrcPath() + "/" + className + ".java")
				.getAbsolutePath();
		BufferedWriter bwt = service.openFile(javaFileName);
		List<StringBuffer> rows = new ArrayList<StringBuffer>();

		// packaget文
		rows.add(new StringBuffer("package ").append(packageName).append(";"));
		// import文
		Map<String, String> simpleNameMap = new HashMap<String, String>();
		Iterator<String> it = classNameMap.keySet().iterator();
		while (it.hasNext()) {
			String name = (String) it.next();
			String simpleName = name.substring(name.lastIndexOf(".") + 1);
			if (simpleNameMap.get(simpleName) == null) {
				simpleNameMap.put(simpleName, name);
				rows.add(new StringBuffer("import ").append(name).append(";"));
			} else {
				rows
						.add(new StringBuffer("//import ").append(name).append(
								";"));
			}
		}
		rows.add(new StringBuffer("import ").append(TestCase.class.getName())
				.append(";"));
		rows.add(new StringBuffer("import ").append(Log.class.getName())
				.append(";"));
		rows.add(new StringBuffer("import ").append(LogFactory.class.getName())
				.append(";"));
		// class
		rows.add(new StringBuffer("public class ").append(className).append(
				" extends TestCase {"));
		// codeを追加
		rows.addAll(code);
		Iterator<StringBuffer> it2 = rows.iterator();
		int cnt = 0;
		while (it2.hasNext()) {
			StringBuffer row = it2.next();
			bwt.write(row.toString(), 0, row.length());
			bwt.newLine();
			bwt.flush();
			cnt++;
			if (cnt > lineLimit) {
				service.closeFile(bwt);
				log.info(limtMsg);
				throw new LineOverException("LineLimit");
			}
		}
		bwt.write("}", 0, "}".length());
		bwt.newLine();
		bwt.flush();
		service.closeFile(bwt);
		log
				.info("TestClassMaker:java file '" + javaFileName
						+ "' was written!");

	}

	/**
	 * テストコードを出力するファイルのフルパス名を参照する
	 * 
	 * @return
	 */
	public String getFilePath() {
		return filePath;
	}

	/**
	 * テストコードを出力するファイルのフルパス名をセットする
	 * 
	 * @param string
	 */
	public void setFilePath(String string) {
		filePath = string;
	}

	/**
	 * 作成しないasset命令のフィルターをセットする。
	 * 
	 * @param objName
	 *            除外asset命令の正規表現
	 */
	public void setAsserFilter(String objName) {
		assertFilter.put(objName, objName);
	}

	/**
	 * 作成しないasset命令のフィルターを解除する。
	 * 
	 * @param objName
	 */
	public void removeAssetFilter(String objName) {
		assertFilter.remove(objName);
	}

	/**
	 * @return lineLimit を戻します。
	 */
	public int getLineLimit() {
		return lineLimit;
	}

	/**
	 * @param lineLimit
	 *            lineLimit を設定。
	 */
	public void setLineLimit(int lineLimit) {
		this.lineLimit = lineLimit;
	}

	public void setEncode(String encode) {
		this.service.setEncode(encode);
	}

	public String getSrcPath() {
		return srcPath;
	}

	public void setSrcPath(String srcPath) {
		this.srcPath = srcPath;
	}
}
