/*   Copyright 2008  KPS Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.co.kpscorp.meema.engine;

import java.lang.reflect.Array;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import jp.co.kpscorp.meema.util.Util;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Beanのメソッドとその戻り値のオブジェクトを含むクラス
 * 
 * @version 2004/07/30
 * @author Katsusuke
 * @see
 */
public class BeanMethod {
	private List<BeanMethod> beanMethods = new LinkedList<BeanMethod>();

	private boolean isCheckOn;

	private boolean isArray = false;

	private boolean isList = false;

	private boolean isMap = false;

	private boolean isReturningWrapper = false;

	private boolean isReturningException = false;

	private boolean isMethodGetter = false;

	private String propName;

	private Integer index;

	private int depth;

	private Method method;

	private Object bean;

	private Object returnObject;

	private String beanName;

	private String returnObjectName;

	/**
	 * Logger for this class
	 */
	private static Log logger = LogFactory.getLog(BeanMethod.class);

	/**
	 * 解析元のBean用のコンストラクター
	 * 
	 * @param returnObject
	 * @throws Exception
	 */
	public BeanMethod(Object returnObject) throws Exception {
		// 大元のBeanの処理
		this.depth = 0;
		this.returnObject = returnObject;
		checkReturnObj(returnObject);
	}

	/**
	 * List/Arrayに対するget(n)メソッド用コンストラクター
	 * 
	 * @param oyaBm
	 * @param index
	 * @throws Exception
	 */
	public BeanMethod(BeanMethod oyaBm, int index) throws Exception {
		commonInit(oyaBm);
		this.index = new Integer(index);
		if (bean.getClass().isArray()) {
			ifArray(index);
		} else {
			ifList(index);
		}
	}

	private void ifArray(int index) throws SecurityException,
			NoSuchMethodException {
		int l = Array.getLength(bean);
		this.propName = addZeroToIndex(index, l);
		Class[] c = { Object.class, int.class };
		this.method = Array.class.getMethod("get", c);
		// [index]の処理
		isMethodGetter = true;
		try {
			returnObject = Array.get(bean, index);
			checkReturnObj(returnObject);
			// toString()でExceptionを出すかCheck
			if (returnObject != null) {
				returnObject.toString();
			}
		} catch (Exception e) {
			returnObject = e.toString();
			isReturningException = true;
			if (!(e.getCause() instanceof IllegalArgumentException)) {
				logger.warn("method:" + this.method.toString() + ":"
						+ e.toString() + "/cause=" + e.getCause());
			}
		}
		isArray = true;
	}

	private void ifList(int index) throws NoSuchMethodException {
		int l = ((List) bean).size();
		this.propName = addZeroToIndex(index, l);
		Class c[] = { int.class };
		this.method = bean.getClass().getMethod("get", c);
		// get(n)メソッドの処理
		isMethodGetter = true;
		try {
			returnObject = ((List) bean).get(index);
			checkReturnObj(returnObject);
			// toString()でExceptionを出すかCheck
			if (returnObject != null) {
				returnObject.toString();
			}
		} catch (Exception e) {
			returnObject = e.toString();
			isReturningException = true;
			if (!(e.getCause() instanceof IllegalArgumentException)) {
				logger.warn("method:" + this.method.toString() + ":"
						+ e.toString() + "/cause=" + e.getCause());
			}
		}
		isList = true;
	}

	/**
	 * @param oyaBm
	 */
	private void commonInit(BeanMethod oyaBm) {
		this.bean = oyaBm.getReturnObject();
		this.depth = oyaBm.getDepth() + 1;
		this.beanName = oyaBm.getReturnObjectName();
	}

	private String addZeroToIndex(int index, int l) {
		int keta = Integer.toString(l).length();
		StringBuffer zeros = new StringBuffer();
		for (int i = 0; i < keta; i++)
			zeros.append("0");
		String s = zeros.append(Integer.toString(index)).toString();
		return s.substring(s.length() - keta);
	}

	/**
	 * Mapに対するget(xx)メソッド用コンストラクター
	 * 
	 * @param oyaBm
	 * @param key
	 * @throws Exception
	 */
	public BeanMethod(BeanMethod oyaBm, String key) throws Exception {
		commonInit(oyaBm);
		this.propName = key;
		Class c[] = { Object.class };
		this.method = bean.getClass().getMethod("get", c);
		// get(xx)メソッドの処理
		isMethodGetter = true;
		try {
			if (bean instanceof Map) {
				returnObject = ((Map) bean).get(key);
			}
			checkReturnObj(returnObject);
			// toString()でExceptionを出すかCheck
			if (returnObject != null) {
				returnObject.toString();
			}
		} catch (Exception e) {
			returnObject = e.toString();
			isReturningException = true;
			if (!(e.getCause() instanceof IllegalArgumentException)) {
				logger.warn("method:" + this.method.toString() + ":"
						+ e.toString() + "/cause=" + e.getCause());
			}
		}
		isMap = true;
	}

	/**
	 * Iteratorに対するnext()メソッド用コンストラクター
	 * 
	 * @param oyaBm
	 * @throws Exception
	 */
	public BeanMethod(BeanMethod oyaBm, Iterator it) throws Exception {
		commonInit(oyaBm);
		this.propName = "next";
		this.method = it.getClass().getMethod("next", (Class[]) null);
		// nextメソッドの処理
		isMethodGetter = true;
		try {
			returnObject = it.next();
			checkReturnObj(returnObject);
			// toString()でExceptionを出すかCheck
			if (returnObject != null) {
				returnObject.toString();
			}
		} catch (Exception e) {
			returnObject = e.toString();
			isReturningException = true;
			if (!(e.getCause() instanceof IllegalArgumentException)) {
				logger.warn("method:" + this.method.toString() + ":"
						+ e.toString() + "/cause=" + e.getCause());
			}
		}
		isMap = false;
	}

	/**
	 * その他のメソッド用コンストラクター
	 * 
	 * @param oyaBm
	 * @param method
	 * @throws Exception
	 */
	public BeanMethod(BeanMethod oyaBm, Method method) {
		commonInit(oyaBm);
		this.method = method;
		// メソッド名をプロパティ名に変換
		this.propName = getPropNameFromGetterName(method.getName());
		// 普通のgetterメソッドの処理
		String methodName = method.getName();
		if ((methodName.startsWith("get") || methodName.startsWith("is")
				|| methodName.equals("size") || method.getReturnType() == Iterator.class)
				&& method.getParameterTypes().length == 0) {
			isMethodGetter = true;
			try {
				returnObject = method.invoke(bean, (Object[]) null);
				checkReturnObj(returnObject);
				// toString()でExceptionを出すかCheck
				if (returnObject != null) {
					returnObject.toString();
				}
			} catch (Exception e) {
				returnObject = e.toString();
				isReturningException = true;
				if (!(e.getCause() instanceof IllegalArgumentException)) {
					logger.warn("method:" + this.method.toString() + ":"
							+ e.toString() + "/cause=" + e.getCause());
				}
			}
		} else
			return;
	}

	private String getPropNameFromGetterName(String getterName) {
		if (getterName.startsWith("get") && getterName.length() > 3)
			return getterName.substring(3, 4).toLowerCase()
					+ getterName.substring(4);
		else if (getterName.startsWith("is") && getterName.length() > 2)
			return getterName.substring(2, 3).toLowerCase()
					+ getterName.substring(3);
		else
			return getterName;
	}

	private void checkReturnObj(Object returnObject) {
		// returnObjectの型Check
		if (returnObject instanceof BigDecimal)
			isReturningWrapper = true;
		else if (returnObject instanceof Boolean)
			isReturningWrapper = true;
		else if (returnObject instanceof Byte)
			isReturningWrapper = true;
		else if (returnObject instanceof Short)
			isReturningWrapper = true;
		else if (returnObject instanceof Integer)
			isReturningWrapper = true;
		else if (returnObject instanceof Long)
			isReturningWrapper = true;
		else if (returnObject instanceof Float)
			isReturningWrapper = true;
		else if (returnObject instanceof Double)
			isReturningWrapper = true;
		else if (returnObject instanceof String)
			isReturningWrapper = true;
		else if (returnObject == null)
			isReturningWrapper = true;
	}

	/**
	 * 必要に応じてキャストしたメソッドの呼び出しのコードを返す。
	 * 
	 * @return
	 */
	public String getMethodString() {
		String returnClassName = null;
		Class c = method.getReturnType();
		if (returnObject != null && !c.equals(returnObject.getClass())
				&& !c.isPrimitive())
			returnClassName = getReturnType().getSimpleName();
		StringBuffer sb = new StringBuffer();
		if (returnClassName != null)
			sb.append("(" + returnClassName + ") ");
		if (isList()) {
			sb.append(getBeanName() + ".get(" + index + ")");
		} else if (isMap()) {
			sb.append(getBeanName() + ".get(\"" + Util.escapeString(propName)
					+ "\")");
		} else if (isArray()) {
			sb.append("Array.get(" + getBeanName() + "," + index + ")");
		} else
			sb.append(getBeanName() + "." + getExAopMehodName(method) + "()");
		return sb.toString();
	}

	private String getExAopMehodName(Method m) {
		String s = m.getName();
		// s2aop対応
		int i = s.indexOf("$$");
		if (i != -1) {
			s = s.substring(0, i);
		}
		return s;

	}

	/**
	 * 戻りの型名を返す
	 * 
	 * @return
	 */
	public Class<? extends Object> getReturnType() {
		if (method != null) {
			Class c = method.getReturnType();
			if (c != Object.class && !c.isInterface()) {
				return (Class<? extends Object>) c;
			} else if (getReturnObject().getClass().isLocalClass()
					|| getReturnObject().getClass().isMemberClass()) {
				return (Class<? extends Object>) c;
			}
		}
		return getReturnObject().getClass();

	}

	/**
	 * メソッドの戻りのオブジェクトを返す
	 * 
	 * @return
	 */
	public Object getReturnObject() {
		return returnObject;
	}

	/**
	 * MapのgetメソッドのKeyを返す。
	 * 
	 * @return
	 */
	public String getPropertyName() {
		return propName;
	}

	/**
	 * 対象メソッドを返す。
	 * 
	 * @return
	 */
	public Method getMethod() {
		return method;
	}

	/**
	 * メソッドを実装しているBeanを返す。
	 * 
	 * @return
	 */
	public Object getBean() {
		return bean;
	}

	/**
	 * メソッドがゲッターかどうか
	 * 
	 * @return
	 */
	public boolean isMethodGetter() {
		return isMethodGetter;
	}

	/**
	 * 戻り値がプリミティブ型かそのラッパークラスかどうか
	 * 
	 * @return
	 */
	public boolean isReturningWrapper() {
		return isReturningWrapper;
	}

	/**
	 * BeanListかどうか
	 * 
	 * @return
	 */
	public boolean isList() {
		return isList;
	}

	/**
	 * BeanがMapかどうか
	 * 
	 * @return
	 */
	public boolean isMap() {
		return isMap;
	}

	/**
	 * 再帰の深さの数を返す
	 * 
	 * @return
	 */
	public int getDepth() {
		return depth;
	}

	/**
	 * Listの場合、返すオブジェクトの位置を返す
	 * 
	 * @return
	 */
	public Integer getIndex() {
		return index;
	}

	/**
	 * メソッドを実装しているBeanに付けた変数名を返す
	 * 
	 * @return
	 */
	public String getBeanName() {
		return beanName;
	}

	/**
	 * メソッドの戻りのオブジェクトに付けた変数名を返す
	 * 
	 * @return
	 */
	public String getReturnObjectName() {
		return returnObjectName;
	}

	/**
	 * メソッドを実装しているBeanに付けた変数名をセットする
	 * 
	 * @param string
	 */
	public void setBeanName(String string) {
		beanName = string;
	}

	/**
	 * メソッドの戻りのオブジェクトに付けた変数名をセットする
	 * 
	 * @return
	 */
	public void setReturnObjectName(String string) {
		returnObjectName = string;
	}

	/**
	 * このメソッドの戻り値のオブジェクトの持つメソッドの集合（1階層下）を返す
	 * 
	 * @return
	 */
	public List<BeanMethod> getBeanMethods() {
		return beanMethods;
	}

	/**
	 * このメソッドの戻り値のオブジェクトの持つメソッドの集合（1階層下）をセットする
	 * 
	 * @param list
	 */
	public void setBeanMethods(List<BeanMethod> list) {
		beanMethods = list;
	}

	public boolean isReturningException() {
		return isReturningException;
	}

	/**
	 * @return isArray を戻します。
	 */
	public boolean isArray() {
		return isArray;
	}

	public boolean isCheckOn() {
		return isCheckOn;
	}

	public void setCheckOn(boolean isCheckOn) {
		this.isCheckOn = isCheckOn;
	}

}
