/*   Copyright 2008  KPS Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.co.kpscorp.meema.engine;

import java.lang.reflect.Array;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import jp.co.kpscorp.meema.engine.plugin.Plugin;
import jp.co.kpscorp.meema.util.Util;

/**
 * Beanのプロパティを取り扱う。List型のプロパティに関してはそこにセットされたBeanも再帰的に処理する。 <br>
 * 処理対象になるプロパティはget/isで始まる引数の無いメソッドのみを対象とする。 <br>
 * 実際の処理は導入された各Pluginが行う。 <br>
 * 
 * @version 2004/07/28
 * @author Katsusuke
 * @see
 */
public class Analyzer {
	private Map<String, Plugin> pluginMap = new HashMap<String, Plugin>();

	private int maxDepth = 7;

	private Map<String, String> filterMap = new HashMap<String, String>();

	private Map<String, String> filterMap2 = new HashMap<String, String>();

	private List<Object> objectList = new LinkedList<Object>();

	private boolean getList = false;

	// 同じオブジェクトを検査するか
	private boolean sameObject = false;

	public Analyzer() {
		setObjectFilter(".*\\.Class");
	}

	/**
	 * Pluginを導入する
	 * 
	 * @param key
	 *            removeする時に使用するKey
	 * @param plugin
	 *            Pluginインターフェースを実装したPlugin
	 */
	public void addPlugin(String key, Plugin plugin) {
		pluginMap.put(key, plugin);
	}

	/**
	 * Pluginを削除する。
	 * 
	 * @param key
	 */
	public void removePlugin(String key) {
		pluginMap.remove(key);
	}

	/**
	 * keyを指定してPluginを入手する。
	 * 
	 * @param key
	 * @return
	 */
	public Plugin getPlugin(String key) {
		return pluginMap.get(key);
	}

	/**
	 * 解析対象オブジェクトの型名のフィルターをセットする。
	 * 
	 * なおディフォルトで".Class"を名称に含むクラスを解析対象外としている。
	 * 
	 * @param objName
	 *            除外オブジェクトの名称の正規表現
	 */
	public void setObjectFilter(String objName) {
		filterMap.put(objName, objName);
	}

	/**
	 * 解析対象オブジェクトの型名のフィルターを解除する。
	 * 
	 * @param objName
	 */
	public void removeObjectFilter(String objName) {
		filterMap.remove(objName);
	}

	/**
	 * 解析対象プロパティ名のフィルターをセットする。
	 * 
	 * @param objName
	 *            除外プロパティの名称の正規表現
	 */
	public void setPropFilter(String objName) {
		filterMap2.put(objName, objName);
	}

	/**
	 * 解析対象プロパティ名のフィルターを解除する。
	 * 
	 * @param objName
	 */
	public void removePropFilter(String objName) {
		filterMap2.remove(objName);
	}

	/**
	 * 解析する回帰の深さの最大値を入手する
	 * 
	 * @return
	 */
	public int getMaxDepth() {
		return maxDepth;
	}

	/**
	 * 解析する回帰の深さの最大値をセットする。ディフォルトは５
	 * 
	 * @return
	 */
	public void setMaxDepth(int i) {
		maxDepth = i;
	}

	/**
	 * Beanに関する処理を行う。 <br>
	 * 引数で与えられたBeanに関して各Pluginの処理を行う。
	 * 
	 * @param bean
	 * @param arg
	 *            汎用引数
	 * @throws Exception
	 */
	public void exec(Object bean, Object[] arg) throws Exception {
		Iterator<Plugin> it = pluginMap.values().iterator();
		while (it.hasNext()) {
			Plugin plugin = it.next();
			// setUp
			plugin.setUp(arg);
			BeanMethod bm = new BeanMethod(bean);
			try {
				execObject(bm, plugin);
			} catch (LineOverException e) {
			}
		}
	}

	private void execObject(BeanMethod oyaBm, Plugin plugin) throws Exception {
		oyaBm.setCheckOn(isCheckOn(oyaBm.getReturnObject()));
		plugin.objectBegin(oyaBm);
		if (oyaBm.isCheckOn()) {
			execObjectSub(oyaBm, plugin);
		}
		plugin.objectEnd(oyaBm);
	}

	private void execObjectSub(BeanMethod oyaBm, Plugin plugin)
			throws Exception {
		List<BeanMethod> beanMetods = oyaBm.getBeanMethods();
		// BeanがMapの場合get(xx)検索
		if (oyaBm.getReturnObject() instanceof Map) {
			Set s;
			s = ((Map) oyaBm.getReturnObject()).keySet();
			Iterator it = s.iterator();
			while (it.hasNext()) {
				BeanMethod bm;
				try {
					String key = (String) it.next();
					// propFilter確認
					if (!Util.isFiltering(key, filterMap2)) {
						bm = new BeanMethod(oyaBm, key);
						beanMetods.add(bm);
					}
				} catch (Exception e) {
				}
			}
		}
		// BeanがListの場合get(n)検索
		// Listはiteartorで取るように変更
		if (oyaBm.getReturnObject() instanceof List && getList) {
			for (int index = 0; index < ((List) oyaBm.getReturnObject()).size(); index++) {
				BeanMethod bm;
				try {
					bm = new BeanMethod(oyaBm, index);
					beanMetods.add(bm);
				} catch (Exception e) {
				}
			}
		}
		// Beanがiteratorの場合it.next()の検索
		// 親がMapかListの場合は必要無い
		// if (!(oyaBm.getBean() instanceof Map || oyaBm.getBean() instanceof
		// List)) {
		// Listはiteartorで取るように変更
		if (!(oyaBm.getBean() instanceof Map)) {
			if (oyaBm.getReturnObject() instanceof Iterator) {
				Iterator it = (Iterator) oyaBm.getReturnObject();
				while (it.hasNext()) {
					BeanMethod bm;
					try {
						bm = new BeanMethod(oyaBm, it);
						beanMetods.add(bm);
					} catch (Exception e) {
					}
				}
			}
		}
		// BeanがArrayの場合[n]検索
		if (oyaBm.getReturnObject().getClass().isArray()) {
			Object a = oyaBm.getReturnObject();
			for (int index = 0; index < Array.getLength(a); index++) {
				BeanMethod bm;
				try {
					bm = new BeanMethod(oyaBm, index);
					beanMetods.add(bm);
				} catch (Exception e) {
				}
			}
		}
		// Beanのメソッド検索
		Method[] methods = oyaBm.getReturnType().getMethods();
		for (int i = 0; i < methods.length; i++) {
			BeanMethod bm;
			try {
				bm = new BeanMethod(oyaBm, methods[i]);
				// クラス名もFiltercheckに含める様に変更
				String cname = Util.getExAopName(bm.getBean().getClass());
				if (bm.isMethodGetter()
						&& !Util.isFiltering(
								cname + "." + bm.getPropertyName(), filterMap2))
					beanMetods.add(bm);
			} catch (Exception e) {
			}
		}
		// 親に戻り値のメソッドの集合をセット
		oyaBm.setBeanMethods(beanMetods);
		// 普通のgetterの処理
		Iterator<BeanMethod> it = beanMetods.iterator();
		while (it.hasNext()) {
			BeanMethod bm = it.next();
			if (bm.isReturningWrapper())
				plugin.execElement(bm);
		}
		// Objectが返る場合の処理
		it = beanMetods.iterator();
		while (it.hasNext()) {
			BeanMethod bm = it.next();
			if (!bm.isReturningWrapper() && !bm.isReturningException()
					&& saikiCheck(bm)) {
				// 再帰処理
				execObject(bm, plugin);
			}
		}
	}

	private boolean saikiCheck(BeanMethod bm) {
		if (bm.getDepth() >= maxDepth)
			return false;
		return true;
	}

	private boolean isCheckOn(Object o) {
		// Filterにあるものは検査の対象にしない
		String b2Name = Util.getClassName(o);
		if (Util.isFiltering(b2Name, filterMap)) {
			return false;
		}
		// 同一オブジェクトは検査の対象にしない（javaパッケージは除く)
		try {
			if (!sameObject && checkList(objectList, o)
					&& !o.getClass().getName().startsWith("java"))
				return false;
			objectList.add(o);
		} catch (Exception e) {
		}
		return true;
	}

	/**
	 * contains(o)ではlist内に保持しておるオブジェクトが一致した場合でもtrueに なるため、使用できない。そこで、ここで独自に調査する。
	 * 
	 * @param l
	 * @param o
	 * @return
	 */
	private boolean checkList(List<Object> l, Object o) {
		Iterator<Object> it = l.iterator();
		while (it.hasNext()) {
			// if (it.next().equals(o)) {
			if (it.next() == o) {
				return true;
			}
		}
		return false;
	}

	public boolean isGetList() {
		return getList;
	}

	public void setGetList(boolean getList) {
		this.getList = getList;
	}

	public boolean isSameObject() {
		return sameObject;
	}

	public void setSameObject(boolean sameObject) {
		this.sameObject = sameObject;
	}

}
