package org.maskat.core.node;

/**
 * テーブルノード
 * <pre>
 * テーブル構造を表現するノード。
 * sampleTable
 * |    |col1|col2|col3|
 * |row1|   1|   4|   7|
 * |row2|   2|   5|   8|
 * |row3|   3|   6|   9|
 * を表現する場合、
 * <code>
 * TableNode table = new TableNode("sampleTable");
 * table.newRow("row1").add("col1", "1").add("col2", "4").add("col3", "7");
 * table.newRow("row1").add("col1", "2").add("col2", "5").add("col3", "8");
 * table.newRow("row1").add("col1", "3").add("col2", "6").add("col3", "9");
 * </code>
 * とする。
 * また上記は
 * <code>
 * TableNode table = new TableNode("sampleTable");
 * table.newRow("row1");
 * table.add("col1", "1");
 * table.add("col2", "4");
 * table.add("col3", "7");
 * table.newRow("row1");
 * table.add("col1", "2");
 * table.add("col2", "5");
 * table.add("col3", "8");
 * table.newRow("row1");
 * table.add("col1", "3");
 * table.add("col2", "6");
 * table.add("col3", "9");
 * </code>
 * と等価。
 * </pre>
 */
public class TableNode implements Node {

	private final String tableName;
	private CompositeNode currentRow = null;
	private CompositeNode rows = null;
	
	/**
	 * コンストラクタ
	 * @param tableName テーブル名
	 */
	public TableNode(String tableName) {
		this.tableName = tableName;
		rows = new CompositeNode();
	}
	
	/**
	 * 新規行作成
	 * @param rowName 行名
	 * @return this
	 */
	public TableNode newRow(String rowName) {
		currentRow = new CompositeNode();
		rows.add(new ParentNode(rowName, currentRow));
		return this;
	}
	
	/**
	 * 新規カラム作成
	 * <pre>
	 * <b>このメソッド呼び出しに先駆けてnewRow()を呼び出すこと。</b>
	 * </pre>
	 * @param colName カラム名
	 * @param value カラム値
	 * @return this
	 */
	public TableNode add(String colName, String value) {
		currentRow.add(new LeafNode(colName, value));
		return this;
	}

	public String getXML() {
		return new ParentNode(tableName, rows).getXML();
	}
}
