package org.maskat.core.gen;

import java.io.File;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

import org.maskat.core.MaskatConfig;
import org.xml.sax.SAXException;
/**
 * テンプレートジェネレータ
 * <pre>
 * 実行時にmaskat.propertyを読み込みます。
 * 必要な項目は
 * generate.src.path => ソースファイル出力パス (最後の「/」必須)
 * generate.xml.path => イベントXMLファイルパス (最後の「/」必須)
 * generate.code.chageline => 生成ソースの改行コード
 * 
 * 実行時引数にはイベントXMLのパスを指定できます。
 * 指定がある場合、指定のXMLファイルのみを生成対象とします。
 * 実行時引数がない場合はgenerate.xml.pathに指定されたフォルダ内の
 * *_e.xmlファイルをすべて処理します。
 * 
 * javaファイルがすでに存在している場合
 * そのjavaファイルの生成はスキップされます。
 * 
 * ex)
 * // 指定のXMLを生成対象とする場合 
 * java org.maskat.core.gen.MaskatTemplateGenerator &lt;eventXMLPath&gt;...
 * // generate.xml.path内のイベントファイルを生成対象とする場合
 * java org.maskat.core.gen.MaskatTemplateGenerator
 * </pre>
 */
public class MaskatTemplateGenerator {

	MaskatConfig config;

	public static void main(String[] args) throws SAXException, IOException {
		
		MaskatTemplateGenerator generator = new MaskatTemplateGenerator();
		
		generator.log("Start", "MaskatTemplateGenerator");
		
		if (0 == args.length) {
			// 引数がない場合 xmlPathのイベントファイルすべてが対象
			String[] eventFiles = generator.getEventXmlFiles();
			for (int i=0; i < eventFiles.length; i++) {
				generator.log("Analize", eventFiles[i]);
				generator.generate(eventFiles[i]);
			}
		} else {
			for (int i=0; i < args.length; i++) {
				generator.log("Analize", args[i]);
				generator.generate(args[i]);
			}
		}
		
		generator.log("End", "MaskatTemplateGenerator");
	}
	
	public MaskatTemplateGenerator() throws FileNotFoundException, IOException {
		config = new MaskatConfig();
	}
	
	/**
	 * generate.xml.pathからイベントXMLファイルパスの取得
	 * @return 見つかったイベントXMLファイルパスの配列
	 */
	private String[] getEventXmlFiles() {
		File eventFileDir = new File(config.getXmlPath());
		List eventFileList = getEventXmlFiles(eventFileDir);
		return (String[]) eventFileList.toArray(new String[eventFileList.size()]);
	}
	
	/**
	 * 指定ディレクトリからイベントXMLファイルパスの取得
	 * <pre>
	 * サブディレクトリを再帰的に検索します。
	 * </pre>
	 * @param directory ディレクトリ
	 * @return 見つかったイベントXMLファイルパスの配列
	 */
	private List getEventXmlFiles(File directory) {
		File[] eventFiles = directory.listFiles((new FileFilter() {
			public boolean accept(File pathname) {
				if (pathname.isDirectory()) {
					return true;
				}
				if(-1 != pathname.getName().indexOf("_e.xml")) {
					return true;
				}
				return false;
			}
		}));

		List filePaths = new ArrayList(eventFiles.length);
		for (int i = 0; i < eventFiles.length; i++) {
			if (eventFiles[i].isDirectory()) {
				filePaths.addAll(getEventXmlFiles(eventFiles[i]));
			} else {
				filePaths.add(eventFiles[i].getAbsolutePath());
			}
		}
		return filePaths;
	}
	
	/**
	 * クラス生成
	 * @param xmlpath 生成元イベントXMLパス
	 * @throws SAXException
	 * @throws IOException
	 */
	private void generate(String xmlpath) throws SAXException, IOException {
		
		EventXmlParser parser = new EventXmlParser(xmlpath);
		String packageName = parser.getPacageName();
		ClassContent[] classes = parser.getClassContents();
		
		if (generatePackage(packageName)) {
			for (int i = 0; i < classes.length; i++) {
				String content = generateClassContent(packageName, classes[i]);
				writeFile(packageName, classes[i].getClassName(), content);
			}
		} else {
			throw new IOException(packageName + "パッケージ作成に失敗");
		}
	}
	
	/**
	 * パッケージディレクトリ作成
	 * @param packageName パッケージ名
	 * @return true=正常終了
	 */
	private boolean generatePackage(String packageName) {
		if (null == packageName || "".equals(packageName)) {
			return true;
		}
		File current = new File(getFullPassByPackageName(packageName));
		if (current.exists()) {
			return true;
		}
		return current.mkdirs();
	}
	
	/**
	 * クラス内容を作成
	 * @param packageName パッケージ名
	 * @param classContent クラスコンテンツ
	 * @return ソース
	 */
	private String generateClassContent(String packageName, ClassContent classContent) {
		StringBuffer buffer = new StringBuffer();
		buffer.append("package " + packageName + ";" + config.getChangeLine());
		buffer.append(config.getChangeLine());
		buffer.append("import org.maskat.core.MaskatRequest;" + config.getChangeLine());
		buffer.append("import org.maskat.core.MaskatResponseBuilder;" + config.getChangeLine());
		buffer.append(config.getChangeLine());
		buffer.append("public class " + classContent.getClassName() + " {" + config.getChangeLine());
		buffer.append(config.getChangeLine());
		String[] methods = classContent.getClassMethosds();
		for (int i = 0; i < methods.length; i++) {
			buffer.append("\tpublic void " + methods[i] + "(MaskatRequest request, MaskatResponseBuilder builder) {" + config.getChangeLine());
			buffer.append("\t\t// TODO ここに処理コードを記述" + config.getChangeLine());
			buffer.append("\t}");
		}
		buffer.append(config.getChangeLine());
		buffer.append("}");
		return buffer.toString();
	}
	
	/**
	 * ファイル書き込み
	 * @param packageName パッケージ名
	 * @param className クラス名
	 * @param content ソース
	 * @throws IOException 
	 */
	private void writeFile(String packageName, String className, String content) throws IOException {
		String filePath = getFullPassByPackageName(packageName) + "/" + className + ".java";
		File file = new File(filePath);
		if (file.exists()) {
			log("Skip", "File exist " + packageName + "." +className + ".java");
			return;
		}

		FileOutputStream fileOutputStream = null;
		OutputStreamWriter outputStreamWriter = null;
		try {
			fileOutputStream = new FileOutputStream(filePath);
			outputStreamWriter = new OutputStreamWriter(fileOutputStream, Charset.forName(config.getCharSet()));
			outputStreamWriter.write(content);
		} finally {
			if (null != outputStreamWriter) {
				outputStreamWriter.close();
			}
			if (null != fileOutputStream) {
				fileOutputStream.close();
			}
		}
		log("Generate", packageName + "." +className + ".java");
	}
	
	/**
	 * パッケージ名からフルパスを取得
	 * @param packageName パッケージ名
	 * @return フルパス
	 */
	private String getFullPassByPackageName(String packageName) {
		String directry = packageName.replace('.', '/');
		return config.getSrcPath() + directry;
	}
	
	/**
	 * ログ出力
	 * <pre>
	 * [title]value
	 * の書式で出力される
	 * </pre>
	 * @param title タイトル
	 * @param detail 詳細 
	 */
	private void log(String title, String detail) {
		System.out.println("[" + title + "]" + detail);
	}
}
