/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.dog.core;

import static org.maru.common.type.PrimitiveWrapperTypes.isPrimitiveOrWrapperType;
import static org.maru.common.util.ConditionUtil.isNotNull;
import static org.maru.common.util.ConditionUtil.verifyNotNull;
import static org.maru.dog.core.ClassChecker.checkClassType;
import static org.maru.dog.core.ClassChecker.checkInstanceType;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.maru.common.Key;
import org.maru.common.KeyGen;
import org.maru.common.Reaper;
import org.maru.dog.Config;
import org.maru.dog.Director;
import org.maru.dog.bind.ActualFieldNameBinderBuilder;
import org.maru.dog.bind.ActualFieldNameMapBinderBuilder;
import org.maru.dog.bind.InstanceBinderBuilder;
import org.maru.dog.bind.InternalCallableBinder;
import org.maru.dog.bind.MapBinderBuilder;

/**
 * The implementation class of {@link Director}
 *
 */
public final class DirectorImpl implements Director {

    private static final ThreadLocal<Map<Key<?>, List<Configuration>>> LOCAL_CONFIGURATIONS = new ThreadLocal<Map<Key<?>, List<Configuration>>>(){

        @Override
        protected synchronized Map<Key<?>, List<Configuration>> initialValue() {
            return new HashMap<Key<?>, List<Configuration>>();
        }
    };

    private static final ThreadLocal<Map<Key<?>, List<Configuration>>> LOCAL_CONFIGURATIONS_FOR_FIELD_NAME_BINDINGS = new ThreadLocal<Map<Key<?>, List<Configuration>>>(){

        @Override
        protected synchronized Map<Key<?>, List<Configuration>> initialValue() {
            return new HashMap<Key<?>, List<Configuration>>();
        }
    };

    public DirectorImpl(Config... configs) {
        LOCAL_CONFIGURATIONS.remove();
        LOCAL_CONFIGURATIONS_FOR_FIELD_NAME_BINDINGS.remove();
        if(isNotNull(configs) && configs.length > 0) {
            initialize(configs);
        }
    }

    private void initialize(Config... configs) {
        Map<Key<?>, List<Configuration>> localConfigurations = LOCAL_CONFIGURATIONS.get();
        Map<Key<?>, List<Configuration>> localConfigForFieldNameBindings = LOCAL_CONFIGURATIONS_FOR_FIELD_NAME_BINDINGS.get();
        setupConfig(localConfigurations, localConfigForFieldNameBindings, Occasion.INITIALIZING, configs);
        LOCAL_CONFIGURATIONS.set(localConfigurations);
        LOCAL_CONFIGURATIONS_FOR_FIELD_NAME_BINDINGS.set(localConfigForFieldNameBindings);
        Reaper.execute();

    }

    private static enum Occasion {
        INITIALIZING,
        REPLACING
    }
    private static void setupConfig(Map<Key<?>, List<Configuration>> localConfigurations,
            Map<Key<?>, List<Configuration>> localConfigForFieldNameBindings, Occasion occasion, Config... configs) {

        for (Config config : configs) {
            config.make();
            Map<Key<?>, List<Configuration>> configurationsMap = config.getBindingDeclarator().getConfigurationMap();
            registerConfiguration(localConfigurations, configurationsMap, occasion);
            Map<Key<?>, List<Configuration>> configurationsForFieldNameMap = config.getBindingDeclarator().getConfigurationsForFieldNameMap();
            registerConfiguration(localConfigForFieldNameBindings, configurationsForFieldNameMap, occasion);
        }
    }

    private static void registerConfiguration(Map<Key<?>, List<Configuration>> localConfigurations, Map<Key<?>, List<Configuration>> definedConfigurations, Occasion occasion) {
        for (Map.Entry<Key<?>, List<Configuration>> entry : definedConfigurations.entrySet()) {
            Key<?> defKey = entry.getKey();
            List<Configuration> configurations = entry.getValue();

            if (configurations.size() > 0) {

                switch(occasion) {
                    case INITIALIZING:
                        if (!localConfigurations.containsKey(defKey)) {
                            localConfigurations.put(defKey, configurations);

                        } else {
                            throw new IllegalArgumentException("The definition related to " + defKey.toString() + " has already existed.");
                        }
                        break;
                    case REPLACING:
                        localConfigurations.put(defKey, configurations);
                        break;
                    default:
                        throw new IllegalArgumentException("Unexpected occasion");
                }

            }
        }
    }

    /**
     * @see org.maru.dog.Director#bind(Object, Object...)
     */
    public <T, K>Director bind(T target, K... inputs) {

        if(!isPrimitiveOrWrapperType(target.getClass())) {
            // validating the target and inputs instances.
            validateTargetAndInputs(target, inputs);

            InternalCallableBinder callableBinder = getBinder(target, inputs);
            callableBinder.callableBind();
        }
        return this;

    }

    /**
     * @see org.maru.dog.Director#bind(Object, Map)
     */
    public <T> Director bind(T target, Map<String, ?> inputs) {
        if (!isPrimitiveOrWrapperType(target.getClass())) {
            validateTargetAndInputs(target, inputs);

            InternalCallableBinder callableBinder = getMapBinder(target, inputs);
            callableBinder.callableBind();
        }
        return this;
    }

    /**
     * @see org.maru.dog.Director#bindByFieldName(Object, Object...)
     */
    public <T, K> Director bindByFieldName(T target, K... inputs) {
        if(!isPrimitiveOrWrapperType(target.getClass())) {
            // validating the target and inputs instances.
            validateTargetAndInputs(target, inputs);

            InternalCallableBinder callableBinder = getActualFieldNameBinder(target, inputs);
            callableBinder.callableBind();
        }
        return this;
    }

    /**
     * @see org.maru.dog.Director#bindByFieldName(Object, Map)
     */
    public <T> Director bindByFieldName(T target, Map<String, ?> inputs) {
        if (!isPrimitiveOrWrapperType(target.getClass())) {
            validateTargetAndInputs(target, inputs);

            InternalCallableBinder callableBinder = getActualFieldNameMapBinder(target, inputs);
            callableBinder.callableBind();
        }
        return this;
    }

    /**
     * @see org.maru.dog.Director#getInstance(Object, Object...)
     */
    public <T, K> T getInstance(T target, K... inputs) {
        if(!isPrimitiveOrWrapperType(target.getClass())) {
            validateTargetAndInputs(target, inputs);
            InternalCallableBinder callableBinder = getBinder(target, inputs);
            return  (T)callableBinder.callableBind();
        } else {
            return target;
        }
    }

    /**
     * @see org.maru.dog.Director#getInstance(Object, Map)
     */
    public <T> T getInstance(T target, Map<String, ?> inputs) {
        if (!isPrimitiveOrWrapperType(target.getClass())) {
            validateTargetAndInputs(target, inputs);

            InternalCallableBinder callableBinder = getMapBinder(target, inputs);
            return (T) callableBinder.callableBind();
        } else {
            return target;
        }
    }

    /**
     * @see org.maru.dog.Director#getInstanceByFieldNameBind(Object, Object...)
     */
    public <T, K> T getInstanceByFieldNameBind(T target, K... inputs) {
        if(!isPrimitiveOrWrapperType(target.getClass())) {
            validateTargetAndInputs(target, inputs);
            InternalCallableBinder callableBinder = getActualFieldNameBinder(target, inputs);
            return  (T) callableBinder.callableBind();
        } else {
            return target;
        }
    }

    /**
     * @see org.maru.dog.Director#getInstanceByFieldNameBind(Object, Map)
     */
    public <T> T getInstanceByFieldNameBind(T target, Map<String, ?> inputs) {
        if(!isPrimitiveOrWrapperType(target.getClass())) {
            validateTargetAndInputs(target, inputs);
            InternalCallableBinder callableBinder = getActualFieldNameMapBinder(target, inputs);
            return (T) callableBinder.callableBind();
        } else {
            return target;
        }
    }

    /**
     * @see org.maru.dog.Director#getInstance(Class, Object...)
     */
    public <T, K> T getInstance(Class<T> target, K... inputs) {
        if(!isPrimitiveOrWrapperType(target)) {
            validateTargetAndInputs(target, inputs);
            T instance = ObjectFactory.newInstance(target);

            InternalCallableBinder callableBinder = getBinder(instance, inputs);
            return (T) callableBinder.callableBind();
        } else {
            throw new IllegalArgumentException("The target argument is primitive or wrapper type. Can not bind dependencies to " + target.getName());
        }
    }

    /**
     * @see org.maru.dog.Director#getInstance(Class, Object...)
     */
    public <T> T getInstance(Class<T> target, Map<String, ?> inputs) {
        if(!isPrimitiveOrWrapperType(target)) {
            validateTargetAndInputs(target, inputs);
            T instance = ObjectFactory.newInstance(target);

            InternalCallableBinder callableBinder = getMapBinder(instance, inputs);
            return (T) callableBinder.callableBind();
        } else {
            throw new IllegalArgumentException("The target argument is primitive or wrapper type. Can not bind dependencies to " + target.getName());
        }
    }

    /**
     * @see org.maru.dog.Director#getInstanceByFieldNameBind(Class, Object...)
     */
    public <T, K> T getInstanceByFieldNameBind(Class<T> target, K... inputs) {
        if(!isPrimitiveOrWrapperType(target)) {
            validateTargetAndInputs(target, inputs);
            T instance = ObjectFactory.newInstance(target);

            InternalCallableBinder callableBinder = getActualFieldNameBinder(instance, inputs);
            return (T) callableBinder.callableBind();
        } else {
            throw new IllegalArgumentException("The target argument is primitive or wrapper type. Can not bind dependencies to " + target.getName());
        }
    }

    /**
     * @see org.maru.dog.Director#getInstanceByFieldNameBind(Class, Map)
     */
    public <T> T getInstanceByFieldNameBind(Class<T> target, Map<String, ?> inputs) {
        if(!isPrimitiveOrWrapperType(target)) {
            validateTargetAndInputs(target, inputs);
            T instance = ObjectFactory.newInstance(target);

            InternalCallableBinder callableBinder = getActualFieldNameMapBinder(instance, inputs);
            return (T) callableBinder.callableBind();
        } else {
            throw new IllegalArgumentException("The target argument is primitive or wrapper type. Can not bind dependencies to " + target.getName());
        }
    }

    /**
     * @see org.maru.dog.Director#removeAllConfig()
     */
    public Director removeAllConfig() {
        LOCAL_CONFIGURATIONS.remove();
        LOCAL_CONFIGURATIONS_FOR_FIELD_NAME_BINDINGS.remove();
        return this;
    }

    /**
     * @see org.maru.dog.Director#replaceConfig(Config...)
     */
    public Director replaceConfig(Config... configs) {
        Map<Key<?>, List<Configuration>> localConfigurations = LOCAL_CONFIGURATIONS.get();
        Map<Key<?>, List<Configuration>> localConfigForFieldNameBindings = LOCAL_CONFIGURATIONS_FOR_FIELD_NAME_BINDINGS.get();
        setupConfig(localConfigurations, localConfigForFieldNameBindings, Occasion.REPLACING, configs);
        LOCAL_CONFIGURATIONS.set(localConfigurations);
        LOCAL_CONFIGURATIONS_FOR_FIELD_NAME_BINDINGS.set(localConfigForFieldNameBindings);
        return this;
    }

    /**
     * Retruns {@link InternalCallableBinder} for default bindings. The default
     * bindings indicate that it's to process the bindings which qualify the
     * proper annotations.
     *
     * @param target target object
     * @param inputs input objects.
     * @return {@link InternalCallableBinder}
     */
    private <T, K> InternalCallableBinder getBinder(T target, K... inputs) {
        List<Configuration> configurations = getLocalConfigurations(target);
        InstanceBinderBuilder<T, K> builder = new InstanceBinderBuilder<T, K>(configurations, target, inputs);
        builder.build();
        return builder.getInstanceBinder().getCallableBinder(target, inputs);
    }

    /**
     * Retruns {@link InternalCallableBinder} for the field name bindings.
     * @param target target object
     * @param inputs input objects.
     * @return {@link InternalCallableBinder}
     */
    private <T, K> InternalCallableBinder getActualFieldNameBinder(T target, K... inputs) {
        List<Configuration> configurations = getLocalConfigurationForFieldNameBindings(target);
        ActualFieldNameBinderBuilder<T, K> builder = new ActualFieldNameBinderBuilder<T, K>(configurations, target, inputs);
        builder.build();
        return builder.getInstanceBinder().getCallableBinder(target, inputs);
    }

    /**
     * Retruns {@link InternalCallableBinder} for default map data bindings.
     *
     * @param target target object
     * @param inputs map type input data.
     * @return {@link InternalCallableBinder}
     */
    private <T> InternalCallableBinder getMapBinder(T target, Map<String, ?> inputs) {
        MapBinderBuilder<T> builder = new MapBinderBuilder<T>(target, inputs);
        builder.build();
        return builder.getMapBinder().getCallableBinder(target, inputs);
    }

    /**
     * Retruns {@link InternalCallableBinder} for default map data bindings.
     *
     * @param target target object
     * @param inputs map type input data.
     * @return {@link InternalCallableBinder}
     */
    private <T> InternalCallableBinder getActualFieldNameMapBinder(T target, Map<String, ?> inputs) {
        ActualFieldNameMapBinderBuilder<T> builder = new ActualFieldNameMapBinderBuilder<T>(target, inputs);
        builder.build();
        return builder.getMapBinder().getCallableBinder(target, inputs);
    }

    private <T> List<Configuration> getLocalConfigurations(T target) {
        Key<?> key = KeyGen.getKey(target.getClass());
        return LOCAL_CONFIGURATIONS.get().get(key);
    }

    private <T> List<Configuration> getLocalConfigurationForFieldNameBindings(T target) {
        Key<?> key = KeyGen.getKey(target.getClass());
        return LOCAL_CONFIGURATIONS_FOR_FIELD_NAME_BINDINGS.get().get(key);
    }

    private static <T, K> void validateTargetAndInputs(T target, K... inputs) {
        checkNotNull(target, inputs);
        checkInstanceType(target);
        checkInputInstanceTypes(inputs);
    }

    private static <T, K> void validateTargetAndInputs(T target, Map<String, ?> inputs) {
        checkNotNull(target, inputs);
        checkInstanceType(target);
    }

    private static <T, K> void validateTargetAndInputs(Class<T> target, K... inputs) {
        checkNotNull(target, inputs);
        checkClassType(target);
        checkInputInstanceTypes(inputs);
    }

    private static <T, K> void validateTargetAndInputs(Class<T> target, Map<String, ?> inputs) {
        checkNotNull(target, inputs);
        checkClassType(target);
    }

    private static <T, K> void checkNotNull(T target, K... inputs) {
        verifyNotNull(target, "The target instance is null.");
        for (K input : inputs) {
            verifyNotNull(input, "One of the input instances is null.");
        }
    }

    private static <T, K> void checkNotNull(T target, Map<String, ?> input) {
        verifyNotNull(target, "The target instance is null.");
        verifyNotNull(input, "The map type input is null.");
    }


    private static void checkInputInstanceTypes(Object... inputs) {
        if (isNotNull(inputs) && inputs.length > 0) {
            for (Object input : inputs) {
                checkInstanceType(input);
            }
        } else {
            throw new IllegalArgumentException("The input objects must need at least one instance.");
        }
    }

}
