/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.dog.core;

import java.lang.reflect.Field;
import java.lang.reflect.Member;
import java.lang.reflect.Method;
import java.lang.reflect.Type;

import org.maru.common.AbstractClassAnalyzer;
import org.maru.common.type.GenericClassType;
import org.maru.common.util.StringUtil;
import org.maru.dog.util.RegexpUtil;

/**
 * Abstract marked builder.
 *
 */
public abstract class AbstractDefinitionBuilder<T> extends AbstractClassAnalyzer<T> implements Builder {

    private static final String UPPERCSE_ALL_ALPHABET = "[A-Z]";

    protected Class<T> target;

    protected final Definition<T> definition;

    public AbstractDefinitionBuilder(Class<T> target, Definition<T> definition) {
        this.target = target;
        this.definition = definition;
    }

    /**
     * build structure information of a class.
     */
    public Builder build() {
        scan(target);
        return this;
    }

    protected abstract boolean isStoredDefinition();

    protected abstract Definition<T> applyDefinition();

    protected abstract Definition<T> retrieveDefinition();


    public Definition<T> createAndStoreDefinition() {
        if (!isStoredDefinition()) {
            build();
            applyDefinition();
            return this.definition;
        } else {
            return retrieveDefinition();
        }
    }


    public void setTarget(Class<T> target) {
        this.target = target;
    }


    @SuppressWarnings("unchecked")
    @Override
    protected void scan(Class<T> clazz) {
        Type superClass = clazz.getGenericSuperclass();

        Class<? super T> classType = (Class<? super T>) GenericClassType.getClassFromType(superClass);
        if (classType != null) {
            recursiveClassScan(classType);
        }

        Type[] interfaces = clazz.getGenericInterfaces();
        scanInterfaces(interfaces);
        createStructureInformation(clazz);
    }

    /**
     * create {@link MarkedPoint} object.
     */
    @Override
    protected void createStructureInformation(Class<? super T> target) {
        Field[] fields = target.getDeclaredFields();
        for (Field f : fields) {
            makeMarkedPoint(f);
        }

        Method[] methods = target.getDeclaredMethods();
        for (Method m : methods) {
            makeMarkedPoint(m);
        }
    }

    /**
     * Make {@link MarkedPoint} instance and store it into map.
     *
     * @param member
     */
    public abstract void makeMarkedPoint(Member member);



    /**
     * Change converter method name to key name. This method just convert the
     * first character of method name to lower case.
     *
     * @param keyName
     *            converted method name.
     * @return key name
     */
    protected static String canonicalizeMethodName(String keyName) {
        if (keyName.startsWith("get") || keyName.startsWith("set")) {
            // remove "get" or "set" from method name.
            keyName = keyName.substring(3, keyName.length());
        } else if (keyName.startsWith("is")) {
            keyName = keyName.substring(2, keyName.length());
        }

        if (StringUtil.isEmpty(keyName)) {
            return keyName;
        }
        char[] ch = keyName.toCharArray();
        String fc = Character.toString(ch[0]);
        /*
         * If a first character is upper case, then check second character.
         * If the second character is not upper case, finally convert first
         * character to lower case.
         */
        if (RegexpUtil.matches(fc, UPPERCSE_ALL_ALPHABET)) {
            if (ch.length > 1
                    && !RegexpUtil.matches(Character.toString(ch[1]),
                            UPPERCSE_ALL_ALPHABET)) {
                fc = fc.toLowerCase();
                ch[0] = fc.charAt(0);
                keyName = new String(ch);
            }

        }

        return keyName;
    }
}
