/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.dog.bind;

import static org.maru.common.type.PrimitiveWrapperTypes.getWrapperType;
import static org.maru.common.type.PrimitiveWrapperTypes.isPrimitiveType;
import static org.maru.common.util.ConditionUtil.isNotNull;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Type;

import org.maru.common.type.GenericClassType;
import org.maru.dog.core.MarkedPoint;

public final class TypeSafeChecker {

    private TypeSafeChecker(){}

    /**
     * Verifies if input marked point and target marked point are functionally
     * equivalent.
     *
     * @param inputMarkedPoint
     *            input marked point.
     * @param targetMarkedPoint
     *            target marked point.
     */
    public static void verifyTypeSafe(InputMarkedPoint<?> inputMarkedPoint,
            TargetMarkedPoint<?> targetMarkedPoint) {
        if (isNotNull(inputMarkedPoint.getConverterDef().converterClass)) {
            verifyTypeSafeWithConverter(inputMarkedPoint, targetMarkedPoint);
        } else {
            verifyTypeSafeWithNoConverter(inputMarkedPoint, targetMarkedPoint);
        }
    }

    /*
     * type safe check method including converter method.
     */
    private static void verifyTypeSafeWithConverter(
            InputMarkedPoint<?> inputMarkedPoint, TargetMarkedPoint<?> targetMarkedPoint) {
        GenericClassType inputType = getInputGenericClassType(inputMarkedPoint);

        Type convertedValueType = inputMarkedPoint.getConverterDef().convertedValueType;
        GenericClassType parameter = GenericClassType.getGenericClassType(
                isPrimitiveType(convertedValueType) ? getWrapperType(convertedValueType) : convertedValueType);

        Type returnValueType = inputMarkedPoint.getConverterDef().converterMethod.getGenericReturnType();
        GenericClassType returnType = GenericClassType.getGenericClassType(
                isPrimitiveType(returnValueType) ? getWrapperType(returnValueType) : returnValueType);

        GenericClassType targetType = getTargetGenericClassType(targetMarkedPoint);

        checkTypeSafe(inputType, parameter, returnType, targetType);
    }

    /*
     * type safe check method not including converter method.
     */
    private static void verifyTypeSafeWithNoConverter(
            InputMarkedPoint<?> inputMarkedPoint, TargetMarkedPoint<?> targetMarkedPoint) {
        GenericClassType inputType = getInputGenericClassType(inputMarkedPoint);

        GenericClassType targetType = getTargetGenericClassType(targetMarkedPoint);

        checkTypeSafe(inputType, targetType);
    }

    /**
     * Gets {@link GenericClassType} in the input marked point.
     *
     * @param markedPoint
     *            input marked point.
     * @return {@link GenericClassType} object.
     */
    static GenericClassType getInputGenericClassType(
            MarkedPoint<?> markedPoint) {
        GenericClassType genericClassType;
        if (markedPoint.getMember() instanceof Field) {
            Field field = (Field) markedPoint.getMember();
            Type type = field.getGenericType();
            genericClassType = GenericClassType.getGenericClassType(
                    isPrimitiveType(type) ? getWrapperType(type) : type);
        } else {
            Method method = (Method) markedPoint.getMember();
            Type type = method.getGenericReturnType();
            genericClassType = GenericClassType.getGenericClassType(
                    isPrimitiveType(type) ? getWrapperType(type) : type);
        }
        return genericClassType;
    }

    /**
     * Gets {@link GenericClassType} in the target marked point.
     *
     * @param markedPoint
     *            target marked point.
     * @return {@link GenericClassType} object.
     */
    static GenericClassType getTargetGenericClassType(
            MarkedPoint<?> markedPoint) {
        GenericClassType genericClassType;
        if (markedPoint.getMember() instanceof Field) {
            Field field = (Field) markedPoint.getMember();
            Type type = field.getGenericType();
            genericClassType = GenericClassType.getGenericClassType(
                    isPrimitiveType(type) ? getWrapperType(type) : type);
        } else {
            Method method = (Method) markedPoint.getMember();
            Type type = method.getGenericParameterTypes()[0];
            genericClassType = GenericClassType.getGenericClassType(
                    isPrimitiveType(type) ? getWrapperType(type) : type);
        }
        return genericClassType;
    }

    static void checkTypeSafe(GenericClassType inputType,
            GenericClassType parameter, GenericClassType returnType,
            GenericClassType targetType) {
        if (!(inputType.equals(parameter) && targetType.equals(returnType))) {
            throw new IllegalArgumentException("The inpout type ["
                    + inputType.getType()
                    + "] and the parameter type on converter method ["
                    + parameter.getType() + "], the target type ["
                    + targetType.getType()
                    + "] and the return type on converter method ["
                    + returnType.getType() + "] are not type safe.");
        }
    }

    static void checkTypeSafe(GenericClassType inputType,
            GenericClassType targetType) {
        if (!inputType.equals(targetType)) {
            throw new IllegalArgumentException("The input type ["
                    + inputType.getType() + " and the target type ["
                    + targetType.getType() + "] are not type safe.");
        }
    }
}
