/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.dog.bind;

import static org.maru.common.util.ConditionUtil.verifyNotNull;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.maru.common.Key;
import org.maru.common.KeyGen;
import org.maru.common.annotation.Identifier;
import org.maru.common.annotation.Value;
import org.maru.common.type.GenericClassType;
import org.maru.common.util.StringUtil;
import org.maru.dog.converter.Converter;

final class ConverterDef {

    Class<?> converterClass = null;
    Method converterMethod = null;

    /**
     *  The converter annotation {@link @Converter} or any annotations that are marked
     * {@link @ConverterIdentifier}
     */
    Annotation converterAnnotation = null;

    /**
     * A type of converted value. the converted value on converter method
     * must be positioned at first parameter.
     */
    Type convertedValueType = null;

    /**
     * The list of converter's method parameters.
     */
    final List<Key<?>> identifierParameters = new ArrayList<Key<?>>();

    ConverterDef() {
    }

    ConverterDef(Class<?> converterClass, Method converterMethod,
            Annotation converterAnnotation) {
        this.converterClass = converterClass;
        if (!converterMethod.isAccessible()) {
            converterMethod.setAccessible(true);
        }
        this.converterMethod = converterMethod;
        this.converterAnnotation = converterAnnotation;
    }

    void verifyConverter() {
        verifyNotNull(converterClass, "The converter class is null.");
        verifyNotNull(converterMethod,"The converter method is null.");
        verifyNotNull(converterAnnotation, "The converter annotation is null.");

        if (converterAnnotation instanceof Converter) {
            convertedValueType = converterMethod.getGenericParameterTypes()[0];
        } else {
            /*
             * get and check the first parameter is annotated @Value for
             * representing converted value.
             */
            Annotation[][] parameterAnnotations = converterMethod.getParameterAnnotations();
            for (int i = 0; i < parameterAnnotations.length; i++) {
                for (int j = 0; j < parameterAnnotations[i].length; j++) {

                    if ((parameterAnnotations[i][j] instanceof Value)) {
                        Key<?> valueKey = KeyGen.getKey(parameterAnnotations[i][j].annotationType());
                        if (identifierParameters.contains(valueKey)) {
                            throw new IllegalArgumentException(valueKey + "is not unique.");
                        }
                        identifierParameters.add(valueKey);
                        convertedValueType = converterMethod.getGenericParameterTypes()[i];
                    }
                    if ((parameterAnnotations[i][j] instanceof Identifier)) {
                        Identifier identifier = (Identifier) parameterAnnotations[i][j];
                        String identifierName = identifier.value();
                        if (StringUtil.isNotEmpty(identifierName)) {
                            Key<?> identifierNameKey = KeyGen.getKey(identifierName);
                            if (identifierParameters.contains(identifierNameKey)) {
                                throw new IllegalArgumentException(identifierNameKey + "is not unique.");
                            }
                            identifierParameters.add(identifierNameKey);
                        } else {
                            throw new IllegalArgumentException("The name of Identification is not set to the parameter.");
                        }
                    }
                }
            }

            if (convertedValueType == null) {
                throw new IllegalArgumentException(
                        "There is no @Value annotation on the parameter of the covnerter method.");
            }

            if (converterAnnotation.annotationType().getDeclaredMethods().length != identifierParameters.size() - 1) {
                throw new IllegalArgumentException(
                        "Attributes of converter annotation and parameter's attributes on a converter method "
                                + "must be same length.");
            }
        }
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("ConverterDef [attributeParameters=" + identifierParameters);
        sb.append(", convertedValueType=" + convertedValueType);
        sb.append(", converterAnnotation=" + converterAnnotation);
        sb.append(", converterClass=" + converterClass);
        sb.append(", converterMethod=" + converterMethod + "]");
        return sb.toString();
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((convertedValueType == null) ? 0 : convertedValueType.hashCode());
        result = prime * result + ((converterAnnotation == null) ? 0 : converterAnnotation.hashCode());
        result = prime * result + ((converterClass == null) ? 0 : converterClass.hashCode());
        result = prime * result + ((converterMethod == null) ? 0 : converterMethod.hashCode());
        result = prime * result + ((identifierParameters == null) ? 0 : identifierParameters.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        ConverterDef other = (ConverterDef) obj;
        if (convertedValueType == null) {
            if (other.convertedValueType != null)
                return false;
        } else if (!GenericClassType.getGenericClassType(convertedValueType).equals(GenericClassType.getGenericClassType(other.convertedValueType)))
            return false;
        if (converterAnnotation == null) {
            if (other.converterAnnotation != null)
                return false;
        } else if (!converterAnnotation.equals(other.converterAnnotation))
            return false;
        if (converterClass == null) {
            if (other.converterClass != null)
                return false;
        } else if (!converterClass.equals(other.converterClass))
            return false;
        if (converterMethod == null) {
            if (other.converterMethod != null)
                return false;
        } else if (!converterMethod.equals(other.converterMethod))
            return false;
        if (identifierParameters == null) {
            if (other.identifierParameters != null)
                return false;
        } else if (!Arrays.equals(identifierParameters.toArray(), other.identifierParameters.toArray()))
            return false;
        return true;
    }

}
