/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.dog.bind;

import static org.maru.common.type.PrimitiveWrapperTypes.isPrimitiveOrWrapperType;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.maru.common.Key;
import org.maru.common.KeyGen;
import org.maru.dog.core.Builder;
import org.maru.dog.core.Configuration;
import org.maru.dog.core.Definition;

abstract class AbstractBinderBuilder<T, K> implements Builder, DefinitionBuilder {

    protected InstanceBinder<T, K> instanceBinder;
    protected final List<Configuration> configurations;

    protected final ClassBuilder<T> targetBuilder;
    protected final List<ClassBuilder<K>> inputBuilders;

    public AbstractBinderBuilder(List<Configuration> configurations, InstanceBinder<T, K> instanceBinder,
            T target, K... inputs) {
        this.targetBuilder = new TargetClassBuilder<T>(target, this);
        this.inputBuilders = createInputClasBuilder(inputs);

        this.configurations = configurations;
        this.instanceBinder = instanceBinder;
    }

    @SuppressWarnings("unchecked")
    private List<ClassBuilder<K>> createInputClasBuilder(K... inputs) {
        List<ClassBuilder<K>> inputBuilders = new ArrayList<ClassBuilder<K>>();
        for (K input : inputs) {
            Class<K> inputClass = (Class<K>) input.getClass();
            if (isPrimitiveOrWrapperType(inputClass)) {
                continue;
            }
            inputBuilders.add(new InputClassBuilder<K>(input, this));
        }
        return inputBuilders;
    }

    public InstanceBinder<T, K> getInstanceBinder() {
        return this.instanceBinder;
    }

    @SuppressWarnings("unchecked")
    public Builder build() {
        Map<Key<?>,Definition<K>> inputDefinitions = getInputDefinitions();

        targetBuilder.build();
        Definition<T> targetDefinition = targetBuilder.getDefinition();

        BinderProducer binderProducer = new InstanceBinderProducer<T, K>(
                targetDefinition, inputDefinitions, configurations,
                (InstanceBinderImpl<T, K>) instanceBinder);
        binderProducer.validateAndComposeBinder();
        instanceBinder = (InstanceBinder<T, K>) binderProducer.getBinder();
        return this;
    }

    private Map<Key<?>,Definition<K>> getInputDefinitions() {
        Map<Key<?>,Definition<K>> definitions = new HashMap<Key<?>,Definition<K>>();
        for (ClassBuilder<K> builder : inputBuilders) {
            builder.build();
            Key<?> key = KeyGen.getKey(builder.getType());
            Definition<K> definition = builder.getDefinition();
            definitions.put(key, definition);
        }
        return definitions;
    }

}
