/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.dog.bind;

import static org.maru.common.util.ConditionUtil.isNotNull;

import java.lang.reflect.Field;
import java.lang.reflect.Member;
import java.lang.reflect.Method;

import org.maru.common.Key;
import org.maru.common.KeyGen;
import org.maru.common.util.StringUtil;
import org.maru.dog.DefinitionException;
import org.maru.dog.annotation.Bound;
import org.maru.dog.annotation.From;
import org.maru.dog.core.Definition;
import org.maru.dog.core.MarkedPoint;

import static org.maru.dog.util.MethodUtil.isSetterMethod;;

final class TargetMarkedDefinitionBuilder<T> extends AbstractTargetDefinitionBuilder<T> {

    TargetMarkedDefinitionBuilder(Class<T> type) {
        super(type, new BindingDefinitionImpl<T>(type));
    }

    @Override
    public void makeMarkedPoint(Member member) {
        MarkedPoint<T> markedPoint = getTargetMarkedPoint(member, target);
        if (isNotNull(markedPoint)) {
            String keyName = markedPoint.getName();
            Key<?> key = KeyGen.getKey(keyName);
            ((BindingDefinition<T>)definition).putMarkedPoint(key, markedPoint);
        }
    }

    @Override
    protected TargetMarkedPoint<T> getTargetMarkedPoint(Member member, Class<T> target) {
        TargetMarkedPoint<T> markedPoint = null;
        if (member instanceof Field) {
            Field f = (Field) member;
            Bound bound = (Bound) f.getAnnotation(Bound.class);
            String keyName = f.getName();

            if (bound != null) {
                if (StringUtil.isNotEmpty(bound.name())) {
                    keyName = bound.name();
                }
            } else {
                return null;
            }

            Class<?> inputClass = null;

            From from = f.getAnnotation(From.class);
            if (from != null) {
                inputClass = from.value();
            }

            // create marked point
            markedPoint = createTargetMarkedPointInstance(target, keyName, f, inputClass);

        }
        if (member instanceof Method) {
            Method m = (Method) member;
            Bound bound = (Bound) m.getAnnotation(Bound.class);

            if (bound != null) {
                String methodName = m.getName();

                // check whether or not the method is the java beans accessor.
                if (isSetterMethod(methodName, m)) {

                    String keyName = null;

                    if (StringUtil.isNotEmpty(bound.name())) {
                        keyName = bound.name();
                    } else {
                        /*
                         * convert keyName from method name. Remove "get" or "set"
                         * from method name. If the name of the keyName
                         * starts with Upper case, change it to lower case.
                         */
                        keyName = canonicalizeMethodName(methodName);
                    }

                    if (StringUtil.isEmpty(keyName)) {
                        throw new IllegalArgumentException(
                                "Fail to make bound identifier. It might not be set the bound name, or setter method name is illegal style");
                    }

                    Class<?> inputClass = null;

                    From from = m.getAnnotation(From.class);
                    if (from != null) {
                        inputClass = from.value();
                    }

                    markedPoint = createTargetMarkedPointInstance(target, keyName, m, inputClass);

                } else {
                    throw new DefinitionException(m.getName() + " method is not setter method following JavaBeans.");
                }
            }
        }
        return markedPoint;
    }

    @Override
    protected boolean isStoredDefinition() {
       return BindingDefinitionCache.containsTargetDefinition(KeyGen.getKey(target));
    }

    @SuppressWarnings("unchecked")
    @Override
    protected Definition<T> applyDefinition() {
        return (Definition<T>) BindingDefinitionCache.applyTargetDefinition(KeyGen.getKey(target), definition);
    }

    @SuppressWarnings("unchecked")
    @Override
    protected Definition<T> retrieveDefinition() {
        return (Definition<T>) BindingDefinitionCache.getTargetDefinition(KeyGen.getKey(target));
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("TargetMarkedDefinitionBuilder [target=" + target + ",");
        sb.append("definition=" + definition + "]");
        return  sb.toString();
    }
}
