/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.dog.bind;

import static org.maru.common.util.ConditionUtil.isNotNull;
import static org.maru.dog.bind.BinderFactory.createMemberBinder;
import static org.maru.dog.bind.TypeSafeChecker.verifyTypeSafe;

import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.maru.common.Key;
import org.maru.common.KeyGen;
import org.maru.dog.core.Binder;
import org.maru.dog.core.Configuration;
import org.maru.dog.core.Definition;
import org.maru.dog.core.MarkedPoint;

final class InstanceBinderProducer<T, K> implements BinderProducer {
    private final Definition<T> targetDefinition;
    private final Map<Key<?>,Definition<K>> inputDefinitions;
    private final InstanceBinderImpl<T, K> instanceBinder;
    private final List<Configuration> configurations;

    InstanceBinderProducer(Definition<T> targetDefinition,
            Map<Key<?>,Definition<K>> inputDefinitions,
            List<Configuration> configurations,
            InstanceBinderImpl<T, K> instanceBinder) {
        this.targetDefinition = targetDefinition;
        this.inputDefinitions = inputDefinitions;
        this.instanceBinder = instanceBinder;
        this.configurations = configurations;
    }

    public  Binder getBinder() {
        return instanceBinder;
    }

    public void validateAndComposeBinder() {
        Key<?> targetTypeKey = KeyGen.getKey(targetDefinition.getType());
        for (Entry<Key<?>, MarkedPoint<T>> entry : ((BindingDefinition<T>)targetDefinition).getMarkedPoints().entrySet()) {
            Key<?> targetBoundKey = entry.getKey();
            TargetMarkedPoint<T> targetMarkedPoint = (TargetMarkedPoint<T>)entry.getValue();
            Class<?> inputClass = null;

            //  get binding configuration if it's defined.
            BindingConfiguration bindingConfiguration = searchBindingConfiguration(targetTypeKey, targetBoundKey);
            if (isNotNull(bindingConfiguration)) {
                inputClass = bindingConfiguration.getInputClass();
                Key<?> bindKey = isNotNull(bindingConfiguration.getBindName()) ? KeyGen.getKey(bindingConfiguration.getBindName()) : targetBoundKey;

                // if binding context is created, then go to next loop
                BindingContextImpl context = createBindingContext(targetMarkedPoint, bindKey, inputDefinitions, inputClass, bindingConfiguration);

                if (isNotNull(context)) {
                    MemberBinder memberBinder = createMemberBinder(context, new BindingFactoryImpl());
                    instanceBinder.addMemberBinder(memberBinder);
                }
                continue;
            }

            inputClass = targetMarkedPoint.getInputClass();

            BindingContextImpl context = createBindingContext(targetMarkedPoint, targetBoundKey, inputDefinitions, inputClass, null);
            if (isNotNull(context)) {
                MemberBinder memberBinder = createMemberBinder(context, new BindingFactoryImpl());
                instanceBinder.addMemberBinder(memberBinder);
            }

        }
    }

    /**
     * search and get defined {@link BindingConfiguration} if it's defined in LOCAL_CONFIGURATION_INFOS
     *
     * @param targetTypeKey
     * @param targetBoundKey
     * @return {@link BindingConfiguration}
     */
    private BindingConfiguration searchBindingConfiguration(Key<?> targetTypeKey, Key<?> targetBoundKey) {

        BindingConfiguration configuration = null;
        if (isNotNull(configurations) && configurations.size() > 0) {
            for (Configuration c : configurations) {
                Key<?> configKey = KeyGen.getKey(((BindingConfiguration)c).getBoundName());
                if (targetBoundKey.equals(configKey)) {
                    configuration = (BindingConfiguration)c;
                    break;
                }
            }
        }
        return configuration;
    }

    /**
     * Creates and returns {@link BindingContextImpl}
     */
    @SuppressWarnings("unchecked")
    private static <T, K> BindingContextImpl createBindingContext(TargetMarkedPoint<T> targetMarkedPoint, Key<?> targetBoundKey,
            Map<Key<?>,Definition<K>> inputDefinitions, Class<?> inputClass, BindingConfiguration configuration) {

        BindingContextImpl bindingContext = null;
        if (isNotNull(inputClass)) {
            Key<?> inputClassKey = KeyGen.getKey(inputClass);
            if (inputDefinitions.containsKey(inputClassKey)) {
                Definition<?> inputDefinition = inputDefinitions.get(inputClassKey);

                if (((BindingDefinition<K>)inputDefinition).containsMarkedPoint(targetBoundKey)) {
                    InputMarkedPoint<K> inputMarkedPoint = (InputMarkedPoint<K>)((BindingDefinition<K>)inputDefinition).getMarkedPoint(targetBoundKey);

                    // type safe check.
                    verifyTypeSafe(inputMarkedPoint, targetMarkedPoint);

                    bindingContext = new BindingContextImpl(targetMarkedPoint, inputMarkedPoint, configuration);
                }
            }
        } else {
            int duplicateCount = 0;
            InputMarkedPoint<K> inputMarkedPoint = null;
            for (Entry<Key<?>, Definition<K>> inputEntry : inputDefinitions.entrySet()) {
                for (Entry<Key<?>, MarkedPoint<K>> definedEntry : ((BindingDefinition<K>) inputEntry.getValue()).getMarkedPoints().entrySet()) {
                    InputMarkedPoint<K> tmpInputMarkedPoint = (InputMarkedPoint<K>)definedEntry.getValue();
                    if (tmpInputMarkedPoint.getName().equals(targetMarkedPoint.getName())) {
                        if (duplicateCount == 0) {
                            inputMarkedPoint = tmpInputMarkedPoint;
                        }
                        duplicateCount++;
                    }
                }
            }

            if (duplicateCount > 1) {
                throw new IllegalArgumentException("The name [" + inputMarkedPoint.getName() + "] is duplicate.");
            }

            if (duplicateCount == 1) {
                // type safe check.
                verifyTypeSafe(inputMarkedPoint, targetMarkedPoint);

                bindingContext = new BindingContextImpl(targetMarkedPoint, inputMarkedPoint, configuration);
            }
        }
        return bindingContext;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("InstanceBinderProducer [targetDefinition=" + targetDefinition + ",");
        sb.append("inputDefinitions=" + inputDefinitions + ",");
        sb.append("instanceBinder=" + instanceBinder + ",");
        sb.append("configurations=" + configurations + "]");
        return  sb.toString();
    }
}
