/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.common;

import java.lang.ref.ReferenceQueue;
import java.util.AbstractMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import static org.maru.common.util.ConditionUtil.isNotNull;
import static org.maru.common.util.ConditionUtil.isNull;

public final class MapFactory {

    private MapFactory(){}

    public static <K, V> ConcurrentMap<K, V> createConcurrentSoftMap() {
        return new ConcurrentSoftMap<K, V>();
    }

    public static <K, V> ConcurrentMap<K, V> createConcurrentSoftMap(int initialCapacity) {
        return new ConcurrentSoftMap<K, V>(initialCapacity);
    }

    public static <K, V> ConcurrentMap<K, V> createConcurrentSoftMap(int initialCapacity, float loadFactor, int concurrencyLevel) {
        return new ConcurrentSoftMap<K, V>(initialCapacity, loadFactor, concurrencyLevel);
    }

    public static <K, V> ConcurrentMap<K, V> createConcurrentSoftMap(Map<? extends K, SoftReferences<? extends V>> map) {
        return new ConcurrentSoftMap<K, V>(map);
    }

    public static ReferenceQueue<Object> getInternalQueue() {
        return InternalReferenceQueue.queue;
    }

    static class InternalReferenceQueue {
        static final ReferenceQueue<Object> queue = new ReferenceQueue<Object>();
    }

    static class ConcurrentSoftMap<K, V> extends AbstractMap<K, V> implements ConcurrentMap<K, V> {

        private final ConcurrentMap<K, SoftReferences<V>> map;

        transient private final ReferenceQueue<Object> queue = InternalReferenceQueue.queue;

        ConcurrentSoftMap() {
            map = new ConcurrentHashMap<K, SoftReferences<V>>();
        }

        ConcurrentSoftMap(int initialCapacity) {
            map = new ConcurrentHashMap<K, SoftReferences<V>>(initialCapacity);
        }

        ConcurrentSoftMap(int initialCapacity, float loadFactor, int concurrencyLevel)  {
            map = new ConcurrentHashMap<K, SoftReferences<V>>(initialCapacity, loadFactor, concurrencyLevel);
        }

        ConcurrentSoftMap(Map<? extends K, SoftReferences<? extends V>> m) {
            map = new ConcurrentHashMap(m);
        }

        @Override
        public V put(K key, V value) {
            SoftReferences<V> ref = new ReapableSoftValue<K, V>(map, key, value, queue);
            map.put(key, ref);
            return ref.get();
        }

        @Override
        public V get(Object key) {
            V value = null;
            SoftReferences<V> softRef = map.get(key);
            if (isNotNull(softRef)) {
                value = softRef.get();

                if (isNull(value)) {
                    return null;
                }
            }
            return value;
        }

        @Override
        public V remove(Object key) {
            SoftReferences<V> softRef = map.remove(key);
            if (isNull(softRef)) {
                return null;
            }
            return softRef.get();
        }

        @Override
        public void clear() {
            map.clear();
        }

        @Override
        public int size() {
            return map.size();
        }

        public V putIfAbsent(K key, V value) {
            SoftReferences<V> ref = map.putIfAbsent(key, new ReapableSoftValue<K, V>(map, key, value, queue));
            if (isNotNull(ref) && isNotNull(ref.get())) {
                return ref.get();
            }
            return value;
        }

        public boolean remove(Object key, Object value) {
            return map.remove(key, new ReapableSoftValue(map, key, value, queue));
        }

        public boolean replace(K key, V oldValue, V newValue) {
            return map.replace(key, new ReapableSoftValue<K, V>(map, key, oldValue, queue), new ReapableSoftValue<K, V>(map, key, newValue, queue));
        }

        public V replace(K key, V value) {
            SoftReferences<V> ref = map.replace(key, new ReapableSoftValue<K, V>(map, key, value, queue));
            return ref.get();
        }

        @Override
        public Set<java.util.Map.Entry<K, V>> entrySet() {
            Set<Entry<K, V>> entries = new LinkedHashSet<Entry<K, V>>();

            for (final Entry<K, SoftReferences<V>> entry : map.entrySet()) {
                final V value = entry.getValue().get();
                if (isNotNull(value)) {
                    entries.add(new Entry<K, V>() {
                        public K getKey() {
                            return entry.getKey();
                        }

                        public V getValue() {
                            return value;
                        }

                        public V setValue(V val) {
                            entry.setValue(new ReapableSoftValue<K, V>(map, entry.getKey(), val, queue));
                            return val;
                        }
                    });
                }
            }
            return entries;
        }


    }

    static class ReapableSoftValue<K, V> extends SoftReferences<V> {
        final ConcurrentMap<K, SoftReferences<V>> referentMap;
        final K key;

        public ReapableSoftValue(ConcurrentMap<K, SoftReferences<V>> referentMap, K key,
                V value, ReferenceQueue<Object> queue) {
            super( value, queue);
            this.key = key;
            this.referentMap = referentMap;
        }

        public void reapReferent() {
            referentMap.remove(key);
        }

    }


}
