/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.m4hv.extensions.constraints;

import static java.lang.annotation.ElementType.ANNOTATION_TYPE;
import static java.lang.annotation.ElementType.CONSTRUCTOR;
import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.ElementType.METHOD;
import static java.lang.annotation.ElementType.PARAMETER;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

import java.lang.annotation.Documented;
import java.lang.annotation.Retention;
import java.lang.annotation.Target;

import javax.validation.Constraint;
import javax.validation.Payload;

import org.maru.m4hv.extensions.validator.NotHaveInValidCharacterValidator;

/**
 * NotHaveInValidCharacter Validator<br>
 * This validates if the string type of value has invalid characters which are
 * defined in resourcefile. You need to define invalid characters in the
 * resource file (By default, it's InValidCharacters(_locale).properties) on the
 * classpath root.
 *
 * There are two ways definition in the resource file like bellow<br>
 *
 * <pre>
 *     1.key=value
 *       In this case, value must be one character for example 'A'.
 *
 *     2.key=char1,char2,char3
 *       In this case values must be on character for example A,B,C
 * </pre>
 *
 * Since {@code NotHaveInValidCharacter} validator treats comma is separater for
 * value, if you want comma to be invalid character, you need to define commna
 * like "key=,".
 *
 * When you create your own invalid character resource file like
 * "MyInvalidCharacter.properties" on the classpath root, always you have to
 * indicate its resource name "MyInvalidCharacter" to
 * {@code NotHaveInValidCharacter}{@link #resource()}.
 *
 * And if you want to use particular invalid character in your resource file,
 * you need to indicate its key names to {@code NotHaveInValidCharacter}
 * {@link #keys()}.
 *
 *
 */
@Target({ METHOD, FIELD, ANNOTATION_TYPE, CONSTRUCTOR, PARAMETER })
@Retention(RUNTIME)
@Constraint(validatedBy = NotHaveInValidCharacterValidator.class)
@Documented
public @interface NotHaveInValidCharacter {

    String message() default "{org.maru.m4hv.extensions.NotHaveInValidCharacter.message}";

    Class<?>[] groups() default {};

    Class<? extends Payload>[] payload() default {};

    String resource() default "InValidCharacters";

    String[] keys() default {};

    @Target({ METHOD, FIELD, ANNOTATION_TYPE, CONSTRUCTOR, PARAMETER })
    @Retention(RUNTIME)
    @Documented
    @interface List {
        NotHaveInValidCharacter[] value();
    }
}
