/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.m4hv.extensions.constraints;

import static java.lang.annotation.ElementType.METHOD;
import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.ElementType.ANNOTATION_TYPE;
import static java.lang.annotation.ElementType.CONSTRUCTOR;
import static java.lang.annotation.ElementType.PARAMETER;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

import java.lang.annotation.Documented;
import java.lang.annotation.Retention;
import java.lang.annotation.Target;

import javax.validation.Constraint;
import javax.validation.Payload;

import org.maru.m4hv.extensions.validator.ActualDateValidator;

/**
 * Actual Date(Year Month Day) validator<br>
 *
 * This validator validates the date accuracy and the range of date.
 * You can define the date range using attributes of from and until on {@code ActualDate} annotation.
 * The date range defaults from 1970/01/01 GMT. The 'until' attribute on {@code ActualDate} annotation is
 * by default the empty string. There are four types of default date pattern that we have already defined.
 * See below discription about default time pattern.<br>
 *
 * <pre>
 *    1. yyyy/MM/dd
 *       The date string format is like 2011/06/06.
 *    2. yyyy-MM-dd
 *       The date string format is like 2011-06-06.
 *    3. yyyy.MM.dd
 *       The date string format is like 2011.06.06.
 *    4.yyyyMMdd
 *       The date string format is like 20110606.
 * </pre>
 *
 * If you would like to add or redefine time patterns yourselves, just override the attribute of
 * patterns on {@code ActualDate} annotation. You can define your own pattern by following the format
 * that simple date format class supports.
 */
@Target({METHOD, FIELD, ANNOTATION_TYPE, CONSTRUCTOR, PARAMETER})
@Retention(RUNTIME)
@Constraint(validatedBy = ActualDateValidator.class)
@Documented
public @interface ActualDate {

    String message() default "{org.maru.m4hv.extensions.ActualDate.message}";

    Class<?>[] groups() default {};

    Class<? extends Payload>[] payload() default {};

    String from() default "1970/01/01";

    String until() default "";

    String[] patterns() default {"yyyy/MM/dd","yyyy-MM-dd", "yyyy.MM.dd", "yyyyMMdd"};

    @Target({METHOD, FIELD, ANNOTATION_TYPE, CONSTRUCTOR, PARAMETER})
    @Retention(RUNTIME)
    @Documented
    @interface List {
        ActualDate[] value();
    }
}
