/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.m4hv.extensions.validator;

import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;

import org.maru.m4hv.extensions.constraints.Second;

import static org.maru.m4hv.extensions.util.Condition.isLessThan;
import static org.maru.m4hv.extensions.util.Condition.checkIncludedCorrectRange;
import static org.maru.m4hv.extensions.util.Condition.isIncludedInRange;

/**
 * Second validator for Number type.<br>
 *
 * Now the support types are byte, short int.
 *
 */
public class SecondValidatorForNumber implements ConstraintValidator<Second, Number> {

    private int startSecond;

    private int endSecond;

    private static final int MIN_SECOND = 0;

    private static final int MAX_SECOND = 59;

    public void initialize(Second second) {
        startSecond = checkIncludedCorrectRange(second.from(), MIN_SECOND, MAX_SECOND);
        endSecond = checkIncludedCorrectRange(second.until(), MIN_SECOND, MAX_SECOND);
        if (!isLessThan(startSecond, endSecond)) {
            throw new IllegalArgumentException("The start second [" + startSecond +
                    "] must be less than the end second [" + startSecond + "]");
        }

    }

    /**
     * Validates if the given object is actual second and included in arbitarary defined ranges
     */
    public boolean isValid(Number object, ConstraintValidatorContext constraintContext) {
        if (object == null) {
            return true;
        }

        Integer value;
        if (object instanceof Integer) {
            value = ((Integer)object).intValue();
        } else if (object instanceof Byte) {
            Byte byteValue = ((Byte) object).byteValue();
            value = byteValue.intValue();
        } else if (object instanceof Short) {
            Short shortValue = ((Short)object).shortValue();
            value = shortValue.intValue();
        } else {
            throw new IllegalArgumentException("Unexpected type : " + object.getClass().getName());
        }

        return isIncludedInRange(value, startSecond, endSecond);
    }

}
