/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.m4hv.extensions.validator;

import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;

import org.maru.m4hv.extensions.constraints.Minute;

import static org.maru.m4hv.extensions.util.Condition.isLessThan;
import static org.maru.m4hv.extensions.util.Condition.checkIncludedCorrectRange;
import static org.maru.m4hv.extensions.util.Condition.isIncludedInRange;

/**
 * Minute validator for Number type.<br>
 *
 * Now the support types are byte, short int.
 *
 *
 */
public class MinuteValidatorForNumber implements ConstraintValidator<Minute, Number> {

    private int startMinute;

    private int endMinute;

    private static final int MIN_MINUTE = 0;

    private static final int MAX_MINUTE = 59;

    public void initialize(Minute minute) {
        startMinute = checkIncludedCorrectRange(minute.from(), MIN_MINUTE, MAX_MINUTE);
        endMinute = checkIncludedCorrectRange(minute.until(), MIN_MINUTE, MAX_MINUTE);
        if (!isLessThan(startMinute, endMinute)) {
            throw new IllegalArgumentException("The start minute [" + startMinute +
                    "] must be less than the end minute [" + endMinute + "]");
        }

    }

    /**
     * Validates if the given object is actual minute and included in arbitarary defined ranges
     */
    public boolean isValid(Number object, ConstraintValidatorContext constraintContext) {
        if (object == null) {
            return true;
        }

        Integer value;
        if (object instanceof Integer) {
            value = ((Integer)object).intValue();
        } else if (object instanceof Byte) {
            Byte byteValue = ((Byte) object).byteValue();
            value = byteValue.intValue();
        } else if (object instanceof Short) {
            Short shortValue = ((Short)object).shortValue();
            value = shortValue.intValue();
        } else {
            throw new IllegalArgumentException("Unexpected type : " + object.getClass().getName());
        }

        return isIncludedInRange(value, startMinute, endMinute);
    }

}
