/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.m4hv.extensions.validator;

import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;

import org.maru.m4hv.extensions.constraints.Hour;

import static org.maru.m4hv.extensions.util.Condition.isLessThan;
import static org.maru.m4hv.extensions.util.Condition.checkIncludedCorrectRange;
import static org.maru.m4hv.extensions.util.Condition.isIncludedInRange;

/**
 * Hour validator for Number type.<br>
 *
 * Now the support types are byte, short int.
 *
 */
public class HourValidatorForNumber implements ConstraintValidator<Hour, Number> {

    private int startHour;

    private int endHour;

    private static final int MIN_HOUR = 0;

    private static final int MAX_HOUR = 23;

    public void initialize(Hour hour) {
        startHour = checkIncludedCorrectRange(hour.from(), MIN_HOUR, MAX_HOUR);
        endHour = checkIncludedCorrectRange(hour.until(), MIN_HOUR, MAX_HOUR);
        if (!isLessThan(startHour, endHour)) {
            throw new IllegalArgumentException("The start hour [" + startHour +
                    "] must be less than the end hour [" + endHour + "]");
        }

    }

    /**
     * Validates if the given object is actual hour and included in arbitarary defined ranges
     */
    public boolean isValid(Number object, ConstraintValidatorContext constraintContext) {
        if (object == null) {
            return true;
        }

        Integer value;
        if (object instanceof Integer) {
            value = ((Integer)object).intValue();
        } else if (object instanceof Byte) {
            Byte byteValue = ((Byte) object).byteValue();
            value = byteValue.intValue();
        } else if (object instanceof Short) {
            Short shortValue = ((Short)object).shortValue();
            value = shortValue.intValue();
        } else {
            throw new IllegalArgumentException("Unexpected type : " + object.getClass().getName());
        }
        return isIncludedInRange(value, startHour, endHour);
    }

}
