/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.m4hv.extensions.validator;

import static org.maru.m4hv.extensions.util.DateUtil.canonicalize;

import java.util.Date;

import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;

import org.maru.m4hv.extensions.constraints.Time;

/**
 * Time validator<br>
 *
 * This validator validates the time and the range of time.
 * You can define the time range using attributes of from and until on {@code Time} annotation.
 * The time range defaults from 00:00:00 until 23:59:59.
 * There are three types of default time pattern that we have already defined.
 * See below discription about default time pattern.<br>
 *
 * <pre>
 *    1. HH:mm:ss
 *       This is a default and standard time format.
 *    2. HH:mm
 *       In this case the second is omitted and defauls 0 second. For example if you define 12:45,
 *       then it means 12:45:00
 *    3. HH
 *       In this case the minute and the second are omitted and defauls both are 0s.
 *       For example if you define 12, then it means 12:00:00.
 * </pre>
 *
 * If you would like to add or redefine time patterns yourselves, just override the attribute of
 * patterns on {@code Time} annotation. You can define your own pattern by following the format
 * that simple date format class supports.
 */
public class TimeValidator implements ConstraintValidator<Time, String> {

    private Date startTime;

    private Date endTime;

    private String[] patterns;

    private static final String EXCEPTION_MESSAGE = "Unexpected time : ";

    public void initialize(Time time) {
        patterns = time.patterns();
        String start = time.from();
        startTime = canonicalize(start, patterns, EXCEPTION_MESSAGE);
        String end = time.until();
        endTime = canonicalize(end, patterns, EXCEPTION_MESSAGE);
    }

    /**
     * Validate time passed as a first parameter whether to be included in start
     * and end time defined {@code Time} annotaton.
     */
    public boolean isValid(String time, ConstraintValidatorContext constraintContext) {
        if (time == null) {
            return true;
        }

        Date inputTime;
        try {
            inputTime = canonicalize(time, patterns, EXCEPTION_MESSAGE);
        } catch (IllegalArgumentException e) {
            // If the string time is not successfully canonicalized, it is not the actual time.
            return false;
        }
        if (inputTime.compareTo(startTime) < 0 || inputTime.compareTo(endTime) > 0) {
            return false;
        }
        return true;

    }

}
