/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.common;

import java.lang.reflect.Type;

import org.maru.common.type.GenericClassType;

/**
 * Analyzes the definition of class. This class is just abstract.
 * So you can extend this as much as you would like to do.
 *
 */
public abstract class AbstractClassAnalyzer<T> {

    public AbstractClassAnalyzer() {
    }

    protected abstract void createStructureInformation(Class<? super T> exactType);

    /**
     * root scan method.
     *
     * @param clazz
     *            scanned class.
     */
    protected abstract void scan(Class<T> clazz);

    /**
     * scan class and interface recursively
     *
     * @param clazz
     *            scanned class.
     */
    @SuppressWarnings("unchecked")
    protected void recursiveClassScan(Class<? super T> clazz) {
        if (clazz.equals(Object.class)) {
            return;
        }

        // get super class type
        Type superClass = clazz.getGenericSuperclass();
        Class<? super T> classType = (Class<? super T>) GenericClassType.getClassFromType(superClass);
        if (classType != null) {
            recursiveClassScan(classType);
        }

        Type[] interfaces = clazz.getGenericInterfaces();
        scanInterfaces(interfaces);
        createStructureInformation(clazz);

    }

    /**
     * scan array of interfaces.
     *
     * @param interfaces
     *            scanned array of interfaces.
     */
    protected void scanInterfaces(Type[] interfaces) {
        if (interfaces != null && interfaces.length > 0) {
            for (Type type : interfaces) {
                scanInterface(type);
            }
        }
    }

    /**
     * scan interface recursively.
     *
     * @param type
     *            scanned interface type.
     */
    @SuppressWarnings("unchecked")
    protected void scanInterface(Type type) {
        Class<? super T> classType = (Class<? super T>) GenericClassType.getClassFromType(type);
        Type[] interfaces = classType.getGenericInterfaces();
        if (interfaces != null && interfaces.length > 0) {
            scanInterfaces(interfaces);
        }
        // create Structure Information.
        this.createStructureInformation(classType);
    }

}
