/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.dog.bind;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Member;
import java.lang.reflect.Method;

import org.maru.common.util.StringUtil;
import org.maru.dog.DefinitionException;
import org.maru.dog.annotation.Bind;

import static org.maru.dog.util.MethodUtil.isGetterMethod;

final class InputMarkedPointBuilder<T> extends AbstractInputMarkedPointBuilder<T> {

    InputMarkedPointBuilder(Class<T> type) {
        super(type, new BindingDefinitionImpl<T>(type));
    }

    @Override
    protected InputMarkedPoint<T> getInputMarkedPoint(Member member, Class<T> target) {
        InputMarkedPoint<T> inputMarkedPoint = null;
        if (member instanceof Field) {
            Field f = (Field) member;
            Bind bind = (Bind) f.getAnnotation(Bind.class);
            String keyName = f.getName();
            if (bind != null) {
                if (StringUtil.isNotEmpty(bind.name())) {
                    keyName = bind.name();
                }
            } else {
                return null;
            }

            // check if converter annotation is defined.
            Annotation converterAnnotation = checkAndGetConverterClassAnnotation(f);
            ConverterDef converterDef = new ConverterDef();
            if (converterAnnotation != null) {
                converterDef = getConverterClassInformation(converterAnnotation, f.getGenericType());
                converterDef.verifyConverter();
            }

            // create marked point
            inputMarkedPoint = createInputMarkedPointInstance(target, keyName, f, converterDef);

            return inputMarkedPoint;
        } else {

            Method m = (Method) member;
            Bind bind = (Bind) m.getAnnotation(Bind.class);

            // The only marked method is used to bind.
            if (bind != null) {
                String methodName = m.getName();

                // check whether or not the method is java beans accessor.
                if (isGetterMethod(methodName, m)) {

                    String keyName = null;

                    if (StringUtil.isNotEmpty(bind.name())) {
                            keyName = bind.name();
                    } else {
                        /*
                         * convert keyName from method name. Remove "get" or "set"
                         * from method name.
                         * If the name of the keyName starts with Upper case,
                         * change it to lower case.
                         */
                        keyName = canonicalizeMethodName(methodName);
                    }

                    if (StringUtil.isEmpty(keyName)) {
                        throw new IllegalArgumentException(
                                "Fail to make bind identifier. " +
                                "It might not be set the bind name, or gettter method name is illegal style");
                    }

                    // check if converter annotation is defined.
                    Annotation converterAnnotation = checkAndGetConverterClassAnnotation(m);
                    ConverterDef converterDef = new ConverterDef();
                    if (converterAnnotation != null) {
                        converterDef = getConverterClassInformation(converterAnnotation, m.getGenericReturnType());
                        converterDef.verifyConverter();
                    }

                    inputMarkedPoint = createInputMarkedPointInstance(target, keyName, m, converterDef);
                } else {
                    throw new DefinitionException(m.getName() + " method is not getter method following JavaBeans.");
                }
            }
        }
        return inputMarkedPoint;
    }

}
