/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.dog.bind;

import static org.maru.common.util.ConditionUtil.isNotNull;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Member;
import java.lang.reflect.Method;
import java.lang.reflect.Type;

import org.maru.common.Key;
import org.maru.common.KeyGen;
import org.maru.common.annotation.Execute;
import org.maru.common.reflection.Clazz;
import org.maru.common.type.GenericClassType;
import org.maru.common.util.StringUtil;
import org.maru.dog.DefinitionException;
import org.maru.dog.converter.Converter;
import org.maru.dog.converter.ConverterIdentifier;
import org.maru.dog.core.AbstractMarkedPointBuilder;
import org.maru.dog.core.Definition;

abstract class AbstractInputMarkedPointBuilder<T> extends AbstractMarkedPointBuilder<T>{

    public AbstractInputMarkedPointBuilder(Class<T> target, Definition<T> definition) {
        super(target, definition);
    }

    @Override
    public void makeMarkedPoint(Member member) {
        InputMarkedPoint<T> inputMarkedPoint = getInputMarkedPoint(member, target);
        if (isNotNull(inputMarkedPoint)) {
            String keyName = inputMarkedPoint.getName();
            Key<?> key = KeyGen.getKey(keyName);
            ((BindingDefinition<T>)definition).putMarkedPoint(key, inputMarkedPoint);
        }
    }

    /**
     * Gets {@link InputMarkedPoint}.
     * @param member member
     * @param target target class
     * @return input marked point.
     */
    protected abstract InputMarkedPoint<T> getInputMarkedPoint(Member member, Class<T> target);

    protected static <T> InputMarkedPoint<T> createInputMarkedPointInstance(Class<T> target,
            String name, Member member,
            ConverterDef converterDef) {
        return new InputMarkedPoint<T>(target, name, member, converterDef);
    }

    protected static ConverterDef getConverterClassInformation(
            Annotation converterAnnotation, Type type) {
        ConverterDef converterDef = new ConverterDef();

        if (converterAnnotation instanceof Converter) {
            Converter converter = (Converter) converterAnnotation;

            if (converter != null) {
                converterDef.converterClass = converter.converterClass();
                String executeIdentifier = converter.execute();
                converterDef.converterMethod = checkAndGetConverterClassMethod(
                        executeIdentifier, converterDef.converterClass,
                        GenericClassType.getClassFromType(type));
            }
        } else {
            ConverterIdentifier converterIdentifier = converterAnnotation
                    .annotationType().getAnnotation(ConverterIdentifier.class);

            converterDef.converterClass = converterIdentifier.value();

            converterDef.converterMethod = checkAndGetConverterMethod(
                    converterAnnotation, converterDef.converterClass, type);
        }
        converterDef.converterAnnotation = converterAnnotation;

        return converterDef;
    }

    protected static Method checkAndGetConverterClassMethod(
            String executeIdentifier, Class<?> converterClass, Class<?> type) {
        Method method = null;

        if (StringUtil.isNotEmpty(executeIdentifier)) {
            // gets the identified converter method.
            int duplicatingCount = 0;

            for (Method m : converterClass.getDeclaredMethods()) {
                Execute execute = m.getAnnotation(Execute.class);
                if (isNotNull(execute) && (execute.value().equals(executeIdentifier) || m.getName().equals(executeIdentifier))) {
                    method = m;
                    duplicatingCount++;
                }
            }

            if (duplicatingCount > 1) {
                throw new DefinitionException("[" + executeIdentifier + "] The name of executing identifier is duplicate.");
            }
        } else {
            method = Clazz.getDeclaredMethod(converterClass, "execute", type);

        }
        if (checkSimpleConverterMethodDefinition(method)) {
            return method;

        } else {
            throw new DefinitionException(
                    "The Definition of simple convert class [" + converterClass.getName() + "] is invalid.");
        }

    }

    protected static boolean checkSimpleConverterMethodDefinition(Method method) {
        if (isNotNull(method)
                && !(method.getGenericReturnType().equals(void.class))
                        && method.getGenericParameterTypes().length == 1) {
            return true;
        }
        return false;
    }

    /**
     * checks number of converter class.<br>
     * Do not let you use multiple converter annotations on one field or method.
     *
     * @param member
     */
    protected static Annotation checkAndGetConverterClassAnnotation(Member member) {
        int number = 0;
        Annotation converterAnnotation = null;
        if (member instanceof Field) {
            Field field = (Field) member;
            for (Annotation annotation : field.getAnnotations()) {
                if (isConverterAnnotation(annotation)) {
                    converterAnnotation = annotation;
                    number++;
                }
            }
        } else if (member instanceof Method) {
            Method method = (Method) member;

            for (Annotation annotation : method.getAnnotations()) {
                if (isConverterAnnotation(annotation)) {
                    converterAnnotation = annotation;
                    number++;
                }
            }
        }

        if (number > 1) {
            throw new IllegalArgumentException(
                    "Two converter annotations are specified on one attribute");
        }
        return converterAnnotation;
    }

    /**
     * Check if the annotation passed by the argument has {@link ConverterIdentifier}, then
     * return true or false.
     * @param annotation
     * @return true if annotation has {@link ConverterIdentifier}, otherwise false.
     */
    protected static boolean isConverterAnnotation(Annotation annotation) {
        if (annotation instanceof Converter) {
            return true;
        } else {
            ConverterIdentifier converterIdentifier = annotation.annotationType().getAnnotation(ConverterIdentifier.class);
            if (isNotNull(converterIdentifier)) {
                return true;
            }
            return false;
        }
    }

    protected static Method checkAndGetConverterMethod(
            Annotation converterAnnotation, Class<?> converterClass, Type type) {
        int count = 0;
        Method converterMethod = null;

        for (Method method : converterClass.getDeclaredMethods()) {
            ConverterIdentifier converterIdentifier = method.getAnnotation(ConverterIdentifier.class);

            if(converterIdentifier != null && converterIdentifier.value().equals(converterAnnotation.annotationType())) {
                converterMethod = method;
                count++;
            }
        }

        if (count > 1) {
            throw new IllegalArgumentException(
                    converterAnnotation.annotationType().getName() + " is not unique on the " + converterClass.getName());
        }

        if (isNotNull(converterMethod)
                && !(converterMethod.getGenericReturnType().equals(void.class))
                        && converterMethod.getGenericParameterTypes().length >= 1) {
            return converterMethod;

        } else {
            throw new DefinitionException(
                    "The Definition of the extended convert class [" + converterClass.getName() + "] is invalid.");
        }

    }


}
