/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.dog;

/**
 * Context interface.
 *
 */
public interface Director {

    /**
     * Bind dependencies to the target object from the input objects. If the
     * first parameter which is bound from input objects is primitive type or
     * wrapper type, this method dose not do anything, just return
     * {@link Director}. And if inputs objects include primitve or wrapper type,
     * those objects are ignored.
     *
     * @param target
     *            target object
     * @param inputs
     *            input objects
     * @return {@link Director}
     */
    <T, K> Director bind(T target, K... inputs);

    /**
     * Bind dependencies to target object from the input objects by matching
     * field name.This method call only use actual field name for binding data.
     *
     * @param target
     *            target object
     * @param inputs
     *            input objects
     * @return {@link Director}
     */
    <T, K> Director bindByFieldName(T target, K... inputs);

    /**
     * Bind dependencies to the target object from the input objects. Then
     * return the target object. If the first parameter which is bound from
     * input objects is primitive type or wrapper type, this method returns just
     * target object. (Dose not execute binding.) And if inputs objects include
     * primitve or wrapper type, those objects are ignored.
     *
     * @param target
     *            target object
     * @param inputs
     *            input objects
     * @return bound target object
     */
    <T, K> T getInstance(T target, K... inputs);

    /**
     * Bind dependencies to the target object from he input objects. The target
     * class passed as a parameter is instantiated before that. If the first
     * parameter which is bound from input objects is primitive type or wrapper
     * type, this method throws {@link IllegalArgumentException}. (Dose not
     * execute binding.) And if inputs objects include primitve or wrapper type,
     * those objects are ignored.
     *
     * @param target
     *            target class
     * @param inputs
     *            input objects
     * @return instantiated and bound target object
     */
    <T, K> T getInstance(Class<T> target, K... inputs);

    /**
     * remove all configuration information.
     *
     * @return {@link Director}
     */
    Director removeAllConfig();

    /**
     * Replace current configuration to new configuration.
     *
     * @param configs
     *            The array of {@link Config}
     * @return {@link Director}
     */
    Director replaceConfig(Config... configs);
}
