package org.kaoriha.marimite;

import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * Checkpoint of process time measurement. You should call
 * {@link #start(String)} first and then call {@link #lap(String)} for each
 * checkpoint. This class uses {@link ThreadLocal} so it ignores call stack. If
 * you want lexical variable, use {@link LocalStopwatch}.
 * 
 * @author NAKAZATO Hajime
 * 
 */
public class Stopwatch {
	private static final int BUFFER_SIZE = 100;
	private static final ConcurrentMap<Section, RingBuffer> spanMap = new ConcurrentHashMap<Section, RingBuffer>();

	private static ThreadLocal<Stopwatch> pmTL = new ThreadLocal<Stopwatch>() {
		protected synchronized Stopwatch initialValue() {
			return new Stopwatch("default");
		}
	};

	private long lastSplitTime = System.currentTimeMillis();
	private String lastPoint;

	Stopwatch(String startPoint) {
		lastPoint = startPoint;
	}

	void lapImpl(String point) {
		try {
			long current = System.currentTimeMillis();

			Section s = Section.getInstance(lastPoint, point);
			RingBuffer rb = spanMap.get(s);

			if (rb == null) {
				spanMap.putIfAbsent(s, new RingBuffer(BUFFER_SIZE, s));
				rb = spanMap.get(s);
			}

			rb.put(current - lastSplitTime);
			lastSplitTime = current;
			lastPoint = point;
		} catch (Throwable t) {
			// show must go on.
		}
	}

	/**
	 * Call this at the head of measurement target process.
	 * 
	 * @param point
	 *            Checkpoint name. Must not contain " - ".
	 */
	public static void start(String point) {
		if (pmTL == null) {
			return;
		}
		pmTL.set(new Stopwatch(point));
	}

	/**
	 * @param point
	 *            Checkpoint name. Must not contain " - ".
	 */
	public static void lap(String point) {
		if (pmTL == null) {
			return;
		}
		Stopwatch s = pmTL.get();
		if (s != null) {
			s.lapImpl(point);
		}
	}

	static Long[] get(Section s) {
		RingBuffer b = spanMap.get(s);
		if (b == null) {
			return null;
		}
		return b.getArray();
	}

	static String[] getSectionSet() {
		Set<String> ss = new HashSet<String>();
		for (Section s : spanMap.keySet()) {
			ss.add(s.toString());
		}
		return ss.toArray(new String[ss.size()]);
	}

	static void removeSection(Section s) {
		spanMap.remove(s);
	}
}
