package org.kaoriha.marimite;

import java.math.BigInteger;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;

class Compressor implements Runnable {
	private static final int N_COMPRESS = 100;
	private PreparedStatement selLap;
	private PreparedStatement countLap;
	private PreparedStatement insLap;
	private PreparedStatement delLap;
	private Connection conn = null;

	public void run() {
		conn = null;
		try {
			conn = MarimiteConfig.getConnection();
			selLap = conn
					.prepareStatement("select id, section_id, elapse_med, elapse_med_var, elapse_tail, elapse_tail_var, elapse_worst, elapse_worst_var, sample_date, n_sample from lap where n_sample = ? and section_id = ?  order by id limit ?");
			selLap.setInt(3, N_COMPRESS);
			countLap = conn
					.prepareStatement("select count(*) from lap where n_sample = ? and section_id = ?");
			insLap = conn
					.prepareStatement("insert into lap (id, section_id, elapse_med, elapse_med_var, elapse_tail, elapse_tail_var, elapse_worst, elapse_worst_var, sample_date, n_sample) values (NEXT VALUE FOR seq_lap_id, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
			delLap = conn
					.prepareStatement("delete from lap where n_sample = ? and section_id = ? and id <= ?");
			synchronized (Compressor.class) {
				runImpl();
			}
			selLap.close();
			countLap.close();
			insLap.close();
			delLap.close();
		} catch (SQLException e) {
			// ignore
		} finally {
			if (conn != null) {
				try {
					conn.close();
				} catch (SQLException e1) {
				}
			}
		}
	}

	private void runImpl() throws SQLException {
		PreparedStatement selSection = conn
				.prepareStatement("select id from section");
		ResultSet rs = null;
		try {
			rs = selSection.executeQuery();
			while (rs.next()) {
				long sectionId = rs.getLong(1);
				compress(1, sectionId);
			}
		} finally {
			if (rs != null) {
				rs.close();
			}
		}
	}

	private void compress(int nSample, long sectionId) throws SQLException {
		countLap.setInt(1, nSample);
		countLap.setLong(2, sectionId);
		int nLap;
		ResultSet rs = null;
		try {
			rs = countLap.executeQuery();
			if (!rs.next()) {
				throw new IllegalStateException("never run here.");
			}
			nLap = rs.getInt(1);
			if (nLap < N_COMPRESS * 2) {
				return;
			}
		} finally {
			if (rs != null) {
				rs.close();
			}
		}

		selLap.setInt(1, nSample);
		selLap.setLong(2, sectionId);
		rs = null;
		try {
			BigInteger sumTimestamp = BigInteger.valueOf(0);
			long lastId = 0;
			List<Double> elapseMedList = new ArrayList<Double>(N_COMPRESS);
			List<Double> elapseTailList = new ArrayList<Double>(N_COMPRESS);
			List<Double> elapseWorstList = new ArrayList<Double>(N_COMPRESS);
			double sumElapseMed = 0;
			double sumElapseTail = 0;
			double sumElapseWorst = 0;
			double sumVarMed = 0.0;
			double sumVarTail = 0.0;
			double sumVarWorst = 0.0;
			rs = selLap.executeQuery();
			while (rs.next()) {
				double med = rs.getDouble("elapse_med");
				sumElapseMed += med;
				elapseMedList.add(med);
				sumVarMed += rs.getDouble("elapse_med_var");

				double tail = rs.getDouble("elapse_tail");
				sumElapseTail += tail;
				elapseTailList.add(tail);
				sumVarTail += rs.getDouble("elapse_tail_var");

				double worst = rs.getDouble("elapse_worst");
				sumElapseWorst += worst;
				elapseWorstList.add(worst);
				sumVarWorst += rs.getDouble("elapse_worst_var");

				Timestamp ts = rs.getTimestamp("sample_date");
				sumTimestamp = sumTimestamp.add(BigInteger.valueOf(ts.getTime()));

				lastId = rs.getLong("id");
			}

			insLap.setLong(1, sectionId);
			Timestamp nts = new Timestamp(sumTimestamp.divide(
					BigInteger.valueOf(N_COMPRESS)).longValue());
			insLap.setTimestamp(8, nts);
			insLap.setInt(9, nSample + 1);

			double aveElapseMed = sumElapseMed / N_COMPRESS;
			insLap.setDouble(2, aveElapseMed);
			double aveElapseTail = sumElapseTail / N_COMPRESS;
			insLap.setDouble(4, aveElapseTail);
			double aveElapseWorst = sumElapseWorst / N_COMPRESS;
			insLap.setDouble(6, aveElapseWorst);
			if (nSample == 1) {
				insLap.setDouble(3, getUnbiasedVariance(elapseMedList,
						aveElapseMed));
				insLap.setDouble(5, getUnbiasedVariance(elapseTailList,
						aveElapseTail));
				insLap.setDouble(7, getUnbiasedVariance(elapseWorstList,
						aveElapseWorst));
			} else {
				insLap.setDouble(3, sumVarMed / N_COMPRESS);
				insLap.setDouble(5, sumVarTail / N_COMPRESS);
				insLap.setDouble(7, sumVarWorst / N_COMPRESS);
			}
			insLap.executeUpdate();

			delLap.setInt(1, nSample);
			delLap.setLong(2, sectionId);
			delLap.setLong(3, lastId);
			int deletedCount = delLap.executeUpdate();
			try {
				if (deletedCount == N_COMPRESS) {
					conn.commit();
				} else {
					conn.rollback();
				}
			} catch (SQLException e) {
				conn.rollback();
				throw e;
			}
		} finally {
			if (rs != null) {
				rs.close();
			}
		}

		if (nLap > N_COMPRESS * 3) {
			compress(nSample, sectionId);
		}
		compress(nSample + 1, sectionId);
	}

	private double getUnbiasedVariance(List<Double> sampleList, double ave) {
		double ds = 0.0;
		for (double sample : sampleList) {
			ds += Math.pow(sample - ave, 2.0);
		}
		return ds / (sampleList.size() - 1);
	}
}
