package org.kaoriha.marimite;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.lang.management.ManagementFactory;
import java.net.MalformedURLException;
import java.rmi.RemoteException;
import java.rmi.registry.LocateRegistry;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.MissingResourceException;
import java.util.Properties;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.management.remote.JMXConnectorServer;
import javax.management.remote.JMXConnectorServerFactory;
import javax.management.remote.JMXServiceURL;

/**
 * Initializer of Marimite.
 * 
 * @author NAKAZATO Hajime
 * 
 */
public class MarimiteConfig {
	private static boolean IS_RUNNING = false;
	private static String DB_LOCATION;
	private static String JMX_DOMAIN;
	private static ObjectName LOG_OBJNAME;
	private static ObjectName LIVE_OBJNAME;
	private static final String PK_FILENAME = "marimite.properties";
	private static final String PK_DB_LOCATION = "DatabaseLocation";
	private static final String PK_JNDI_NAME = "JNDIName";
	private static final String PK_RMI_PORT = "RMIPort";
	private static final String PK_JMX_DOMAIN_NAME = "MBeanDomainName";
	private static final String JMX_KEY_MONITOR = "monitor";
	private static final String JMX_VALUE_LIVE = "live";
	private static final String JMX_VALUE_LOG = "log";
	private static final String DIR_SQL = "sql/";
	private static ExecutorService EXEC_LAP_INSERTER;
	private static ScheduledExecutorService EXEC_COMPRESSOR;
	private static String JMX_SERVICE_URL;
	private static String RMI_PORT;
	private static final String JMX_SERVICE_URL_BASE = "service:jmx:rmi:///jndi/rmi://localhost:";
	private static JMXConnectorServer CONNECTOR_SERVER = null;
	private static final int EXEC_LAP_INSERTER_CAPACITY = 10000;

	private static InputStream getPropStream() {
		InputStream is =  Thread.currentThread().getContextClassLoader()
				.getResourceAsStream(PK_FILENAME);
		if (is == null) {
			throw new MissingResourceException("missing property file: "
					+ PK_FILENAME, MarimiteConfig.class.getCanonicalName(), "");
		}
		return is;
	}

	/**
	 * Exports platform MBean server. It also creates RMI registry. Call
	 * {@link #start()} first. JMXConnectorServer will be stopped at
	 * {@link #stop()}.
	 */
	public static synchronized void exportPlatformMBeanServer() {
		InputStream is = getPropStream();
		try {
			try {
				Properties p = new Properties();
				p.load(is);
				RMI_PORT = p.getProperty(PK_RMI_PORT);
				JMX_SERVICE_URL = JMX_SERVICE_URL_BASE + RMI_PORT + "/"
						+ p.getProperty(PK_JNDI_NAME);
			} finally {
				is.close();
			}
		} catch (IOException e) {
			throw new MissingResourceException("missing property file: "
					+ PK_FILENAME, MarimiteConfig.class.getCanonicalName(), "");
		}

		try {
			LocateRegistry.createRegistry(Integer.parseInt(RMI_PORT));
		} catch (RemoteException e) {
			// ignore
		}

		try {
			JMXServiceURL url = new JMXServiceURL(JMX_SERVICE_URL);
			CONNECTOR_SERVER = JMXConnectorServerFactory.newJMXConnectorServer(
					url, null, ManagementFactory.getPlatformMBeanServer());
			CONNECTOR_SERVER.start();
		} catch (MalformedURLException e) {
			throw new IllegalStateException(
					"JNDIName or RMIPort seems illegal.", e);
		} catch (IOException e) {
			throw new IllegalStateException(
					"JMXConnectorServer cannot start. Specified RMIPort is in use?",
					e);
		}
	}

	/**
	 * Starts logging.
	 */
	public static synchronized void start() {
		if (IS_RUNNING) {
			throw new IllegalStateException("Now running. Stop first.");
		}

		InputStream is = getPropStream();
		try {
			try {
				Properties p = new Properties();
				p.load(is);
				JMX_DOMAIN = p.getProperty(PK_JMX_DOMAIN_NAME);
				DB_LOCATION = p.getProperty(PK_DB_LOCATION);
			} finally {
				is.close();
			}
		} catch (IOException e) {
			throw new MissingResourceException("missing property file: "
					+ PK_FILENAME, MarimiteConfig.class.getCanonicalName(), "");
		}

		Connection conn = null;
		try {
			Class.forName("org.h2.Driver");
			conn = DriverManager.getConnection("jdbc:h2:" + DB_LOCATION,
					"user", "sa");
		} catch (ClassNotFoundException e) {
			throw new IllegalStateException("H2 Database Engine not found.");
		} catch (Exception e) {
			throw new IllegalArgumentException(DB_LOCATION
					+ " is not valid location string.", e);
		}

		try {
			executeSQLFile(conn, "createSchema.sql");
		} catch (SQLException e) {
			// ignore: schema has already created.
		} catch (IOException e) {
			throw new IllegalStateException("sql/createSchema.sql not found.",
					e);
		} finally {
			try {
				conn.close();
			} catch (SQLException e) {
			}
		}

		MBeanServer server = ManagementFactory.getPlatformMBeanServer();
		LiveJMXMBean stopwatchMbean = new LiveJMX();
		LogJMXMBean logMbean = new LogJMX();
		try {
			LIVE_OBJNAME = new ObjectName(JMX_DOMAIN, JMX_KEY_MONITOR,
					JMX_VALUE_LIVE);
			LOG_OBJNAME = new ObjectName(JMX_DOMAIN, JMX_KEY_MONITOR,
					JMX_VALUE_LOG);
			server.registerMBean(stopwatchMbean, LIVE_OBJNAME);
			server.registerMBean(logMbean, LOG_OBJNAME);
		} catch (RuntimeException e) {
			throw e;
		} catch (Exception e) {
			throw new RuntimeException("JMX seems not working", e);
		}

		EXEC_LAP_INSERTER = new ThreadPoolExecutor(1, 1, 0L,
				TimeUnit.MILLISECONDS, new LinkedBlockingQueue<Runnable>(
						EXEC_LAP_INSERTER_CAPACITY),
				new ThreadPoolExecutor.DiscardPolicy());
		EXEC_COMPRESSOR = Executors.newSingleThreadScheduledExecutor();
		EXEC_COMPRESSOR.scheduleWithFixedDelay(new Compressor(), 5, 5,
				TimeUnit.SECONDS);
		IS_RUNNING = true;
	}

	/**
	 * Stops logging.
	 */
	public static synchronized void stop() {
		if (!IS_RUNNING) {
			return;
		}

		EXEC_LAP_INSERTER.shutdown();
		EXEC_COMPRESSOR.shutdown();
		try {
			EXEC_LAP_INSERTER.awaitTermination(1L, TimeUnit.SECONDS);
			EXEC_COMPRESSOR.awaitTermination(10L, TimeUnit.SECONDS);
		} catch (InterruptedException e) {
			// ignore: shutdown must go on.
		}

		MBeanServer server = ManagementFactory.getPlatformMBeanServer();
		try {
			server.unregisterMBean(LOG_OBJNAME);
			server.unregisterMBean(LIVE_OBJNAME);
		} catch (Exception e) {
			// ignore: shutdown must go on.
		}

		if (CONNECTOR_SERVER != null) {
			try {
				CONNECTOR_SERVER.stop();
				CONNECTOR_SERVER = null;
			} catch (Exception e) {
				// ignore: shutdown must go on.
			}
		}

		IS_RUNNING = false;
	}

	static Connection getConnection() throws SQLException {
		if (!IS_RUNNING) {
			throw new IllegalStateException("not initialized");
		}

		Connection conn = DriverManager.getConnection("jdbc:h2:" + DB_LOCATION,
				"user", "sa");
		conn.setAutoCommit(false);
		return conn;
	}

	private static void executeSQLFile(Connection conn, String filename)
			throws SQLException, IOException {
		Statement statement = conn.createStatement();
		Reader rawReader = new InputStreamReader(MarimiteConfig.class
				.getClassLoader().getResourceAsStream(DIR_SQL + filename));
		BufferedReader reader = new BufferedReader(rawReader);
		String line;
		StringBuffer fragment = new StringBuffer();
		while ((line = reader.readLine()) != null) {
			if (line.length() == 0) {
				continue;
			}
			fragment.append(line);
			if (line.indexOf(";") != -1) {
				statement.addBatch(fragment.toString());
				fragment = new StringBuffer();
			}
		}
		reader.close();
		statement.executeBatch();
		statement.close();
	}

	static void QueueLapInserter(LapInserter lapInserter) {
		if (!IS_RUNNING) {
			return;
		}
		EXEC_LAP_INSERTER.submit(lapInserter);
	}
}
