/*******************************************************************************
 * ManjyuCalendar
 * Copyright (C) 2012 Toshiki IGA
 * 
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *******************************************************************************/
/*******************************************************************************
 * Copyright (c) 2012 Toshiki IGA and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *      Toshiki IGA - initial API and implementation
 *******************************************************************************/
/*******************************************************************************
 * Copyright 2012 Toshiki IGA and others.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *******************************************************************************/
package org.manjyu.calendar;

import java.util.Calendar;
import java.util.Date;

/**
 * Calendar Utility for Java.
 * 
 * @author Toshiki Iga
 */
public class ManjyuCalendarUtil {
	/**
	 * Hide Constructor.
	 */
	private ManjyuCalendarUtil() {
	}

	/**
	 * Create new calendar.
	 * 
	 * @return Calendar object of just now.
	 */
	public static Calendar getCalendar() {
		return Calendar.getInstance();
	}

	/**
	 * Create new calendar with specified condition.
	 * 
	 * @param Year value. 1 origin. ie: 2012.
	 * @param Month value. 1 origin. 1 to 12.
	 * @param Day value. 1 origin. 1 to 31.
	 * @return Specified day's calendar object. Hour, minute, second is 0. 
	 */
	public static Calendar getCalendar(final int year, final int month, final int day) {
		final Calendar cal = Calendar.getInstance();
		clearHhmmss(cal);

		setYear(cal, year);
		setMonth(cal, month);
		setDay(cal, day);

		return cal;
	}

	/**
	 * Create new calendar with specified date object.
	 * 
	 * @param date New calendar's date.
	 * @return Specified day's calendar object. Hour, minute, second is 0. 
	 */
	public static Calendar getCalendar(final Date date) {
		final Calendar cal = getCalendar();
		cal.setTime(date);
		return cal;
	}

	/**
	 * Clear hour, minute, second part.
	 *
	 * @param cal Target calendar object to clear hour, minute, second part.
	 */
	public static void clearHhmmss(final Calendar cal) {
		cal.set(Calendar.HOUR_OF_DAY, 0);
		cal.set(Calendar.MINUTE, 0);
		cal.set(Calendar.SECOND, 0);
		cal.set(Calendar.MILLISECOND, 0);
	}

	/**
	 * Set year value to Celendar.
	 * 
	 * @param cal Target calendar object.
	 * @param Year value. 1 origin. ie: 2012.
	 */
	public static void setYear(final Calendar cal, final int year) {
		cal.set(Calendar.YEAR, year);
	}

	/**
	 * Get year value from Celendar.
	 * 
	 * @param cal Target calendar object.
	 * @return Year value. 1 origin. ie: 2012
	 */
	public static int getYear(final Calendar cal) {
		return cal.get(Calendar.YEAR);
	}

	/**
	 * Set month value to Celendar.
	 * 
	 * @param cal Target calendar object.
	 * @param Month value. 1 origin. 1 to 12.
	 */
	public static void setMonth(final Calendar cal, final int month) {
		// java.util.Calendar.MONTH is 0 origin.
		cal.set(Calendar.MONTH, month - 1);
	}

	/**
	 * Get month value from Celendar.
	 * 
	 * @param cal Target calendar object.
	 * @return Month value. 1 origin. 1 to 12.
	 */
	public static int getMonth(final Calendar cal) {
		// java.util.Calendar.MONTH is 0 origin.
		return cal.get(Calendar.MONTH) + 1;
	}

	/**
	 * Set day of month value to Celendar.
	 * 
	 * @param cal Target calendar object.
	 * @param Day value. 1 origin. 1 to 31.
	 */
	public static void setDay(final Calendar cal, final int day) {
		cal.set(Calendar.DAY_OF_MONTH, day);
	}

	/**
	 * Get day of month value from Celendar.
	 * 
	 * @param cal Target calendar object.
	 * @return Day value. 1 origin. 1 to 31.
	 */
	public static int getDay(final Calendar cal) {
		return cal.get(Calendar.DAY_OF_MONTH);
	}

	/**
	 * Validate specified year, month, date combination.
	 * 
	 * Validate is do with Calendar.setLenient(false);
	 * 
	 * @param year Year value. 1 origin. ie: 2012.
	 * @param month Month value. 1 origin. 1 to 12.
	 * @param date Date value. 1 origin. 1 to 31.
	 * @return If valid true, Invalid false.
	 */
	public static boolean validateYmd(final int year, final int month, final int date) {
		try {
			final Calendar cal = getCalendar();

			// set to lenient check.
			cal.setLenient(false);
			cal.set(year, month - 1, date);
			// Call getTime() to do validate.
			cal.getTime();
			return true;
		} catch (IllegalArgumentException e) {
			return false;
		}
	}

	/**
	 * Move to last day of month.
	 * 
	 * @param cal Target calendar object to move.
	 */
	public static void moveToLastDayOfMonth(final Calendar cal) {
		// Move to next month.
		cal.add(Calendar.MONTH, 1);
		// Move to first day of month.
		cal.set(Calendar.DAY_OF_MONTH, 1);
		// Move to previous day.
		cal.add(Calendar.DATE, -1);

		// Now, cal is moved to last day of month.
	}
}
