/*******************************************************************************
 * ManjyuBase64
 * Copyright (C) 2012 Toshiki IGA
 * 
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *******************************************************************************/
/*******************************************************************************
 * Copyright (c) 2012 Toshiki IGA and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *      Toshiki IGA - initial API and implementation
 *******************************************************************************/
/*******************************************************************************
 * Copyright 2012 Toshiki IGA and others.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *******************************************************************************/
package org.manjyu.base64;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.Writer;
import java.util.HashMap;
import java.util.Map;

/**
 * Class for convert RFC 2045 MIME Base64
 * 
 * @author Toshiki Iga
 */
class Base64Rfc2045 {
	/**
	 * Convert table for MIME Base64
	 */
	private static final char[] BASE64_CONVERT_TABLE = { 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L',
			'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f', 'g',
			'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', '0', '1',
			'2', '3', '4', '5', '6', '7', '8', '9', '+', '/', '=' /* END OF MIME Base64 */};

	protected static final int END_OF_MIME_BASE64_INDEX = 64;

	char[] getBase64ConvertTable() {
		return BASE64_CONVERT_TABLE;
	}

	void encode(final InputStream inStream, final Writer writer) throws IOException {
		for (;;) {
			// read 3 bytes on every loop.
			final int read1 = inStream.read();
			if (read1 < 0) {
				// END OF FILE
				return;
			}

			////////////////////////////////////////
			// Fist character
			final char char1 = (char) read1;
			// first 6 bit of first char.
			writer.write(getBase64ConvertTable()[char1 / 0x04]);

			////////////////////////////////////////
			// Second character
			final int read2 = inStream.read();
			if (read2 < 0) {
				// END OF FILE

				// last 2 bit of first char + upper 4 bit of second char.
				writer.write(getBase64ConvertTable()[(char1 & 0x03) * 0x10 + 0]);
				writer.write('=');
				writer.write('=');
				return;
			}
			final char char2 = (char) read2;
			// last 2 bit of first char + upper 4 bit of second char.
			writer.write(getBase64ConvertTable()[(char1 & 0x03) * 0x10 + (char2 / 0x10)]);

			////////////////////////////////////////
			// Third character
			int read3 = inStream.read();
			if (read3 < 0) {
				// END OF FILE

				// lower 4 bit of second char + upper 2 bit of third char.
				writer.write(getBase64ConvertTable()[(char2 & 0x0f) * 0x04 + 0]);
				writer.write('=');
				return;
			}
			final char char3 = (char) read3;

			// lower 4 bit of second char + upper 2 bit of third char.
			writer.write(getBase64ConvertTable()[(char2 & 0x0f) * 0x04 + ((char3 / 0x40) & 0x03)]);

			////////////////////////////////////////
			// Fourth character
			// last 6 bit of third char.
			writer.write(getBase64ConvertTable()[char3 & 0x3f]);
		}
	}

	void decode(final Reader reader, final OutputStream outStream) throws IOException {
		for (;;) {
			final int iRead1 = reader.read();
			if (iRead1 < 0) {
				// END OF BASE 64
				break;
			}

			final int index1 = convertChar2Index((char) iRead1);
			if (index1 < 0) {
				// Ignoreable char
				continue;
			}
			if (index1 == END_OF_MIME_BASE64_INDEX) {
				// END OF BASE 64
				break;
			}

			int index2 = -1;
			for (;;) {
				final int iRead2 = reader.read();
				if (iRead2 < 0) {
					// END OF BASE 64
					throw new IOException("Illegal MIME Base64 string. Char must be 4 char set.");
				}
				index2 = convertChar2Index((char) iRead2);
				if (index2 >= 0) {
					break;
				}
			}
			if (index2 == END_OF_MIME_BASE64_INDEX) {
				// END OF BASE 64
				throw new IOException("Illegal MIME Base64 string. Char must be 4 char set.");
			}

			////////////////////////////////////////
			// Fist byte
			// first 6 bit of first char + last 2 bit of first char.
			outStream.write(index1 * 0x04 + (index2 / 0x10));

			int index3 = -1;
			for (;;) {
				final int iRead3 = reader.read();
				if (iRead3 < 0) {
					// END OF BASE 64
					throw new IOException("Illegal MIME Base64 string(1). Char must be 4 char set.");
				}
				index3 = convertChar2Index((char) iRead3);
				if (index3 >= 0) {
					break;
				}
			}

			if (index3 == END_OF_MIME_BASE64_INDEX) {
				// END OF BASE 64
				break;
			}

			////////////////////////////////////////
			// Second byte
			// upper 4 bit of second char + lower 4 bit of second char.
			outStream.write((index2 & 0x0f) * 0x10 + (index3 / 0x04));

			int index4 = -1;
			for (;;) {
				final int iRead4 = reader.read();
				if (iRead4 < 0) {
					// END OF FILE
					throw new IOException("Illegal MIME Base64 string(2). Char must be 4 char set.");
				}
				index4 = convertChar2Index((char) iRead4);
				if (index4 >= 0) {
					break;
				}
			}

			if (index4 == END_OF_MIME_BASE64_INDEX) {
				// END OF BASE 64
				break;
			}

			////////////////////////////////////////
			// Third byte
			// upper 2 bit of third char + last 6 bit of third char.
			outStream.write((index3 & 0x03) * 0x40 + (index4));
		}
	}

	/**
	 * Cache map for BASE 64 CONVERT TABLE.
	 */
	private static final Map<String, Integer> mapBase64ConvertTabelForRfc2045 = new HashMap<String, Integer>();

	int convertChar2Index(final char cRead) {
		synchronized (mapBase64ConvertTabelForRfc2045) {
			if (mapBase64ConvertTabelForRfc2045.isEmpty()) {
				for (int index = 0; index < getBase64ConvertTable().length; index++) {
					mapBase64ConvertTabelForRfc2045.put(String.valueOf(getBase64ConvertTable()[index]),
							Integer.valueOf(index));
				}
			}

			final Integer lookup = mapBase64ConvertTabelForRfc2045.get(String.valueOf(cRead));
			if (lookup == null) {
				return -1;
			}

			return lookup.intValue();
		}
	}
}
