<?php
/**
 * DBクラス
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2007 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: bbs_groupDb.php 2 2007-11-03 04:59:01Z fishbone $
 * @link       http://www.magic3.org
 */
require_once($gEnvManager->getDbPath() . '/baseDb.php');

class bbs_groupDb extends BaseDb
{
	/**
	 * すべての言語を取得
	 *
	 * @param function	$callback			コールバック関数
	 * @return			true=取得、false=取得せず
	 */
	function getAllLang($callback)
	{
		$queryStr = 'SELECT * FROM _language ORDER BY ln_priority';
		$this->selectLoop($queryStr, array(), $callback, null);
	}
	/**
	 * ユーザグループ一覧を取得
	 *
	 * @param function	$callback			コールバック関数
	 * @param string	$lang				言語
	 * @return 			なし
	 */
	function getAllGroup($callback, $lang)
	{
		$queryStr = 'SELECT * FROM bbs_group LEFT JOIN _login_user ON sg_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=  'WHERE sg_language_id = ? ';
		$queryStr .=    'AND sg_deleted = false ';		// 削除されていない
		$queryStr .=  'ORDER BY sg_sort_order';
		$this->selectLoop($queryStr, array($lang), $callback, null);
	}
	/**
	 * ユーザグループの対応言語を取得
	 *
	 * @param int		$id			カテゴリーID
	 * @return bool					true=取得、false=取得せず
	 */
	function getLangByGroupId($id, &$rows)
	{
		$queryStr = 'SELECT ln_id, ln_name, ln_name_en FROM bbs_group LEFT JOIN _language ON sg_language_id = ln_id ';
		$queryStr .=  'WHERE sg_deleted = false ';	// 削除されていない
		$queryStr .=    'AND sg_id = ? ';
		$queryStr .=  'ORDER BY sg_id, ln_priority';
		$retValue = $this->selectRecords($queryStr, array($id), $rows);
		return $retValue;
	}
	/**
	 * ユーザグループの新規追加
	 *
	 * @param int	  $id	カテゴリーID
	 * @param string  $lang			言語ID
	 * @param string  $name			名前
	 * @param int     $index		表示順
	 * @param int     $userId		更新者ユーザID
	 * @param int     $newSerial	新規シリアル番号
	 * @return bool					true = 成功、false = 失敗
	 */
	function addGroup($id, $lang, $name, $index, $userId, &$newSerial)
	{	
		// トランザクション開始
		$this->startTransaction();
		
		if ($id == 0){		// IDが0のときは、カテゴリーIDを新規取得
			// コンテンツIDを決定する
			$queryStr = 'select max(sg_id) as mid from bbs_group ';
			$ret = $this->selectRecord($queryStr, array(), $row);
			if ($ret){
				$cId = $row['mid'] + 1;
			} else {
				$cId = 1;
			}
		} else {
			$cId = $id;
		}
		
		// 前レコードの削除状態チェック
		$historyIndex = 0;
		$queryStr = 'SELECT * FROM bbs_group ';
		$queryStr .=  'WHERE sg_id = ? ';
		$queryStr .=    'AND sg_language_id = ? ';
		$queryStr .=  'ORDER BY sg_history_index DESC ';
		$queryStr .=    'LIMIT 1';
		$ret = $this->selectRecord($queryStr, array($cId, $lang), $row);
		if ($ret){
			if (!$row['sg_deleted']){		// レコード存在していれば終了
				$this->endTransaction();
				return false;
			}
			$historyIndex = $row['sg_history_index'] + 1;
		}
		
		// データを追加
		$queryStr = 'INSERT INTO bbs_group ';
		$queryStr .=  '(sg_id, sg_language_id, sg_history_index, sg_name, sg_sort_order, sg_create_user_id, sg_create_dt) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, ?, ?, ?, now())';
		$this->execStatement($queryStr, array($cId, $lang, $historyIndex, $name, $index, $userId));
		
		// 新規のシリアル番号取得
		$queryStr = 'select max(sg_serial) as ns from bbs_group ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret) $newSerial = $row['ns'];
			
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * ユーザグループをシリアル番号で取得
	 *
	 * @param int		$serial				シリアル番号
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getGroupBySerial($serial, &$row)
	{
		$queryStr  = 'select * from bbs_group LEFT JOIN _login_user ON sg_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=   'WHERE sg_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		return $ret;
	}	
	/**
	 * ユーザグループをカテゴリーIDで取得
	 *
	 * @param int		$id					カテゴリーID
	 * @param string	$langId				言語ID
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getCategoryByGroupId($id, $langId, &$row)
	{
		$queryStr  = 'SELECT * FROM bbs_group LEFT JOIN _login_user ON sg_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=   'WHERE sg_deleted = false ';	// 削除されていない
		$queryStr .=   'AND sg_id = ? ';
		$queryStr .=   'AND sg_language_id = ? ';
		$ret = $this->selectRecord($queryStr, array($id, $langId), $row);
		return $ret;
	}
	/**
	 * ユーザグループの更新
	 *
	 * @param int     $serial		シリアル番号
	 * @param string  $name			名前
	 * @param int     $index		表示順
	 * @param int     $userId		更新者ユーザID
	 * @param int     $newSerial	新規シリアル番号
	 * @return bool					true = 成功、false = 失敗
	 */
	function updateGroup($serial, $name, $index, $userId, &$newSerial)
	{	
		$now = date("Y/m/d H:i:s");	// 現在日時
				
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のシリアルNoのレコードが削除状態でないかチェック
		$historyIndex = 0;		// 履歴番号
		$queryStr  = 'select * from bbs_group ';
		$queryStr .=   'where sg_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		if ($ret){		// 既に登録レコードがあるとき
			if ($row['sg_deleted']){		// レコードが削除されていれば終了
				$this->endTransaction();
				return false;
			}
			$historyIndex = $row['sg_history_index'] + 1;
		} else {		// 存在しない場合は終了
			$this->endTransaction();
			return false;
		}
		
		// 古いレコードを削除
		$queryStr  = 'UPDATE bbs_group ';
		$queryStr .=   'SET sg_deleted = true, ';	// 削除
		$queryStr .=     'sg_update_user_id = ?, ';
		$queryStr .=     'sg_update_dt = ? ';
		$queryStr .=   'WHERE sg_serial = ?';
		$this->execStatement($queryStr, array($userId, $now, $serial));
		
		// 新規レコード追加
		$queryStr = 'INSERT INTO bbs_group ';
		$queryStr .=  '(sg_id, sg_language_id, sg_history_index, sg_name, sg_sort_order, sg_create_user_id, sg_create_dt) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, ?, ?, ?, ?)';
		$this->execStatement($queryStr, array($row['sg_id'], $row['sg_language_id'], $historyIndex, $name, $index, $userId, $now));

		// 新規のシリアル番号取得
		$queryStr = 'select max(sg_serial) as ns from bbs_group ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret) $newSerial = $row['ns'];
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * ユーザグループの削除
	 *
	 * @param int $serialNo			シリアルNo
	 * @param int $userId			ユーザID(データ更新者)
	 * @return						true=成功、false=失敗
	 */
	function delGroup($serialNo, $userId)
	{
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のシリアルNoのレコードが削除状態でないかチェック
		$queryStr  = 'select * from bbs_group ';
		$queryStr .=   'where sg_deleted = false ';		// 未削除
		$queryStr .=     'and sg_serial = ? ';
		$ret = $this->isRecordExists($queryStr, array($serialNo));
		// 存在しない場合は、既に削除されたとして終了
		if (!$ret){
			$this->endTransaction();
			return false;
		}
		
		// レコードを削除
		$queryStr  = 'UPDATE bbs_group ';
		$queryStr .=   'SET sg_deleted = true, ';	// 削除
		$queryStr .=     'sg_update_user_id = ?, ';
		$queryStr .=     'sg_update_dt = now() ';
		$queryStr .=   'WHERE sg_serial = ?';
		$this->execStatement($queryStr, array($userId, $serialNo));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * ユーザグループIDで削除
	 *
	 * @param int $serial			シリアルNo
	 * @param int $userId			ユーザID(データ更新者)
	 * @return						true=成功、false=失敗
	 */
	function delGroupById($serial, $userId)
	{
		// トランザクション開始
		$this->startTransaction();
		
		// コンテンツIDを取得
		$queryStr  = 'select * from bbs_group ';
		$queryStr .=   'where sg_deleted = false ';		// 未削除
		$queryStr .=     'and sg_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		if ($ret){		// 既に登録レコードがあるとき
			if ($row['sg_deleted']){		// レコードが削除されていれば終了
				$this->endTransaction();
				return false;
			}
		} else {		// 存在しない場合は終了
			$this->endTransaction();
			return false;
		}
		$contId = $row['sg_id'];
		
		// レコードを削除
		$queryStr  = 'UPDATE bbs_group ';
		$queryStr .=   'SET sg_deleted = true, ';	// 削除
		$queryStr .=     'sg_update_user_id = ?, ';
		$queryStr .=     'sg_update_dt = now() ';
		$queryStr .=   'WHERE sg_id = ?';
		$this->execStatement($queryStr, array($userId, $contId));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
}
?>
