<?php
/**
 * コンテナクラス
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2010 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: admin_mainAnalyzegraphWidgetContainer.php 4081 2011-04-21 09:24:53Z fishbone $
 * @link       http://www.magic3.org
 */
require_once($gEnvManager->getCurrentWidgetContainerPath() .	'/admin_mainAnalyzeBaseWidgetContainer.php');
require_once($gEnvManager->getCurrentWidgetDbPath() . '/admin_analyzeDb.php');

class admin_mainAnalyzegraphWidgetContainer extends admin_mainAnalyzeBaseWidgetContainer
{
	private $db;	// DB接続オブジェクト
	private $analyzeDb;
	private $sysDb;	// システム情報取得用
	private $graphTypeArray;	// グラフタイプ
	private $termTypeArray;		// 期間タイプ
	private $graphType;			// グラフ種別
	private $path;				// アクセスパス
	private $termType;				// 期間タイプ
	private $completedDate;			// 処理終了日付
	private $graphDataStr;		// グラフデータ
	private $maxPv = 0;				// 日次のページビュー最大値
	private $yTickValueArray;		// Y軸の最大値
	const CF_LAST_DATE_CALC_PV	= 'last_date_calc_pv';	// ページビュー集計の最終更新日
	const DEFAULT_STR_NOT_CALC = '未集計';		// 未集計時の表示文字列
	const DEFAULT_ACCESS_PATH = 'index';		// デフォルトのアクセスパス(PC用アクセスポイント)
	const ACCESS_PATH_ALL = '_all';				// アクセスパスすべて選択
	const DEFAULT_TERM_TYPE = '30day';		// デフォルトの期間タイプ
	const TERM_TYPE_ALL = '_all';				// 全データ表示選択
	const DEFAULT_GRAPH_TYPE = 'pageview';		// デフォルトのグラフ種別
	
	/**
	 * コンストラクタ
	 */
	function __construct()
	{
		// 親クラスを呼び出す
		parent::__construct();
		
		// DBオブジェクト作成
		$this->db = new admin_analyzeDb();
		$this->sysDb = $this->gInstance->getSytemDbObject();
		
		// グラフタイプ
		$this->graphTypeArray = array(	array(	'name' => 'ページビュー',	'value' => 'pageview'),
										array(	'name' => '訪問数',			'value' => 'visit'),
										array(	'name' => '訪問者数',		'value' => 'visitor'));
										
		// 期間タイプ
		$this->termTypeArray = array(	array(	'name' => '30日',	'value' => '30day'),
										array(	'name' => '3ヶ月',	'value' => '3month'),
										array(	'name' => '1年',	'value' => '1year'),
										array(	'name' => 'すべて',	'value' => self::TERM_TYPE_ALL));
		// Y軸の最大値
		$this->yTickValueArray = array(1000000, 500000, 100000, 50000, 10000, 5000, 1000, 500, 100, 0);
	}
	/**
	 * テンプレートファイルを設定
	 *
	 * _assign()でデータを埋め込むテンプレートファイルのファイル名を返す。
	 * 読み込むディレクトリは、「自ウィジェットディレクトリ/include/template」に固定。
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @param object         $param			任意使用パラメータ。そのまま_assign()に渡る
	 * @return string 						テンプレートファイル名。テンプレートライブラリを使用しない場合は空文字列「''」を返す。
	 */
	function _setTemplate($request, &$param)
	{
		return 'analyzegraph.tmpl.html';
	}
	/**
	 * テンプレートにデータ埋め込む
	 *
	 * _setTemplate()で指定したテンプレートファイルにデータを埋め込む。
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @param object         $param			任意使用パラメータ。_setTemplate()と共有。
	 * @param								なし
	 */
	function _assign($request, &$param)
	{
		$act = $request->trimValueOf('act');
		$this->graphType = $request->trimValueOf('graphtype');			// グラフ種別
		if (empty($this->graphType)) $this->graphType = self::DEFAULT_GRAPH_TYPE;
		$this->path = $request->trimValueOf('path');		// アクセスパス
		if (empty($this->path)) $this->path = self::DEFAULT_ACCESS_PATH;
		$this->termType = $request->trimValueOf('term');				// 期間タイプ
		if (empty($this->termType)) $this->termType = self::DEFAULT_TERM_TYPE;
		
		// 出力期間を取得
		$endDate = $this->db->getStatus(self::CF_LAST_DATE_CALC_PV);
		if (empty($endDate)){
			$this->setMsg(self::MSG_GUIDANCE, '集計済みのデータがありません。集計処理を行ってください。');
			$this->tmpl->setAttribute('show_graph', 'visibility', 'hidden');		// グラフ非表示
			$this->tmpl->setAttribute('draw_graph', 'visibility', 'hidden');		// グラフ非表示
		} else {
			// データの先頭の日付を求める
			$ret = $this->db->getOldAccessLog($row);
			if ($ret){		// 集計対象のデータが存在するとき
				$logStartDate = date("Y/m/d", strtotime($row['al_dt']));
			}
			switch ($this->termType){
				case '30day':
					$startDate = date("Y/m/d", strtotime("$endDate -30 day"));			// 30日前
					break;
				case '3month':
					$startDate = date("Y/m/1", strtotime("$endDate -3 month"));		// 3ヶ月前
					break;
				case '1year':
					$startDate = date("Y/m/1", strtotime("$endDate -1 year"));			// 1年前
					break;
				case self::TERM_TYPE_ALL:		// すべてのデータのとき
					$startDate = NULL;
					break;
			}
			// データの先頭日付を修正
			$this->graphDataStr = '';		// グラフデータ
			if (is_null($startDate)){
				$this->completedDate = '';
			} else {
				// 先頭の日付を修正
				if (strtotime($startDate) < strtotime($logStartDate)) $startDate = $logStartDate;
				
				$this->completedDate = date("Y/m/d", strtotime("$startDate -1 day"));// 処理終了日付を前日に設定
			}
			
			// パスパラメータ作成
			$pathParam = $this->path;
			if ($pathParam == self::ACCESS_PATH_ALL){
				$pathParam = NULL;
			}
			
			// ##### グラフ種別に応じてデータ取得 #####
			switch ($this->graphType){
				case 'pageview':		//ページビュー
					// データ取得
					$this->db->getPageViewByDate($pathParam, $startDate, $endDate, array($this, 'pageViewDataLoop'));
					break;
				case 'visit':			// 訪問数
					$this->db->getDailyCountByDate(0/*訪問数*/, $pathParam, $startDate, $endDate, array($this, 'pageViewDataLoop'));
					break;
				case 'visitor':			// 訪問者数
					$this->db->getDailyCountByDate(1/*訪問者数*/, $pathParam, $startDate, $endDate, array($this, 'pageViewDataLoop'));
					break;
			}
			// 集計終了日までのデータを作成
			$startDate = date("Y/m/d", strtotime("$this->completedDate 1 day"));		// 処理終了日翌日
			$this->createGraphData($startDate, $endDate);
					
			// ##### グラフデータ埋め込み #####
			$this->graphDataStr = trim($this->graphDataStr, ',');		// グラフデータ
			if (empty($this->graphDataStr)){
				$this->setMsg(self::MSG_GUIDANCE, '集計済みのデータがありません');
				$this->tmpl->setAttribute('show_graph', 'visibility', 'hidden');		// グラフ非表示
				$this->tmpl->setAttribute('draw_graph', 'visibility', 'hidden');		// グラフ非表示
			} else {
				// グラフデータ作成
				for ($i = 0; $i < count($this->yTickValueArray) -1; $i++){
					if ($this->maxPv >= $this->yTickValueArray[$i + 1]){
						$yMax = $this->yTickValueArray[$i];// Y座標最大値
						break;
					}
				}
				
				// グラフデータを設定
				$this->tmpl->addVar("draw_graph", "y_max", $yMax);
				$this->tmpl->addVar("draw_graph", "line1_data", $this->graphDataStr);
			}
		}
		// グラフ種別メニュー作成
		$this->createGraphTypeMenu();
		
		// アクセスポイントメニュー作成
		$this->createPathMenu();
		
		// 期間メニュー作成
		$this->createTermMenu();
	}
	/**
	 * グラフ種別メニュー作成
	 *
	 * @return なし
	 */
	function createGraphTypeMenu()
	{
		for ($i = 0; $i < count($this->graphTypeArray); $i++){
			$value = $this->graphTypeArray[$i]['value'];
			$name = $this->graphTypeArray[$i]['name'];
			
			$selected = '';
			if ($value == $this->graphType) $selected = 'selected';
			
			$row = array(
				'value'    => $value,			// グラフ種別ID
				'name'     => $name,			// グラフ種別
				'selected' => $selected			// 選択中かどうか
			);
			$this->tmpl->addVars('graphtype_list', $row);
			$this->tmpl->parseTemplate('graphtype_list', 'a');
		}
	}
	/**
	 * アクセスパスメニュー作成
	 *
	 * @return								なし
	 */
	function createPathMenu()
	{
		$selected = '';
		if ($this->path == self::ACCESS_PATH_ALL){// アクセスパスすべて選択
			$selected = 'selected';
		}
		$row = array(
			'value'    => self::ACCESS_PATH_ALL,			// アクセスパス
			'name'     => 'すべて表示',			// 表示文字列
			'selected' => $selected														// 選択中かどうか
		);
		$this->tmpl->addVars('path_list', $row);
		$this->tmpl->parseTemplate('path_list', 'a');
		
		$this->_db->getPageIdList(array($this, 'pageIdLoop'), 0/*ページID*/);
	}
	/**
	 * ページID、取得したデータをテンプレートに設定する
	 *
	 * @param int $index			行番号(0～)
	 * @param array $fetchedRow		フェッチ取得した行
	 * @param object $param			未使用
	 * @return bool					true=処理続行の場合、false=処理終了の場合
	 */
	function pageIdLoop($index, $fetchedRow, $param)
	{
		$selected = '';
		if ($fetchedRow['pg_path'] == $this->path){
			$selected = 'selected';
		}
		$name = $this->convertToDispString($fetchedRow['pg_path']) . ' - ' . $this->convertToDispString($fetchedRow['pg_name']);			// ページ名
		$row = array(
			'value'    => $this->convertToDispString($fetchedRow['pg_path']),			// アクセスパス
			'name'     => $name,			// ページ名
			'selected' => $selected														// 選択中かどうか
		);
		$this->tmpl->addVars('path_list', $row);
		$this->tmpl->parseTemplate('path_list', 'a');
		return true;
	}
	/**
	 * 期間タイプ選択メニュー作成
	 *
	 * @return なし
	 */
	function createTermMenu()
	{
		for ($i = 0; $i < count($this->termTypeArray); $i++){
			$value = $this->termTypeArray[$i]['value'];
			$name = $this->termTypeArray[$i]['name'];
			
			$selected = '';
			if ($value == $this->termType) $selected = 'selected';
			
			$row = array(
				'value'    => $value,			// ページID
				'name'     => $name,			// ページ名
				'selected' => $selected														// 選択中かどうか
			);
			$this->tmpl->addVars('term_list', $row);
			$this->tmpl->parseTemplate('term_list', 'a');
		}
	}
	/**
	 * ページビューデータを取得
	 *
	 * @param int $index			行番号(0～)
	 * @param array $fetchedRow		フェッチ取得した行
	 * @param object $param			未使用
	 * @return bool					true=処理続行の場合、false=処理終了の場合
	 */
	function pageViewDataLoop($index, $fetchedRow, $param)
	{
		// 日付を取得
		switch ($this->graphType){
			case 'pageview':		//ページビュー
				$pvDate = $fetchedRow['ap_date'];
				break;
			case 'visit':			// 訪問数
			case 'visitor':			// 訪問者数
				$pvDate = $fetchedRow['aa_date'];
				break;
		}
		$endDate = date("Y/m/d", strtotime("$pvDate -1 day"));		// 1日前
		$total = $fetchedRow['total'];
		if ($total > $this->maxPv) $this->maxPv = $total;				// 日次のページビュー最大値
		
		// 0データの期間埋める
		if (!is_null($this->completedDate)){
			$date = date("Y/m/d", strtotime("$this->completedDate 1 day"));

			while (true){
				if (strtotime($date) > strtotime($endDate)) break;

				// グラフ用のデータ作成
				$this->graphDataStr .= '[\'' . $date . '\',0],';
				$date = date("Y/m/d", strtotime("$date 1 day"));
			}
		}
		
		// グラフ用のデータ作成
		$this->graphDataStr .= '[\'' . $pvDate . '\',' . $total . '],';

		// 処理終了日付を更新
		$this->completedDate = $pvDate;
		return true;
	}
	/**
	 * 集計終了日までのグラフデータを作成する
	 *
	 * @param date   $startDate		開始日
	 * @param date   $endDate		終了日
	 * @return						なし
	 */
	function createGraphData($startDate, $endDate)
	{
		$date = $startDate;
		while (true){
			if (strtotime($date) > strtotime($endDate)) break;

			// グラフ用のデータ作成
			$this->graphDataStr .= '[\'' . $date . '\',0],';
			$date = date("Y/m/d", strtotime("$date 1 day"));
		}
	}
}
?>
