<?php
/**
 * 各種テキスト変換マネージャー
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2010 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: textConvManager.php 3767 2010-11-04 01:01:17Z fishbone $
 * @link       http://www.magic3.org
 */
require_once(M3_SYSTEM_INCLUDE_PATH . '/common/htmlEdit.php');
require_once(M3_SYSTEM_INCLUDE_PATH . '/common/core.php');

class TextConvManager extends Core
{
	private $db;						// DBオブジェクト
	private $rootUrl;					// ルートURL
	private $contentType;				// コンテンツタイプ
	private $contentDt;					// コンテンツ作成日時
	const DEFAULT_MOBILE_IMAGE_WIDTH = 240;		// 携帯用画像のデフォルトサイズ(幅)
	const DEFAULT_MOBILE_IMAGE_HEIGHT = 320;	// 携帯用画像のデフォルトサイズ(高さ)
	const DEFAULT_MOBILE_IMAGE_FILE_EXT = 'gif';		// 携帯用画像の形式
	
	/**
	 * コンストラクタ
	 */
	function __construct()
	{
		// 親クラスを呼び出す
		parent::__construct();
		
		// システムDBオブジェクト取得
		$this->db = $this->gInstance->getSytemDbObject();
	}
	/**
	 * キー値テーブルの値を使用してテキストを変換
	 *
	 * @param string $src		変換するデータ
	 * @param string $dest      変換後データ
	 * @param bool $convBr      キーワード変換部分の改行コードをBRタグに変換するかどうか
	 * @return bool				true=成功、false=失敗
	 */
	function convByKeyValue($src, &$dest, $convBr=false)
	{
		// データをコピー
		$dest = $src;

		// キーワードを取得
		$keywords = array();
		$matches = array();
		$pattern = '/(' . preg_quote(M3_TAG_START) . '([A-Z0-9_]+)' . preg_quote(M3_TAG_END) . ')/u';
		preg_match_all($pattern, $src, $matches, PREG_SET_ORDER);
		for ($i = 0; $i < count($matches); $i++){
			$value = $matches[$i][2];
			if (!in_array($value, $keywords)) $keywords[] = $value;
		}
		// キーワードを変換
		for ($i = 0; $i < count($keywords); $i++){
			$key = $keywords[$i];
			$value = $this->db->getKeyValue($key, $tmp);
			if ($convBr){// 改行コード変換の場合
				$value = HtmlEdit::convLineBreakToBr($value);
			}
			$dest = str_replace(M3_TAG_START . $key . M3_TAG_END, $value, $dest);
		}
		return true;
	}
	/**
	 * Magic3タグを削除
	 *
	 * @param string $src		変換するデータ
	 * @return string			変換後データ
	 */
	function deleteM3Tag($src)
	{
		$startTag = str_replace('[', '\[', M3_TAG_START);		// 「[」を正規表現用に「\[」に変換
		$endTag = str_replace(']', '\]', M3_TAG_END);		// 「[」を正規表現用に「\[」に変換
		$search = '/' . $startTag . 'M3_.*' . $endTag . '/';
		return preg_replace($search, '', $src);
	}
	/**
	 * 指定のタグを削除(タグで囲まれた領域は削除しない)
	 *
	 * @param string $src			変換するデータ
	 * @param string,array $tags	削除するタグ
	 * @return string				変換後データ
	 */
	function deleteTag($src, $tags)
	{
		$searchArray = array();
		$replaceArray = array();
		if (is_array($tags)){		// 配列のとき
			$tagArray = $tags;
		} else {		// 文字列のとき
			$tagArray = array($tags);
		}
		
		// 変換処理
		for ($i = 0;$i < count($tagArray); $i++){
			$searchArray[] = "'<" . $tagArray[$i] . "[^>]*?>'si";
			$searchArray[] = "'</" . $tagArray[$i] . "[^>]*?>'si";
			$replaceArray[] = '';
			$replaceArray[] = '';
		}
		$dest = preg_replace($searchArray, $replaceArray, $src);
		return $dest;
	}
	/**
	 * 絵文字画像をMagic3絵文字タグに変換
	 *
	 * @param string $src		変換するデータ
	 * @param string $dest      変換後データ
	 * @return bool				true=成功、false=失敗
	 */
	function convToEmojiTag($src, &$dest)
	{
		global $gEnvManager;
		
		$emojiUrl = $gEnvManager->getEmojiImagesUrl();
		$emojiUrl = str_replace('/', '\/', $emojiUrl);		// 「/」を正規表現用に「\/」に変換
		
		$str = '/<img[^<]*?src="' . $emojiUrl . '\/.*?\.gif\?code=(\d+?)"[^>]*?\/>/';
        $dest = preg_replace_callback($str, array($this, "_replace_to_emoji_tag_callback"), $src);
		return true;
	}
	/**
	 * Magic3絵文字タグ変換コールバック関数
	 *
	 * @param array $matchData		検索マッチデータ
	 * @return string				変換後データ
	 */
    function _replace_to_emoji_tag_callback($matchData)
	{
		global $EMOJI;		// 絵文字マップ情報
		
		// 絵文字情報読み込み
		require_once(M3_SYSTEM_INCLUDE_PATH . '/data/emojiMap.php');// 絵文字マップ情報
						
		// 絵文字コードよりMagic3絵文字タグを作成
		$code = intval($matchData[1]);		// 絵文字コード
		if ($code < 0 || count($EMOJI) <= $code) return '';		// エラーチェック
		
        $code++;		// 絵文字コード(内部コードは1以上)
		$tag = M3_TAG_START . M3_TAG_MACRO_EMOJI_CODE . ':' . $code . M3_TAG_END;
		return $tag;
    }
	/**
	 * Magic3絵文字タグを絵文字画像に変換
	 *
	 * @param string $src		変換するデータ
	 * @param string $dest      変換後データ
	 * @return bool				true=成功、false=失敗
	 */
	function convFromEmojiTag($src, &$dest)
	{
		$startTag = str_replace('[', '\[', M3_TAG_START . M3_TAG_MACRO_EMOJI_CODE);		// 「[」を正規表現用に「\[」に変換
		$endTag = str_replace(']', '\]', M3_TAG_END);		// 「[」を正規表現用に「\[」に変換
		$str = '/' . $startTag . ':(\d+?)' . $endTag . '/';
        $dest = preg_replace_callback($str, array($this, "_replace_from_emoji_tag_callback"), $src);
		return true;
	}
	/**
	 * Magic3絵文字タグ変換コールバック関数
	 *
	 * @param array $matchData		検索マッチデータ
	 * @return string				変換後データ
	 */
    function _replace_from_emoji_tag_callback($matchData)
	{
		global $gEnvManager;
		global $EMOJI;		// 絵文字マップ情報

		// 絵文字情報読み込み
		require_once(M3_SYSTEM_INCLUDE_PATH . '/data/emojiMap.php');// 絵文字マップ情報

		$emojiUrl = $gEnvManager->getEmojiImagesUrl();
				
		// 絵文字コードよりMagic3絵文字タグを作成
		// 配列のインデックスは、「絵文字コード-1」で指定
        $index = intval($matchData[1]) -1;		// 絵文字コード
		if ($index < 0 || count($EMOJI) <= $index) return '';// エラーチェック

		$emojiImageUrl = $emojiUrl . '/' . $EMOJI[$index]['filename'] . '.gif?code=' . $index;
		$tag = '<img src="' . $emojiImageUrl . '" />';
		return $tag;
    }
	/**
	 * Magic3絵文字タグを携帯端末に合わせて変換
	 *
	 * @param string $src		変換するデータ
	 * @param string $dest      変換後データ
	 * @return bool				true=成功、false=失敗
	 */
	function convEmoji($src, &$dest)
	{
		$startTag = str_replace('[', '\[', M3_TAG_START . M3_TAG_MACRO_EMOJI_CODE);		// 「[」を正規表現用に「\[」に変換
		$endTag = str_replace(']', '\]', M3_TAG_END);		// 「[」を正規表現用に「\[」に変換
		$str = '/' . $startTag . ':(\d+?)' . $endTag . '/';
        $dest = preg_replace_callback($str, array($this, "_replace_emoji_callback"), $src);
		return true;
	}
	/**
	 * Magic3絵文字タグ変換コールバック関数
	 *
	 * @param array $matchData		検索マッチデータ
	 * @return string				変換後データ
	 */
    function _replace_emoji_callback($matchData)
	{
		global $gEnvManager;
		global $gInstanceManager;
		global $EMOJI;		// 絵文字マップ情報

		// 絵文字情報読み込み
		require_once(M3_SYSTEM_INCLUDE_PATH . '/data/emojiMap.php');// 絵文字マップ情報
		
		// 絵文字変換ライブラリ読み込み
		require_once($gEnvManager->getLibPath() . '/MobilePictogramConverter-1.2.0/MobilePictogramConverter.php');
		
		$agent = $gInstanceManager->getMobileAgent();
		$emojiUrl = $gEnvManager->getEmojiImagesUrl();
				
		// 絵文字コードよりMagic3絵文字タグを作成
		// 配列のインデックスは、「絵文字コード-1」で指定
        $index = intval($matchData[1]) -1;		// 絵文字コード
		if ($index < 0 || count($EMOJI) <= $index) return '';// エラーチェック
		
		$emojiData = pack('H*', $EMOJI[$index]['i']);		// imodeSJISデータ
		$mpc = MobilePictogramConverter::factory($emojiData, MPC_FROM_FOMA, MPC_FROM_CHARSET_SJIS);

		if ($agent->isDoCoMo()){	// ドコモ端末のとき
			$dest = $mpc->Convert(MPC_TO_FOMA, MPC_TO_OPTION_RAW);
		} else if ($agent->isEZweb()){	// au端末のとき
			$dest = $mpc->Convert(MPC_TO_EZWEB, MPC_TO_OPTION_RAW);
		} else if ($agent->isSoftBank()){	// ソフトバンク端末のとき
			$dest = $mpc->Convert(MPC_TO_SOFTBANK, MPC_TO_OPTION_RAW);
		} else {		// その他の端末のとき(PC用)
			$emojiImageUrl = $emojiUrl . '/' . $EMOJI[$index]['filename'] . '.gif?code=' . $index;
			$dest = '<img src="' . $emojiImageUrl . '" />';
		}
		return $dest;
    }
	/**
	 * PC用コンテンツを携帯用コンテンツに自動変換
	 *
	 * 以下の自動変換処理を行う
	 *  ・画像の自動縮小
	 *  ・tableタグ→divタグ
	 *
	 * @param string $src			変換するデータ
	 * @param string $rootUrl   	リソース用のルートURL
	 * @param string $contentType	コンテンツタイプ
	 * @param timestamp $contentDt	コンテンツの作成日時
	 * @return string				変換後データ
	 */
	function autoConvPcContentToMobile($src, $rootUrl, $contentType, $contentDt)
	{
		$this->rootUrl = $rootUrl;					// ルートURL
		$this->contentType = $contentType;
		$this->contentDt = $contentDt;
		$dest = $src;
		
		// 表示しないタグを削除。データ量を減らす。
		$removeTags = array('script', 'style', 'iframe', 'noframes?');
		foreach ($removeTags as $value){
			$dest = preg_replace('/<'.$value.'\b[^>]*?>.*?<\/'.$value.'\b[^>]*?>/si', '', $dest);
		}
		
		// テーブルタグを除く
		$dest = $this->deleteTag($dest, 'table');
		$dest = $this->deleteTag($dest, 'tbody');
		$dest = $this->deleteTag($dest, 'thead');
		$dest = $this->deleteTag($dest, 'tfoot');
		$dest = $this->deleteTag($dest, 'tr');
		
		// TH,TD,CAPTIONタグをDIVに変換
/*		$dest = preg_replace('/<th\b[^>]*?>/si', '<div>', $dest);
		$dest = preg_replace('/<\/th\b[^>]*?>/si', '</div>', $dest);
		$dest = preg_replace('/<td\b[^>]*?>/si', '<div>', $dest);
		$dest = preg_replace('/<\/td\b[^>]*?>/si', '</div>', $dest);*/
		$dest = preg_replace('/<(\/?)th\b[^>]*?>/si', '<\\1div>', $dest);
		$dest = preg_replace('/<(\/?)td\b[^>]*?>/si', '<\\1div>', $dest);
		$dest = preg_replace('/<(\/?)caption\b[^>]*?>/si', '<\\1div>', $dest);
		
		// UL,OL,LIタグをDIVに変換
		$dest = preg_replace('/<(\/?)(?:ul|ol|li)>/si', '<\\1div>', $dest);
		
		// DL,DT,DDタグをDIVに変換
		$dest = preg_replace('/<(\/?)(?:dl|dt|dd)>/si', '<\\1div>', $dest);
		
		// PタグをDIVに変換
		$dest = preg_replace('/<p\b[^>]*?>/si', '<div>', $dest);
		$dest = preg_replace('/<\/p\b[^>]*?>/si', '</div>', $dest);
		$dest = preg_replace('/<div>\s*<\/div>/si', '', $dest);		// 空のDIVタグ削除
		
		// BLOCKQUOTEタグの処理
		$dest = preg_replace('/<\/?blockquote>/si', '<hr />', $dest);
		
		// PREタグの処理
		$dest = preg_replace_callback('/<pre\b[^>]*?>(.*?)<\/pre\b[^>]*?>/si', create_function('$matches', 'return "<hr />".nl2br(trim($matches[1]))."<hr />";'), $dest);
		
		// 連続したタグをまとめる
		$search = array(
			"'(?:<hr\b[^>]*?>\s*){2,}'si",
			"'(?:<br\b[^>]*?>\s*){3,}'si"
		);
		$replace = array(
			"<hr />",
			"<br /><br />"
		);
		$dest = preg_replace($search, $replace, $dest);
		
		// 画像を携帯用に作成
		$str = '/<img[^<]*?src\s*=\s*[\'"]+(.+?)[\'"]+[^>]*?>/si';
		$dest = preg_replace_callback($str, array($this, "_replace_to_mobile_callback"), $dest);
		return $dest;
	}
	/**
	 * IMGタグ変換コールバック関数
	 *
	 * @param array $matchData		検索マッチデータ
	 * @return string				変換後データ
	 */
    function _replace_to_mobile_callback($matchData)
	{
		global $gEnvManager;
		
		// 画像のパスを取得
		$relativePath = '';
		$imageFile = $matchData[1];
		$imageUrl = $matchData[1];
		if (strStartsWith($imageUrl, '/')){
			$relativePath = $gEnvManager->getRelativePathToSystemRootUrl($gEnvManager->getDocumentRootUrl() . $imageUrl);
		} else {
			if ($gEnvManager->isSystemUrlAccess($imageUrl)){		// システム内のファイルのとき
				$relativePath = $gEnvManager->getRelativePathToSystemRootUrl($imageUrl);
			}
		}
		if (empty($relativePath)){		// システム管理外の画像はそのまま出力
			$destTag = $matchData[0];
		} else {
			$resDir = '/' . M3_DIR_NAME_RESOURCE . '/';
			if (strStartsWith($relativePath, $resDir)){		// リソースディレクトリ以下のリソースのみ変換
				$imageFile = $gEnvManager->getSystemRootPath() . $relativePath;
				$destImageFilename = preg_replace('/.[^.]+$/', '', basename($relativePath)) . '.' . self::DEFAULT_MOBILE_IMAGE_FILE_EXT;	// 作成画像のファイル名
				$destImageRelativePath = dirname(substr($relativePath, strlen($resDir))) . '/' . $destImageFilename;
				$destImageFile = $gEnvManager->getResourcePath() . '/widgets/' . $this->contentType . '/' . M3_DIR_NAME_MOBILE . '/' . $destImageRelativePath;
				$destImageUrl = $gEnvManager->getResourceUrl() . '/widgets/' . $this->contentType . '/' . M3_DIR_NAME_MOBILE . '/' . $destImageRelativePath;

				// ファイルと日時をチェック
				$createImage = true;
				if (file_exists($destImageFile) && strtotime($this->contentDt) < filemtime($destImageFile)){
					$createImage = false;
				}
				
				// 画像の作成
				$isNoErr = true;
				if ($createImage){
					$imageSize = getimagesize($imageFile);
					$imageType = $this->_getImageType($imageSize['mime']);
					
					// ファイル拡張子のチェック
					if (empty($imageType)){
						$errMsg = 'ファイル形式が不明です';
						$isNoErr = false;		// エラー発生
					}
					
					if ($isNoErr){
						// 画像格納用のディレクトリ作成
						$destDir = dirname($destImageFile);
						if (!file_exists($destDir)) mkdir($destDir, M3_SYSTEM_DIR_PERMISSION, true/*再帰的*/);
						
						// 画像のサイズを求める
						$srcWidth = $imageSize[0];
						$srcHeight = $imageSize[1];
						$destWidth = $srcWidth;
						$destHeight = $srcHeight;
						if ($srcWidth > $srcHeight){
							if ($srcWidth > self::DEFAULT_MOBILE_IMAGE_WIDTH){
								$destWidth = self::DEFAULT_MOBILE_IMAGE_WIDTH;
								$destHeight = $srcHeight * (self::DEFAULT_MOBILE_IMAGE_WIDTH / $srcWidth);
							}
						} else {
							if ($srcHeight > self::DEFAULT_MOBILE_IMAGE_HEIGHT){
								$destWidth = $srcWidth * (self::DEFAULT_MOBILE_IMAGE_HEIGHT / $srcHeight);
								$destHeight = self::DEFAULT_MOBILE_IMAGE_HEIGHT;
							}
						}
						
						// 携帯用のため画像はすべてgifで作成
						$imageObj = $this->_createImage($imageType, $imageFile, $destWidth, $destHeight);
						$ret = $this->_outputImage(self::DEFAULT_MOBILE_IMAGE_FILE_EXT, $imageObj, $destImageFile);
						if (!$ret) $isNoErr = false;		// エラー発生
					}
				}
				if ($isNoErr){
					// 幅、高さを設定し直す
					$destTag = $matchData[0];
					$str = '/width\s*=\s*[\'"]+(.+?)[\'"]/si';
					$destTag = preg_replace($str, '', $destTag);
					$str = '/height\s*=\s*[\'"]+(.+?)[\'"]/si';
					$destTag = preg_replace($str, '', $destTag);
					$str = '/style\s*=\s*[\'"]+(.+?)[\'"]/si';		// 「style」属性を削除
					$destTag = preg_replace($str, '', $destTag);
					//$str = '/<img\s*/si';
					//$destTag = preg_replace($str, '<img height="' . $destHeight . '" ', $destTag);

					// 画像のURLを変換
					$destTag = str_replace($matchData[1], $destImageUrl, $destTag);
				} else {// エラー発生の場合は元のファイルのまま
					$destTag = $matchData[0];					
				}
			} else {
				$destTag = $matchData[0];
			}
		}
		return $destTag;
    }
	/**
	 * 携帯用のHTMLをきれいにする
	 *
	 * 以下の自動変換処理を行う
	 *  ・HTML4仕様のタグをXHTML仕様に変換
	 *
	 * @param string $src			変換するデータ
	 * @return string				変換後データ
	 */
	function cleanMobileTag($src)
	{
		$dest = $src;
		$dest = preg_replace('/<blink>(.*?)<\/blink>/si', '<span style="text-decoration:blink;">$1</span>', $dest);
		$dest = preg_replace('/<center>(.*?)<\/center>/si', '<div style="text-align:center;">$1</div>', $dest);
		$str = '/<font\b(.*?)>(.*?)<\/font\b[^>]*?>/si';
		$dest = preg_replace_callback($str, array($this, '_replace_font_callback'), $dest);
		
		// MARQUEEタグの変換
		$str = '/<marquee\b(.*?)>(.*?)<\/marquee\b[^>]*?>/si';
		$dest = preg_replace_callback($str, array($this, '_replace_markquee_callback'), $dest);
		return $dest;
	}
	/**
	 * FONTタグ変換コールバック関数
	 *
	 * @param array $matchData		検索マッチデータ
	 * @return string				変換後データ
	 */
    function _replace_font_callback($matchData)
	{
		// 属性をスタイル属性に直す
		$style = '';
		$str = '/color\s*=\s*[\'"]+(.+?)[\'"]/si';		// 文字色
		if (preg_match($str, $matchData[1], $matches)) $style .= 'color:' . $matches[1] . ';';
		$str = '/size\s*=\s*[\'"]+(.+?)[\'"]/si';		// 文字サイズ
		if (preg_match($str, $matchData[1], $matches)){
			if (strStartsWith($matches[1], '+')){
				$fontSize = 'larger';
			} else if (strStartsWith($matches[1], '-')){
				$fontSize = 'smaller';
			} else {
				switch ($matches[1]){
					case 1:
						$fontSize = 'xx-small';
						break;
					case 2:
						$fontSize = 'x-small';
						break;
					case 3:
						$fontSize = 'small';
						break;
					case 4:
						$fontSize = 'medium';
						break;
					case 5:
						$fontSize = 'large';
						break;
					case 6:
						$fontSize = 'x-large';
						break;
					case 7:
						$fontSize = 'xx-large';
						break;
					default:
						$fontSize = $matches[1];
						break;
				}
			}
			$style .= 'font-size:' . $fontSize . ';';
		}
		$destTag = '<span style="' . $style . '">' . $matchData[2] . '</span>';
		return $destTag;
	}
	/**
	 * MARQUEEタグ変換コールバック関数
	 *
	 * @param array $matchData		検索マッチデータ
	 * @return string				変換後データ
	 */
    function _replace_markquee_callback($matchData)
	{
		// 属性をスタイル属性に直す
		$style = 'display:-wap-marquee;';
		$str = '/bgcolor\s*=\s*[\'"]+(.+?)[\'"]/si';		// 背景色
		if (preg_match($str, $matchData[1], $matches)) $style .= 'background-color:' . $matches[1] . ';';
		$str = '/behavior\s*=\s*[\'"]+(.+?)[\'"]/si';		// マーキースタイル
		if (preg_match($str, $matchData[1], $matches)) $style .= '-wap-marquee-style:' . $matches[1] . ';';
		$str = '/direction\s*=\s*[\'"]+(.+?)[\'"]/si';		// 表示方向
		if (preg_match($str, $matchData[1], $matches)) $style .= '-wap-marquee-dir:' . $matches[1] . ';';
		$str = '/loop\s*=\s*[\'"]+(.+?)[\'"]/si';		// 回数
		if (preg_match($str, $matchData[1], $matches)) $style .= '-wap-marquee-loop:' . $matches[1] . ';';
		
		$destTag = '<span style="' . $style . '">' . $matchData[2] . '</span>';
		return $destTag;
	}
	/**
	 * HTMLをテキストに変換
	 *
	 * @param string $src		変換するデータ
	 * @return string			変換後データ
	 */
	function htmlToText($src)
	{
		$search = array("'<script[^>]*?>.*?</script>'si",	// javascriptを削除
						"'<[\/\!]*?[^<>]*?>'si",  // htmlタグを削除
						"'([\r\n])[\s]+'",  // 空白文字を削除
						"'&(quot|#34);'i",  // htmlエンティティを置換
						"'&(amp|#38);'i",
						"'&(lt|#60);'i",
						"'&(gt|#62);'i",
						"'&(nbsp|#160);'i",
						"'&(iexcl|#161);'i",
						"'&(cent|#162);'i",
						"'&(pound|#163);'i",
						"'&(copy|#169);'i",
						"'&#(\d+);'e");  // phpとして評価

		$replace = array("",
							"",
							"\\1",
							"\"",
							"&",
							"<",
							">",
							" ",
							chr(161),
							chr(162),
							chr(163),
							chr(169),
							"chr(\\1)");

		return preg_replace($search, $replace, $src);
	}
	/**
	 * 画像の種別を取得
	 *
	 * @param string $mime	MIMEコンテンツタイプ
	 * @return string		画像の種別
	 */
	function _getImageType($mime)
	{
		if ($mime != ''){
			if ($mime == 'image/gif')	return 'gif';
			if ($mime == 'image/jpeg')	return 'jpeg';
			if ($mime == 'image/jpg')	return 'jpeg';
			if ($mime == 'image/pjpeg')	return 'jpeg';
			if ($mime == 'image/png')	return 'png';
		}
		return '';
	}		
	/**
	 * リサイズ画像を作成
	 *
	 * @param string $type	MIMEコンテンツタイプ
	 * @param string $path	拡張子
	 * @param int $width	幅
	 * @param int $height	高さ
	 * @return object		画像オブジェクト
	 */
	function _createImage($type, $path, $width, $height)
	{
		// 画像作成
		switch ($type){
			case "jpeg":
				$img = @imagecreatefromjpeg($path);
				break;
			case "gif":
				$img = @imagecreatefromgif($path);
				break;
			case "png":
				$img = @imagecreatefrompng($path);
				break;
			default:
				return false;
		}
		
		// size for thumbnail
		$srcWidth = imagesx($img);
		$srcHeight = imagesy($img);
/*
		if ($width > $height){
			$n_height = $height * ($size / $width);
			$n_width = $size;
		} else {
			$n_width = $width * ($size / $height);
			$n_height = $size;
		}
		
		$x = 0;
		$y = 0;
		if ($n_width < $size) $x = round(($size - $n_width) / 2);
		if ($n_height < $size) $y = round(($size - $n_height) / 2);
		*/
		// imagecreatetruecolor
		$thumb = imagecreatetruecolor($width, $height);
		
//		$bgcolor = imagecolorallocate($thumb, 255, 255, 255);
//		imagefill($thumb, 0, 0, $bgcolor);
		
		// imagecopyresized (imagecopyresampled)
		if (function_exists("imagecopyresampled")){
			if (!imagecopyresampled($thumb, $img, 0, 0, 0, 0, $width, $height, $srcWidth, $srcHeight)){
				if (!imagecopyresized($thumb, $img, 0, 0, 0, 0, $width, $height, $srcWidth, $srcHeight)) return false;
			}
		} else {
			if (!imagecopyresized($thumb, $img, 0, 0, 0, 0, $width, $height, $srcWidth, $srcHeight)) return false;
		}
		return $thumb;
	}
	/**
	 * 画像を出力
	 *
	 * @param string $type	MIMEコンテンツタイプ
	 * @param object $image	画像オブジェクト
	 * @param string $path	ファイル保存の場合のパス
	 * @return bool			true=成功、false=失敗
	 */
	function _outputImage($type, &$image, $path = null)
	{
		$ret = false;
		if (is_null($path)){
			switch ($type){
				case "jpeg":
					$ret = imagejpeg($image);
					break;
				case "gif":
					$ret = imagegif($image);
					break;
				case "png":
					$ret = imagepng($image);
					break;
			}
		} else {
			switch ($type){
				case "jpeg":
					$ret = imagejpeg($image, $path);
					break;
				case "gif":
					$ret = imagegif($image, $path);
					break;
				case "png":
					$ret = imagepng($image, $path);
					break;
			}
		}
		// イメージを破棄
		imagedestroy($image);
		
		return $ret;
	}
}
?>
