<?php
/**
 * コンテナクラス
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2009 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: admin_ec_mainStockWidgetContainer.php 2260 2009-08-27 05:59:32Z fishbone $
 * @link       http://www.magic3.org
 */
require_once($gEnvManager->getCurrentWidgetContainerPath() . '/admin_ec_mainBaseWidgetContainer.php');
require_once($gEnvManager->getCurrentWidgetDbPath() .	'/ec_stockDb.php');
require_once($gEnvManager->getCurrentWidgetDbPath() .	'/productDb.php');

class admin_ec_mainStockWidgetContainer extends admin_ec_mainBaseWidgetContainer
{
	private $db;	// DB接続オブジェクト
	private $productDb;	// DB接続オブジェクト
	private $ecObj;			// 共通ECオブジェクト
	private $search_categoryId;		// 選択中のカテゴリーID
	private $today;	// 今日
	private $yesterday;	// 前日
	private $day7;			// 7日後
	private $day30;			// 30日後
	const STANDARD_PRICE = 'selling';		// 通常価格
	const DEFAULT_TAX_TYPE = 'sales';			// デフォルト税種別
	const PRICE_OBJ_ID = "eclib";		// 価格計算オブジェクトID
	const DEFAULT_LIST_COUNT = 20;			// 最大リスト表示数
	
	/**
	 * コンストラクタ
	 */
	function __construct()
	{
		// 親クラスを呼び出す
		parent::__construct();
		
		// DBオブジェクト作成
		$this->db = new ec_stockDb();
		$this->productDb = new productDb();
		
		// EC用共通オブジェクト取得
		$this->ecObj = $this->gInstance->getObject(self::PRICE_OBJ_ID);
		$this->ecObj->initByDefault();		// デフォルト値で初期化
	}
	/**
	 * テンプレートファイルを設定
	 *
	 * _assign()でデータを埋め込むテンプレートファイルのファイル名を返す。
	 * 読み込むディレクトリは、「自ウィジェットディレクトリ/include/template」に固定。
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @param object         $param			任意使用パラメータ。そのまま_assign()に渡る
	 * @return string 						テンプレートファイル名。テンプレートライブラリを使用しない場合は空文字列「''」を返す。
	 */
	function _setTemplate($request, &$param)
	{
		return 'admin_stock.tmpl.html';
	}
	/**
	 * テンプレートにデータ埋め込む
	 *
	 * _setTemplate()で指定したテンプレートファイルにデータを埋め込む。
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @param object         $param			任意使用パラメータ。_setTemplate()と共有。
	 * @param								なし
	 */
	function _assign($request, &$param)
	{
		// ユーザ情報、表示言語
		$userId		= $this->gEnv->getCurrentUserId();
		$defaultLangId	= $this->gEnv->getDefaultLanguage();
		
		// 日付作成
		$this->today = date("Y/m/d");	// 今日
		$this->yesterday = date("Y/m/d", strtotime("-1 day"));	// 前日
		$this->day7 = date("Y/m/d", strtotime("7 day"));			// 7日後
		$this->day30 = date("Y/m/d", strtotime("30 day"));			// 30日後
		
		// ##### 検索条件 #####
		$pageNo = $request->trimValueOf('page');				// ページ番号
		if (empty($pageNo)) $pageNo = 1;
		// DBの保存設定値を取得
		$maxListCount = self::DEFAULT_LIST_COUNT;				// 表示項目数
		$this->search_categoryId = $request->trimValueOf('category');		// 検索カテゴリー
		$keyword = $request->trimValueOf('keyword');			// 検索キーワード
		
		$act = $request->trimValueOf('act');
		if ($act == 'search'){		// 検索のとき
			$pageNo = 1;		// ページ番号初期化
		}
		
		// カテゴリー一覧を作成
		$this->productDb->getAllCategoryByLoop($defaultLangId, array($this, 'categoryLoop'));
						
		// ###### 検索条件を作成 ######
		// 総数を取得
		$totalCount = $this->productDb->searchProductCount($keyword, $this->search_categoryId, $defaultLangId);
		$pageCount = (int)(($totalCount -1) / $maxListCount) + 1;		// 総ページ数

		// 表示するページ番号の修正
		if ($pageNo < 1) $pageNo = 1;
		if ($pageNo > $pageCount) $pageNo = $pageCount;

		// 商品リストを表示
		$this->productDb->searchProduct($keyword, $this->search_categoryId, $defaultLangId, $maxListCount, ($pageNo -1) * $maxListCount, array($this, 'productListLoop'));

		// ページング用リンク作成
		$pageLink = '';
		if ($pageCount > 1){	// ページが2ページ以上のときリンクを作成
			for ($i = 1; $i <= $pageCount; $i++){
				$linkUrl = $this->gEnv->getDefaultAdminUrl() . '?' . M3_REQUEST_PARAM_OPERATION_COMMAND . '=' . M3_REQUEST_CMD_CONFIG_WIDGET . 
				'&' . M3_REQUEST_PARAM_WIDGET_ID . '=' . $this->gEnv->getCurrentWidgetId() . '&task=productlist&keyword=' . $keyword . '&category=' . $this->search_categoryId . '&page=' . $i;
				if ($i == $pageNo){
					$link = '&nbsp;' . $i;
				} else {
					$link = '&nbsp;<a href="' . $this->getUrl($linkUrl, true) . '" >' . $i . '</a>';
				}
				$pageLink .= $link;
			}
		}
		// 検出順を作成
		$startNo = ($pageNo -1) * $maxListCount +1;
		$endNo = $pageNo * $maxListCount > $totalCount ? $totalCount : $pageNo * $maxListCount;
		$this->tmpl->addVar("show_productlist", "page_link", $pageLink);
		$this->tmpl->addVar("_widget", "page_link", $pageLink);
		$this->tmpl->addVar("_widget", "total_count", $totalCount);
		$this->tmpl->addVar("search_range", "start_no", $startNo);
		$this->tmpl->addVar("search_range", "end_no", $endNo);
		if ($totalCount > 0) $this->tmpl->setAttribute('search_range', 'visibility', 'visible');// 検出範囲を表示
		
		// パラメータ再設定
		$this->tmpl->addVar("_widget", "search_word", $keyword);
		$this->tmpl->addVar("_widget", "day_today",	date("n/j"));		// 今日(フォーマットm/d)
		$this->tmpl->addVar("_widget", "day_7",		date("n/j", strtotime("7 day")));			// 7日後
		$this->tmpl->addVar("_widget", "day_30",	date("n/j", strtotime("30 day")));		// 30日後
		
			
		// パスの設定
		$this->tmpl->addVar("_widget", "admin_url", $this->getUrl($this->gEnv->getDefaultAdminUrl()));
		$this->tmpl->addVar('_widget', 'widget_id', $this->gEnv->getCurrentWidgetId());
	}
	/**
	 * 取得したデータをテンプレートに設定する
	 *
	 * @param int $index			行番号(0～)
	 * @param array $fetchedRow		フェッチ取得した行
	 * @param object $param			未使用
	 * @return bool					true=処理続行の場合、false=処理終了の場合
	 */
	function productListLoop($index, $fetchedRow, $param)
	{
		// 行カラーの設定
		/*$lineColor = '';
		if ($index % 2 != 0){
			$lineColor = 'class="even"';		// 偶数行
		}*/
		
		// データ再取得
		$ret = $this->productDb->getProductBySerial($fetchedRow['pt_serial'], $row, $row2, $row3, $row4, $row5);
		if ($ret){
			$langId = $row['pt_language_id'];
			
			// 価格を取得
			$priceArray = $this->getPrice($row2, self::STANDARD_PRICE);
			$price = $priceArray['pp_price'];	// 価格
			$currency = $priceArray['pp_currency_id'];	// 通貨
			$taxType = $row['pt_tax_type_id'];					// 税種別			

			// 価格作成
			$this->ecObj->setCurrencyType($currency, $langId);		// 通貨設定
			$this->ecObj->setTaxType($taxType, $langId);		// 税種別設定
			$totalPrice = $this->ecObj->getPriceWithTax($price, $dispPrice);	// 税込み価格取得
			if (empty($taxType)) $dispPrice = '(未)';		// 税種別未選択のとき
			
			$visible = '';
			if ($row['pt_visible']){	// 項目の表示
				$visible = 'checked';
			}
		}
		$productId = $row['pt_id'];
		
		// 先頭の日付を求める
		$stockCount = -1;
		$stockCount7 = -1;
		$stockCount30 = -1;

		$ret = $this->db->getStockStartDate($productId, $this->yesterday, $startDate);
		if ($ret){
			// 在庫予定を取得
			$ret = $this->db->getStockRecordsByDate($productId, $startDate, ''/*終了日付*/, 0/*すべての区分*/, $stockRecords);
			if ($ret){
				// 在庫数を求める
				$total = 0;
				$findStart = false;		// 先頭を検出したかどうか
				$foreDate = '';
				for ($i = 0; $i < count($stockRecords); $i++){
					$quantity = intval($stockRecords[$i]['sp_quantity']);// 数量
					$plantype = intval($stockRecords[$i]['sp_plan_type']);
					$planDate = $this->convertToProperDate($stockRecords[$i]['sp_scheduled_dt']);
					if ($plantype == 3) $findStart = true;		// 先頭(棚卸)を検出したかどうか

					// 先頭が検出されていないときは読み飛ばし
					if (!$findStart) continue;
					
					// 在庫数計算
					switch ($plantype){
						case 1:
							$total += $quantity;
							break;
						case 2:
							$total -= $quantity;
							break;
						case 3:
							$total = $quantity;
							break;
					}
					
					// 日付を比較する
					if ($planDate != $foreDate){		// 日付が変更のとき
						if ($planDate < $this->today){
							$stockCount = $total;
							$stockCount7 = $total;
							$stockCount30 = $total;
						} else if ($planDate < $this->day7){
							$stockCount7 = $total;
							$stockCount30 = $total;
						} else if ($planDate < $this->day30){
							$stockCount30 = $total;
						} else {
							break;
						}
					}
					$foreDate = $planDate;		// 前回の日付
				}
			}
		}
		if ($stockCount == -1){
			$stockCountStr = '-';
		} else {
			$stockCountStr = $stockCount;
		}
		if ($stockCount7 == -1){
			$stockCount7Str = '-';
		} else {
			$stockCount7Str = $stockCount7;
		}
		if ($stockCount30 == -1){
			$stockCount30Str = '-';
		} else {
			$stockCount30Str = $stockCount30;
		}
		$row = array(
			//'line_color' => $lineColor,											// 行のカラー
			'no' => $index + 1,													// 行番号
			'serial' => $this->convertToDispString($fetchedRow['pt_serial']),	// シリアル番号
			'id' => $productId,			// ID
			'name' => $this->convertToDispString($row['pt_name']),		// 名前
			'code' => $this->convertToDispString($row['pt_code']),		// 商品コード
			'price' => $dispPrice,													// 価格(税込)
			'stock_count' => $stockCountStr,		// 在庫数
			'stock_count_7' => $stockCount7Str,		// 7日後在庫数
			'stock_count_30' => $stockCount30Str,		// 30日後在庫数
			'update_dt' => $this->convertToDispDateTime($row['pt_create_dt'])	// 更新日時
		);
		$this->tmpl->addVars('itemlist', $row);
		$this->tmpl->parseTemplate('itemlist', 'a');
		
		// 表示中のコンテンツIDを保存
		$this->dispContIdArray[] = $fetchedRow['pt_id'];
		return true;
	}
	/**
	 * 価格取得
	 *
	 * @param array  	$srcRows			価格リスト
	 * @param string	$priceType			価格のタイプ
	 * @return array						取得した価格行
	 */
	function getPrice($srcRows, $priceType)
	{
		for ($i = 0; $i < count($srcRows); $i++){
			if ($srcRows[$i]['pp_price_type_id'] == $priceType){
				return $srcRows[$i];
			}
		}
		return array();
	}
	/**
	 * 取得したカテゴリーをテンプレートに設定する
	 *
	 * @param int $index			行番号(0～)
	 * @param array $fetchedRow		フェッチ取得した行
	 * @param object $param			未使用
	 * @return bool					true=処理続行の場合、false=処理終了の場合
	 */
	function categoryLoop($index, $fetchedRow, $param)
	{
		$selected = '';
		if ($fetchedRow['pc_id'] == $this->search_categoryId){
			$selected = 'selected';
		}
		$name = $this->convertToDispString($fetchedRow['pc_name']);

		$row = array(
			'value'    => $this->convertToDispString($fetchedRow['pc_id']),			// カテゴリーID
			'name'     => $name,			// カテゴリー名
			'selected' => $selected														// 選択中かどうか
		);
		$this->tmpl->addVars('category_list', $row);
		$this->tmpl->parseTemplate('category_list', 'a');
		return true;
	}
}
?>
