<?php
/**
 * 画面制御マネージャー
 *
 * 画面の作成、遷移を処理する
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2009 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: pageManager.php 1753 2009-04-18 06:50:19Z fishbone $
 * @link       http://www.magic3.org
 */
require_once(M3_SYSTEM_INCLUDE_PATH . '/common/core.php');

class PageManager extends Core
{
	private $popupMsg;				// ポップアップダイアログメッセージ
	private $showPositionMode;			// ポジション表示モード
	private $showWidget;			// ウィジェットの単体表示
	private $systemHandleMode;			// システム制御遷移モード(1=サイト非公開時)
	private $tmpData;				// データ作成用
	private $db;					// DBオブジェクト
	private $defaultScriptFiles;	// デフォルトで読み込むスクリプトファイル
	private $defaultCssFiles;		// デフォルトで読み込むCSSファイル
	private $defaultAdminScriptFiles;	// デフォルトで読み込むスクリプトファイル(管理用)
	private $defaultAdminCssFiles;		// デフォルトで読み込むCSSファイル(管理用)
	private $headScriptFiles = array();		// ウィジェットからの追加で読み込むスクリプトファイル
	private $headCssFiles = array();		// ウィジェットからの追加で読み込むCSSファイル
	private $pageDefPosition;		// 現在取得しているページ定義のポジション
	private $pageDefRows;			// ページ定義レコード
	private $replaceHeadDone;			// ヘッダマクロ変換処理が完了したかどうか
	private $useHelp = false;		// 標準ヘルプ機能を使用するかどうか
	private $hasScriptCache = false;	// JavaScriptファイルをブラウザにキャッシュさせるかどうか
	private $lateLaunchWidgetList;	// 遅延実行ウィジェットのリスト
	private $latelaunchWidgetParam;		// 遅延実行ウィジェットのパラメータ
	private $viewPositions = array();	// テンプレート上のポジション
	private $viewPosId = array();		// テンプレート上のポジションのタグID
	private $updateParentWindow;		// 親ウィンドウを再描画するかどうか
	private $updateDefSerial;			// 更新する項目のページ定義シリアル番号
	private $headDescription;				// HTMLヘッダ「description」に出力する文字列
	private $headKeywords;				// HTMLヘッダ「keywords」に出力する文字列
	private $headCss = array();			// HTMLヘッダにCSS出力する文字列
	private $headScript = array();		// HTMLヘッダにJavascript出力する文字列
	private $headString = array();		// HTMLヘッダに出力する任意文字列
	private $lastHeadCss;				// 最後に設定したHTMLヘッダにCSS出力する文字列
	private $lastHeadScript;			// 最後に設定したHTMLヘッダにJavascript出力する文字列
	private $lastHeadString;			// 最後に設定したHTMLヘッダに出力する任意文字列
	private $outputHead;				// HTMLヘッダ出力を行ったかどうか
	private $isAbort;					// ページ作成処理を中断するかどうか
	private $libFiles;					// javascript追加用ライブラリ
	private $pageDefRev = 234;				// 画面定義のリビジョン番号
	private $headSubTitle = array();				// ヘッドタグサブタイトル
	const CONFIG_KEY_HEAD_TITLE_FORMAT = 'head_title_format';		// ヘッダ作成用フォーマット
	const ADMIN_WIDGET_ID = 'admin_main';		// 管理用ウィジェットのウィジェットID
	const CONTENT_TYPE_WIKI = 'wiki';		// ページのコンテンツタイプ(Wiki)
	const WIDGET_ID_TAG_START = '{{WIDGETID:';		// 遅延実行用タグ
	const WIDGET_ID_TAG_END = '}}';		// 遅延実行用タグ
	const WIDGET_ID_SEPARATOR = ',';
//	const HEAD_TAG_DESC			= 	'{{HEAD_DESC}}';			// HTMLヘッダ出力用タグ
//	const HEAD_TAG_OPTION		= 	'{{HEAD_OPTION}}';			// HTMLヘッダの追加出力(CSS,JavaScript等)埋め込み用タグ
	const HEAD_TAGS				= '{{HEAD_TAGS}}';				// HTMLヘッダ出力用タグ
	const WIDGET_ICON_IMG_SIZE = 32;			// ウィジェットアイコンサイズ
	const POSITION_TAG_HEAD = 'm3pos_';			// ポジションの識別用タグIDヘッダ
	const WIDGET_TAG_HEAD = 'm3widget_';			// ウィジェットの識別用タグIDヘッダ
	const WIDGET_TYPE_TAG_HEAD = 'm3widgettype_';			// ウィジェット種別の識別用タグIDヘッダ
	const WIDTET_CLASS_NAME = 'm3widget';			// ウィジェットオブジェクトのタグクラス名
	const WIDTET_CLASS_TYPE_0 = 'm3widget_type0';			// ウィジェットオブジェクトのタグクラス(ページ共通でない)
	const WIDTET_CLASS_TYPE_1 = 'm3widget_type1';			// ウィジェットオブジェクトのタグクラス(ページ共通)
	const POSITION_CLASS_NAME = 'm3position';		// ポジションオブジェクトのタグクラス名
	
	// インナーウィジェット用
	const IWIDTET_CMD_CONTENT = 'content';		// コンテンツ取得
	const IWIDTET_CMD_INIT = 'init';			// 初期化
	const IWIDTET_CMD_UPDATE = 'update';		// 更新
	const IWIDTET_CMD_CALC = 'calc';			// 計算
	const DEFAULT_TEMPLATE_CSS_FILE = '/css/template_css.css';		// デフォルトのテンプレートCSSファイル
	//const DEFAULT_HELP_SCRIPT_FILE = 'boxover2.1/boxover.js';		// デフォルトのヘルプスクリプトファイル
	
	// Javascriptライブラリ
	const LIB_JQUERY_EASING			= 'jquery.easing';
	const LIB_JQUERY_JCAROUSEL		= 'jquery.jcarousel';
	const LIB_JQUERY_THICKBOX		= 'jquery.thickbox';
	const LIB_JQUERY_JSHOTKEYS		= 'jquery.jshotkeys';
	const LIB_JQUERY_CYCLE			= 'jquery.cycle';
	const LIB_JQUERY_CODEPRESS		= 'jquery.codepress';
	const LIB_JQUERY_CLUETIP		= 'jquery.cluetip';
	const LIB_JQUERY_SIMPLETREE		= 'jquery.simpletree';
	const LIB_JQUERY_BGIFRAME		= 'jquery.bgiframe';
	const LIB_JQUERY_HOVERINTENT	= 'jquery.hoverintent';
	const LIB_JQUERY_TABLEDND		= 'jquery.tablednd';
	const LIB_JQUERY_SIMPLEMODAL	= 'jquery.simplemodal';
	
	const LIB_JQUERY_UI			= 'jquery-ui';
	const LIB_JQUERY_UI_PLUS	= 'jquery-ui-plus';
	const LIB_FCKEDITOR			= 'fckeditor';
	const LIB_MD5				= 'md5';
	
	// Javascriptライブラリ用のファイル
	const JQUERY_EASING_FILENAME		= 'jquery/jquery.easing.1.3.js';
	//const JQUERY_JCAROUSEL_FILENAME	= 'jquery/jquery.jcarousel.pack.js';
	const JQUERY_JCAROUSEL_FILENAME		= 'jquery/jquery.jcarousel.0.2.3.pack_fixed.js';
	const JQUERY_JCAROUSEL_CSS			= 'jquery/jquery.jcarousel.css';
	const JQUERY_CONTEXTMENU_FILENAME	= 'jquery/jquery.contextmenu.r2.packed.js';
	const JQUERY_THICKBOX_FILENAME		= 'jquery/thickbox-compressed.js';
	const JQUERY_THICKBOX_CSS			= 'jquery/thickbox.css';
	const JQUERY_JSHOTKEYS_FILENAME		= 'jquery/jquery.hotkeys.js';
	const JQUERY_CYCLE_FILENAME			= 'jquery/jquery.cycle.all.pack.js';
	const JQUERY_CODEPRESS_FILENAME		= 'jquery/jquery.Codepress.js';
	const JQUERY_CLUETIP_FILENAME		= 'jquery/jquery.cluetip.js';
	const JQUERY_CLUETIP_CSS			= 'jquery/jquery.cluetip.css';
	const JQUERY_SIMPLETREE_FILENAME	= 'jquery/jquery.simple.tree.js';
	const JQUERY_SIMPLETREE_CSS			= 'jquery/simple.tree.css';
	const JQUERY_BGIFRAME_FILENAME		= 'jquery/jquery.bgiframe.min.js';
	const JQUERY_HOVERINTENT_FILENAME	= 'jquery/jquery.hoverIntent.min.js';
	const JQUERY_TABLEDND_FILENAME		= 'jquery/jquery.tablednd_0_5.js';
	const JQUERY_TABLEDND_CSS			= 'jquery/jquery.tablednd.css';
	const JQUERY_SIMPLEMODAL_FILENAME	= 'jquery/jquery.simplemodal-1.2.3.pack.js';
	
	// ファイル名
	const JQUERY_FILENAME			= 'jquery-1.2.6.pack.js';						// JQuery
	//const JQUERY_FILENAME			= 'jquery-1.3.2.min.js';						// JQuery1.3.2
	const JQUERY_UI_CORE_FILENAME	= 'jquery-ui-core-1.5.2.packed.js';				// JQuery UI Core (core+Interaction)
	const JQUERY_UI_PLUS_FILENAME	= 'jquery-ui-plus-1.5.2.packed.js';				// JQuery UI Core (widgets+effects)
	const FCKEDITOR_FILENAME		= 'fckeditor2.6/fckeditor.js';		// FCKEditor
	const MD5_FILENAME				= 'md5.js';		// MD5
	
	// 設定ファイル
	const FCKEDITOR_CONFIG			= 'm3/fckconfig.js';		// FCKEditor

	// Magic3用スクリプト
	const M3_ADMIN_SCRIPT_FILENAME			= 'm3admin1.1.6.js';				// 管理機能用スクリプト
	const M3_ADMIN_WIDGET_SCRIPT_FILENAME	= 'm3admin_widget1.3.2.js';	// 管理機能(ウィジェット操作)用スクリプト
	const M3_ADMIN_WIDGET_CSS_FILE			= '/m3/widget.css';			// 管理機能(ウィジェット操作)用CSSファイル
	const M3_STD_SCRIPT_FILENAME			= 'm3std1.3.2.js';
	//const M3_AJAX_SCRIPT_FILENAME			= 'm3ajax1.1.js';
	const M3_OPTION_SCRIPT_FILENAME			= 'm3opt1.0.1.js';		// AJAXを含んだオプションライブラリファイル(jQurry必須)
	
	/**
	 * コンストラクタ
	 */
	function __construct()
	{
		global $gInstanceManager;
		global $gEnvManager;
				
		// 親クラスを呼び出す
		parent::__construct();
		
		// システムDBオブジェクト取得
		$this->db = $gInstanceManager->getSytemDbObject();
		
		// 共通スクリプトファイル
		// 「ルート/scripts」ディレクトリからの相対パスで指定する
		$this->defaultScriptFiles = array(self::M3_STD_SCRIPT_FILENAME);
		$this->defaultCssFiles = array();
		
		// 管理機能用Javascript(新規jQuery版)
		$this->defaultAdminScriptFiles = array(self::JQUERY_FILENAME,			// jQuery
											self::JQUERY_UI_CORE_FILENAME,		// jQuery UI Core
											//self::JQUERY_UI_PLUS_FILENAME,		// jQuery UI Plus
											self::JQUERY_JSHOTKEYS_FILENAME,		// jquery.js-hotkeys
											self::JQUERY_CONTEXTMENU_FILENAME,		// jQuery Contextmenu Lib
											//self::FCKEDITOR_FILENAME,			// FCKEditor
											self::M3_STD_SCRIPT_FILENAME,
											self::M3_ADMIN_SCRIPT_FILENAME,
											self::M3_OPTION_SCRIPT_FILENAME);
											//self::M3_AJAX_SCRIPT_FILENAME);
		$this->defaultAdminCssFiles = array();
		
		// Javascriptライブラリ
		$this->libFiles = array(self::LIB_MD5					=>	array(	'script' => array(self::MD5_FILENAME)),			// MD5
								self::LIB_FCKEDITOR				=>	array(	'script' => array(self::FCKEDITOR_FILENAME)),	// FCKEditor
								self::LIB_JQUERY_UI				=>	array(	'script' => array(self::JQUERY_UI_CORE_FILENAME)),	// jquery ui
								self::LIB_JQUERY_UI_PLUS		=>	array(	'script' => array(self::JQUERY_UI_PLUS_FILENAME)),	// jquery ui plus(追加分)
								self::LIB_JQUERY_EASING			=>	array(	'script' => array(self::JQUERY_EASING_FILENAME)),		// jquery.easing用のファイル
								self::LIB_JQUERY_JCAROUSEL		=>	array(	'script' => array(self::JQUERY_JCAROUSEL_FILENAME),// jquery.jcarousel用のファイル
																			'css'	=> array(self::JQUERY_JCAROUSEL_CSS)),
								self::LIB_JQUERY_THICKBOX		=>	array(	'script' => array(self::JQUERY_THICKBOX_FILENAME),// jquery.thickbox用のファイル
																			'css'	=> array(self::JQUERY_THICKBOX_CSS)),
								self::LIB_JQUERY_JSHOTKEYS		=>	array(	'script' => array(self::JQUERY_JSHOTKEYS_FILENAME)),	// jquery.js-hotkeys用のファイル
								self::LIB_JQUERY_CYCLE			=>	array(	'script' => array(self::JQUERY_CYCLE_FILENAME)),		// jquery.cycle用のファイル
								self::LIB_JQUERY_CODEPRESS		=>	array(	'script' => array(self::JQUERY_CODEPRESS_FILENAME)),	// jquery.codepress用のファイル
								self::LIB_JQUERY_CLUETIP		=>	array(	'script' => array(self::JQUERY_CLUETIP_FILENAME),// jquery.cluetip用のファイル
																			'css'	=> array(self::JQUERY_CLUETIP_CSS)),
								self::LIB_JQUERY_SIMPLETREE		=>	array(	'script' => array(self::JQUERY_SIMPLETREE_FILENAME),// jquery.simpletree用のファイル
																			'css'	=> array(self::JQUERY_SIMPLETREE_CSS)),
								self::LIB_JQUERY_BGIFRAME		=>	array(	'script' => array(self::JQUERY_BGIFRAME_FILENAME)),	// jquery.bgiframe用のファイル
								self::LIB_JQUERY_HOVERINTENT	=>	array(	'script' => array(self::JQUERY_HOVERINTENT_FILENAME)),
								self::LIB_JQUERY_TABLEDND		=>	array(	'script' => array(self::JQUERY_TABLEDND_FILENAME),	// jquery.tablednd用のファイル
																			'css'	=> array(self::JQUERY_TABLEDND_CSS)),
								self::LIB_JQUERY_SIMPLEMODAL		=>	array(	'script' => array(self::JQUERY_SIMPLEMODAL_FILENAME)));			// jquery.simplemodal用のファイル

		// 遅延ウィジェットリスト									
		$this->lateLaunchWidgetList = array();
		$this->latelaunchWidgetParam = array();
		
		// DB接続可能なときは設定値を取得
		if ($gEnvManager->canUseDb()){
			$value = $this->db->getSystemConfig('script_cache_in_browser');	// ブラウザにJavaScriptファイルのキャッシュを保持するかどうか
			if ($value != '') $this->hasScriptCache = $value;
			
			// ヘッダ出力用
			$this->headDescription = $this->db->getSiteDefValue(M3_TB_FIELD_SITE_DESCRIPTION);	// HTMLヘッダ「description」に出力する文字列
			$this->headKeywords = $this->db->getSiteDefValue(M3_TB_FIELD_SITE_KEYWORDS);		// HTMLヘッダ「keywords」に出力する文字列
		}
	}
	/**
	 * タイムアウトを停止
	 *
	 * @return なし
	 */
	function setNoTimeout()
	{
		if (ini_get('safe_mode') == '0') set_time_limit(0);
	}
	/**
	 * ポップアップメッセージを設定
	 *
	 * @param string $msg   メッセージ
	 */
	function setPopupMsg($msg)
	{
		$this->popupMsg = $msg;
	}
	/**
	 * HTMLヘッダ「description」に出力する文字列を設定
	 *
	 * @param string $str   出力文字列
	 */
	function setHeadDescription($str)
	{
		$this->headDescription = $str;
	}
	/**
	 * HTMLヘッダ「keywords」に出力する文字列を設定
	 *
	 * @param string $str   出力文字列
	 */
	function setHeadKeywords($str)
	{
		$this->headKeywords = $str;
	}
	/**
	 * HTMLヘッダ「title」のサブタイトル出力する文字列を設定
	 *
	 * @param string,array $str   出力文字列
	 */
	function setHeadSubTitle($str)
	{
		// 配列で格納
		if (is_array($str)){
			$this->headSubTitle = $str;
		} else if (is_string($str) && !empty($str)){
			$this->headSubTitle = array($str);
		}
	}
	/**
	 * HTMLヘッダに出力するCSSの文字列を設定
	 *
	 * @param string $css	追加するCSS内容
	 * @return 				なし
	 */
	function addHeadCss($css)
	{
		$destCss = trim($css);
		if (!empty($destCss)){
			if (!in_array($css, $this->headCss)) $this->headCss[] = $css;
			
			$this->lastHeadCss = $css;			// 最後に設定したHTMLヘッダにCSS出力する文字列
		}
	}
	/**
	 * HTMLヘッダに出力するJavascriptの文字列を設定
	 *
	 * @param string $script	追加するJavascript内容
	 * @return 					なし
	 */
	function addHeadScript($script)
	{
		$destScript = trim($script);
		if (!empty($destScript)){
			if (!in_array($script, $this->headScript)) $this->headScript[] = $script;
			
			$this->lastHeadScript = $script;			// 最後に設定したHTMLヘッダにJavascript出力する文字列
		}
	}
	/**
	 * HTMLヘッダに出力する任意文字列を設定
	 *
	 * @param string $str	追加する任意文字列
	 * @return 				なし
	 */
	function addHeadString($str)
	{
		$destScript = trim($str);
		if (!empty($destScript)){
			if (!in_array($str, $this->headString)) $this->headString[] = $str;
			
			$this->lastHeadString = $str;			// 最後に設定したHTMLヘッダに出力する任意文字列
		}
	}
	/**
	 * HTMLヘッダに出力するCSSファイルを追加
	 *
	 * @param string,array $cssFile	CSSファイルURLパス
	 * @return 				なし
	 */
	function addHeadCssFile($cssFile)
	{
		if (is_array($cssFile)){	// 配列の場合
			for ($i = 0; $i < count($cssFile); $i++){
				$destCss = trim($cssFile[$i]);
				if (!empty($destCss) && !in_array($destCss, $this->headCssFiles)) $this->headCssFiles[] = $destCss;
			}
		} else {		// 文字列の場合
			$destCss = trim($cssFile);
			if (!empty($destCss) && !in_array($destCss, $this->headCssFiles)) $this->headCssFiles[] = $destCss;
		}
	}
	/**
	 * HTMLヘッダに出力するJavascriptファイルを追加
	 *
	 * @param string,array $scriptFile	JavascriptファイルURLパス
	 * @return 					なし
	 */
	function addHeadScriptFile($scriptFile)
	{
		if (is_array($scriptFile)){	// 配列の場合
			for ($i = 0; $i < count($scriptFile); $i++){
				$destScript = trim($scriptFile[$i]);
				if (!empty($destScript) && !in_array($destScript, $this->headScriptFiles)) $this->headScriptFiles[] = $destScript;
			}
		} else {		// 文字列の場合
			$destScript = trim($scriptFile);
			if (!empty($destScript) && !in_array($destScript, $this->headScriptFiles)) $this->headScriptFiles[] = $destScript;
		}
	}
	/**
	 * 表示ポジションを表示するかどうか
	 *
	 * @param int $mode   0=ポジション表示しない(通常画面)、1=ポジション表示、2=ウィジェット込みポジション表示
	 */
	function showPosition($mode)
	{
		$this->showPositionMode = $mode;
	}
	/**
	 * ウィジェットの単体表示を設定
	 */
	function showWidget()
	{
		$this->showWidget = true;
	}
	/**
	 * システム制御遷移モードを設定
	 *
	 * @param int $mode   0=設定なし、1=サイト非公開時
	 */
	function setSystemHandleMode($mode)
	{
		$this->systemHandleMode = $mode;
	}
	/**
	 * システム制御遷移モード取得
	 */
	function getSystemHandleMode()
	{
		return $this->systemHandleMode;
	}
	/**
	 * ヘルプ機能の使用可否を設定
	 *
	 * @param bool $status	ヘルプ機能を使用するかどうか
	 * @return 				なし
	 */
	function setUseHelp($status)
	{
		$this->useHelp = $status;
	}
	/**
	 * ヘルプ機能の使用可否を取得
	 *
	 * @return bool ヘルプ機能を使用するかどうか
	 */
	function getUseHelp()
	{
		return $this->useHelp;
	}
	/**
	 * JavaScriptのブラウザキャッシュの使用可否を設定
	 *
	 * @param bool $status	ブラウザキャッシュを使用するかどうか
	 * @return 				なし
	 */
	function setHasScriptCache($status)
	{
		$this->hasScriptCache = $status;
	}
	/**
	 * ページ作成処理を中断するかどうかを取得
	 *
	 * @return bool		true=中断、false=継続
	 */
	function isPageAbort()
	{
		return $this->isAbort;
	}
	/**
	 * 親ウィンドウを再描画
	 *
	 * @param int $defSerial	ページ定義シリアル番号
	 * @return 					なし
	 */
	function updateParentWindow($defSerial = 0)
	{
		$this->updateParentWindow = true;
		$this->updateDefSerial = $defSerial;			// 更新する項目のページ定義シリアル番号
	}
	/**
	 * CSSファイルの追加
	 *
	 * @param string $path	追加するファイルのパス(「ルート/scripts」ディレクトリからの相対パスで指定する)
	 * @return 				なし
	 */
	function addCssFile($path)
	{
		$destPath = trim($path, '/');
		if (!in_array($destPath, $this->defaultCssFiles)) $this->defaultCssFiles[] = $destPath;
	}
	/**
	 * CSSファイルの追加
	 *
	 * @param string $path	追加するファイルのパス(「ルート/scripts」ディレクトリからの相対パスで指定する)
	 * @return 				なし
	 */
	function addAdminCssFile($path)
	{
		$destPath = trim($path, '/');
		if (!in_array($destPath, $this->defaultAdminCssFiles)) $this->defaultAdminCssFiles[] = $destPath;
	}
	/**
	 * JavaScriptファイルの追加
	 *
	 * @param string $path	追加するファイルのパス(「ルート/scripts」ディレクトリからの相対パスで指定する)
	 * @return 				なし
	 */
	function addScriptFile($path)
	{
		$destPath = trim($path, '/');
		if (!empty($destPath) && !in_array($destPath, $this->defaultScriptFiles)) $this->defaultScriptFiles[] = $destPath;
	}
	/**
	 * JavaScriptファイルの追加
	 *
	 * @param string $path	追加するファイルのパス(「ルート/scripts」ディレクトリからの相対パスで指定する)
	 * @return 				なし
	 */
	function addAdminScriptFile($path)
	{
		$destPath = trim($path, '/');
		if (!in_array($destPath, $this->defaultAdminScriptFiles)) $this->defaultAdminScriptFiles[] = $destPath;
	}
	/**
	 * ページ作成開始
	 *
	 * HTTPヘッダを設定する。セッションを取得する。
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @return 								なし
	 */
	function startPage($request)
	{
		global $gEnvManager;
		global $gRequestManager;
		global $gInstanceManager;
		global $gAccessManager;
		global $gSystemManager;
		global $gDispManager;
		
		// 実行コマンドを取得
		$cmd = $request->trimValueOf(M3_REQUEST_PARAM_OPERATION_COMMAND);
		$task = $request->trimValueOf(M3_REQUEST_PARAM_OPERATION_TASK);
		
		// セッション変数を取得可能にする
		session_start();

		if (!defined('M3_STATE_IN_INSTALL')){		// インストール時はセッションの再作成は行わない
			// セッションを再生成する(セキュリティ対策)
			if ($gSystemManager->regenerateSessionId()){
				$gAccessManager->setOldSessionId(session_id());		// 古いセッションIDを保存
				session_regenerate_id(true);
			}
		}
		// デバッグモードの表示
		if (M3_SYSTEM_DEBUG) echo 'デバッグモード起動中<br />';
		
		// ##### サブページIDの設定 #####
		$subId = $request->trimValueOf(M3_REQUEST_PARAM_PAGE_SUB_ID);// ページサブIDを取得
		if (empty($subId)){			// サブページIDが設定されていないとき
			// URLパラメータからコンテンツ形式を取得し、ページを選択
			// 「http://www.example.com?ページ名」「wcmd」の場合はwikiコンテンツページを選択
			$wikiCmd = $request->trimValueOf(M3_REQUEST_PARAM_OPERATION_WIKI_COMMAND);
			$pageName = $gRequestManager->getWikiPageFromQuery();
			
			if (!empty($wikiCmd) || !empty($pageName)){			// Wikiコンテンツページを指定のとき
				// ページサブIDを取得
				$subId = $this->db->getSubPageIdWithContent(self::CONTENT_TYPE_WIKI, $gEnvManager->getCurrentPageId());
			}
			
			if (empty($subId)) $subId = $gEnvManager->getDefaultPageSubId();
		}
		$gEnvManager->setCurrentPageSubId($subId);// サブページIDを設定
		
		// セッションからユーザ情報を取得
		$userInfo = $gRequestManager->getSessionValueWithSerialize(M3_SESSION_USER_INFO);
		$gInstanceManager->setUserInfo($userInfo);
		
		// 画面設定取得
		$gDispManager->load();
			
		// 必要なスクリプトがあるときは追加
		if (!defined('M3_STATE_IN_INSTALL')){		// インストール時は実行しない
			if ($gEnvManager->isAdminDirAccess()){		// 管理画面へのアクセスのとき
				if ($gEnvManager->isSystemAdmin()){		// 管理者権限がある場合のみ有効
					// ##### ライブラリを追加 #####
					if ($cmd == M3_REQUEST_CMD_CONFIG_WIDGET){	// ウィジェット詳細設定画面のとき
						$this->addScript('', self::LIB_FCKEDITOR);// FCKEditorスクリプト追加
					}
					$this->addScript('', self::LIB_JQUERY_CLUETIP);// HELP用スクリプト追加
					
					// スクリプトが必要なウィジェットをすべて取得
					$this->db->getWidgetsIdWithLib($gEnvManager->getCurrentPageId(), $gEnvManager->getCurrentPageSubId(), $rows);
					for ($i = 0; $i < count($rows); $i++){
						$this->addScript($task, trim($rows[$i]['wd_add_script_lib']));
					}
					if ($cmd == M3_REQUEST_CMD_CONFIG_WIDGET){	// ウィジェット詳細設定画面のとき
						// ウィジェット情報取得
						$widgetId = $request->trimValueOf(M3_REQUEST_PARAM_WIDGET_ID);
						$ret = $this->db->getWidgetInfo($widgetId, $row);
						if ($ret) $this->addScript($task, trim($row['wd_add_script_lib_a']));		// 管理機能用スクリプト
					}
				}
			}
		}
	}
	/**
	 * スクリプト追加情報から、Javascriptファイル、CSSを追加する
	 *
	 * @param string $task				指定タスク
	 * @param string $scriptInfo		スクリプト追加情報
	 * @return 							なし
	 */
	function addScript($task, $scriptInfo)
	{
		$itemArray = explode(';', strtolower(trim($scriptInfo)));// 小文字に変換したものを解析
		for ($i = 0; $i < count($itemArray); $i++){
			$pos = strpos($itemArray[$i], '=');
			if ($pos === false){// 見つからないときは、タスクが指定されないとき
				$libs = trim($itemArray[$i]);
			} else {		// タスク指定のとき
				list($libTask, $libs) = explode('=', trim($itemArray[$i]));
				$libTask = trim($libTask);
				$libs = trim($libs);
				if (empty($libTask) || $libTask != $task) $libs = '';			// タスクが異なるときは追加しない
			}
			if (!empty($libs)){		// // スクリプト、CSSの追加を行うとき
				$libsArray = explode(',', $libs);
				for ($j = 0; $j < count($libsArray); $j++){
					$lib = trim($libsArray[$j]);
					if (isset($this->libFiles[$lib])){		// ライブラリが存在するとき
						if (isset($this->libFiles[$lib]['script'])){
							$scriptFiles = $this->libFiles[$lib]['script'];
							for ($l = 0; $l < count($scriptFiles); $l++){
								$this->addAdminScriptFile($scriptFiles[$l]);		// スクリプト追加
							}
						}
						if (isset($this->libFiles[$lib]['css'])){
							$cssFiles = $this->libFiles[$lib]['css'];
							for ($l = 0; $l < count($cssFiles); $l++){
								$this->addAdminCssFile($cssFiles[$l]);		// CSS追加
							}
						}
					}
				}
			}
		}
	}
	/**
	 * ページ作成終了
	 *
	 * ・最終HTML出力
	 * ・セッション情報の保存
	 * ・ウィジェットで生成されたHTTPヘッダを設定する
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @param bool $getOutput				出力を取得するかどうか
	 * @return string        				最終出力HTML
	 */
	function endPage($request, $getOutput = false)
	{
		global $gAjaxManager;
		global $gRequestManager;
		global $gInstanceManager;
		global $gDispManager;
		
		// ページ作成処理を中断するかどうか
		if ($this->isAbort) return '';
		
		$contents = '';
		
		// 実行コマンドを取得
		$cmd = $request->trimValueOf(M3_REQUEST_PARAM_OPERATION_COMMAND);
		
		// 最終HTML(ページ全体で使用するHTML)の出力
		if ($cmd != M3_REQUEST_CMD_DO_WIDGET){		// ウィジェット単体オペレーションのときは出力しない
			if ($getOutput){
				$contents = $this->getLastContents($request);
			} else {
				echo $this->getLastContents($request);
			}
		}
		
		// セッションへユーザ情報を保存
		$userInfo = $gInstanceManager->getUserInfo();
		$gRequestManager->setSessionValueWithSerialize(M3_SESSION_USER_INFO, $userInfo);
		
		// 画面設定保存
		$gDispManager->save();
			
		// キャッシュリミッタは、各リクエスト毎に(アウトプットバッファー が無効な場合は、
		// session_start()がコールされる 前に) session_cache_limiter()をコールする必要がある。
	 	// キャッシュを残す設定
//	 	session_cache_limiter('private');
//	 	session_cache_expire(5);

		// ########## HTTPヘッダ出力処理 ########
		if (headers_sent($filename, $linenum)){		// HTTPヘッダが既に送信されているとき
			echo "$filename の $linenum 行目でヘッダがすでに送信されています。";
		} else {
			// キャッシュを無効にする場合
			header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');// 過去の日付
			header('Cache-Control: no-store, no-cache, must-revalidate');// HTTP/1.1
			header('Cache-Control: post-check=0, pre-check=0', false);
			header('Pragma: no-cache');
		
			// 更新日時
			header('Last-Modified: ' . gmdate('D, d M Y H:i:s') . ' GMT');
		
			// Ajax用JSON型データをHTTPヘッダに格納
			$data = $gAjaxManager->header($request);
		}
		return $contents;
	}
	/**
	 * ページ作成処理中断
	 *
	 * @return 							なし
	 */
	function abortPage($callback = null)
	{
		global $gInstanceManager;
		global $gRequestManager;
		
		// exit()等でabortPage()が最後の処理になってしまう可能性があるのでなるべく必要な処理を行う
		ob_end_clean();
		
		// セッションへユーザ情報を保存
		$userInfo = $gInstanceManager->getUserInfo();
		$gRequestManager->setSessionValueWithSerialize(M3_SESSION_USER_INFO, $userInfo);
		
		$this->isAbort = true;					// ページ作成処理を中断するかどうか
	}
	/**
	 * 強制終了を実行
	 *
	 * @return 		なし
	 */
	function exitSystem()
	{
		global $gEnvManager;
		global $gAccessManager;
		
		// DBが使用可能であれば、アクセスログのユーザを登録
		if ($gEnvManager->canUseDb()) $gAccessManager->accessLogUser();
		
		exit;
	}
	/**
	 * Joomla!v1.5タグを読み込んでウィジェット実行
	 *
	 * @param string         $srcBuf		バッファデータ
	 * @return string						変換後文字列
	 */
	function launchWidgetByJoomlaTag($srcBuf)
	{
		$replace = array();
		$matches = array();
		$destBuf = $srcBuf;
		
		if (preg_match_all('#<jdoc:include\ type="([^"]+)" (.*)\/>#iU', $srcBuf, $matches)){
			$count = count($matches[1]);
			for($i = 0; $i < $count; $i++)
			{
				$contents = '';
				$type  = $matches[1][$i];
				if (strcasecmp($type, 'head') == 0){		// ヘッダ埋め込みタグの場合
					ob_clean();
					$this->getHeader();
					$contents = ob_get_contents();
				} else if (strcasecmp($type, 'modules') == 0){		// ポジションタグの場合
					$name = '';			// ポジション名
					$style = '';		// 表示スタイル
					$params = explode(' ', $matches[2][$i]);
					for ($j = 0; $j < count($params); $j++){
						list($key, $value) = explode('=', $params[$j]);
						if (strcasecmp($key, 'name') == 0){
							$name = trim($value, "\"'");
						} else if (strcasecmp($key, 'style') == 0){
							$style = trim($value, "\"'");
						}
					}
					if (!empty($name)){		// ポジション名が取得できたとき
						if (empty($style)) $style = 'none';
						
						// ウィジェットの出力を取得
						$contents = $this->getContents($name, $style, false);
					}
				} else if (strcasecmp($type, 'component') == 0){	// メインポジションタグの場合
					$contents = $this->getContents('main', 'none', false);
				} else if (strcasecmp($type, 'message') == 0){	// メッセージタグの場合
				}
				$replace[$i] = $contents;
			}
			ob_clean();
			$destBuf = str_replace($matches[0], $replace, $srcBuf);
		}
		return $destBuf;
	}
	/**
	 * 遅延ウィジェット実行
	 *
	 * 遅延実行インデックスのついているウィジェットをインデックス順に実行し、出力バッファデータ内のタグの位置に出力を埋め込む
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @param string         $srcBuf		バッファデータ
	 * @return string						変換後文字列
	 */
	function lateLaunchWidget($request, $srcBuf)
	{
		global $gEnvManager;
		global $gDesignManager;
		global $gErrorManager;
		
		// 遅延実行ウィジェットをインデックス順にソート
		asort($this->lateLaunchWidgetList, SORT_NUMERIC);
		
		// タグを置換
		$destBuf = $srcBuf;
		foreach ($this->lateLaunchWidgetList as $widgetId => $value){
			// 実行パラメータ取得
			$count = count($this->latelaunchWidgetParam);
			for ($i = 0; $i < $count; $i++){
				list($wId, $index, $confId, $preId, $serial, $style) = $this->latelaunchWidgetParam[$i];
				if ($wId == $widgetId){
					// パラメータ初期化
					$this->lastHeadCss = '';			// 最後に設定したHTMLヘッダにCSS出力する文字列
					$this->lastHeadScript = '';			// 最後に設定したHTMLヘッダにJavascript出力する文字列
					$this->lastHeadString = '';			// 最後に設定したHTMLヘッダに出力する任意文字列
					
					// バッファ作成
					ob_start();

					// ウィジェット実行ファイル取得
					$widgetIndexFile = $gEnvManager->getWidgetsPath() . '/' . $widgetId . '/index.php';

					// その他パラメータ取得
					$configId = $confId;		// ウィジェット定義ID
					if ($configId == 0) $configId = '';
					$prefix = $preId;		// サフィックス文字列
			
					// ウィジェット親のCSS定義があるときは、タグを追加
					if (!empty($style)) echo '<div style="' . $style . '">' . M3_NL;
					
					// ウィジェットの前出力
					echo $gDesignManager->getAdditionalWidgetOutput(true);
				
					// 作業中のウィジェットIDを設定
					$gEnvManager->setCurrentWidgetId($widgetId);

					// ウィジェット定義IDを設定
					$gEnvManager->setCurrentWidgetConfigId($configId);
					
					// ページ定義のシリアル番号を設定
					$gEnvManager->setCurrentPageDefSerial($serial);
		
					// パラメータを設定
					$gEnvManager->setCurrentWidgetPrefix($prefix);		// プレフィックス文字列
		
					// 実行ログを残す
					$this->db->writeWidgetLog($widgetId, 0/*ページ実行*/);
					
					// ウィジェットを実行
					// ウィジェットの呼び出しは、複数回存在する可能性があるのでrequire_once()で呼び出さない
					$msg = 'widget-start(' . $widgetId . ')';
					$gErrorManager->writeDebug(__METHOD__, $msg);		// 時間計測用
					require($widgetIndexFile);
					$msg = 'widget-end(' . $widgetId . ')';
					$gErrorManager->writeDebug(__METHOD__, $msg);		// 時間計測用

					// 作業中のウィジェットIDを解除
					$gEnvManager->setCurrentWidgetId('');
		
					// ウィジェット定義IDを解除
					$gEnvManager->setCurrentWidgetConfigId('');
					
					// ページ定義のシリアル番号を解除
					$gEnvManager->setCurrentPageDefSerial(0);
		
					// パラメータを解除
					$gEnvManager->setCurrentWidgetPrefix('');				// プレフィックス文字列

					// ウィジェットの後出力
					echo $gDesignManager->getAdditionalWidgetOutput(false);
				
					// ウィジェット親のCSS定義があるときは、タグを追加
					if (!empty($style)) echo '</div>' . M3_NL;
					
					// 現在のバッファ内容を取得し、バッファを破棄
					$srcContents = ob_get_contents();
					ob_end_clean();
					
					// ウィジェットの出力を取得
					$tag = self::WIDGET_ID_TAG_START . $widgetId . self::WIDGET_ID_SEPARATOR . $index . self::WIDGET_ID_TAG_END;
					$destBuf = str_replace($tag, $srcContents, $destBuf);
				}
			}
		}
		$destBuf = $this->replaceHead($destBuf);
		return $destBuf;
	}
	/**
	 * ヘッダ部マクロ変換処理
	 *
	 * @param string         $srcBuf		変換元
	 * @return string						変換後文字列
	 */
	function replaceHead($srcBuf)
	{
		$destBuf = $srcBuf;
		
		// ##### ヘッダ部分の置換 #####
		if ($this->outputHead){				// HTMLヘッダ出力を行っているとき
			// タグ変換用文字列の取得
			$replaceStr = $this->getHeaderOutput();
			
			// HTMLヘッダのデータ埋め込み
			$destBuf = str_replace(self::HEAD_TAGS, $replaceStr, $destBuf);
		}
		$this->replaceHeadDone = true;			// ヘッダマクロ変換処理が完了したかどうか
		return $destBuf;
	}
	/**
	 * ウィジェット検索モードの場合のページサブIDの設定
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 */
	function redirectToUpdatedPageSubId($request)
	{
		global $gEnvManager;
		
		// 現在設定されているページIDを取得
		$pageId		= $gEnvManager->getCurrentPageId();
		$pageSubId	= $gEnvManager->getCurrentPageSubId();
		
		// 送信元のウィジェットIDを取得
		$fromWidgetId = $request->trimValueOf(M3_REQUEST_PARAM_FROM);
		
		// 対象のウィジェットIDを取得
		$targetWidgetId = $request->trimValueOf(M3_REQUEST_PARAM_WIDGET_ID);
		
		// 対象のウィジェットのページサブIDを取得
		$ret = $this->db->getSubPageId($targetWidgetId, $pageId, $rows);
		if ($ret){// データが存在する
			if (empty($rows[0]['pd_sub_id'])){		// 共通ウィジェットのときは、送信元にあわせる
				$ret = $this->db->getSubPageId($fromWidgetId, $pageId, $rows2);
				if ($ret){// データが存在する
					if (empty($rows2[0]['pd_sub_id'])){		// 送信元が共通ウィジェットのときは、既に設定されているページサブIDを使用
					} else {
						$gEnvManager->setCurrentPageSubId($rows2[0]['pd_sub_id']);
					}
				}
			} else {
				// 送信元があるか順にチェック
				for ($i = 0; $i < count($rows); $i++){
					$ret = $this->db->isExistsWidgetOnPage($pageId, $rows[$i]['pd_sub_id'], $fromWidgetId);
					if ($ret){	
						break;
					}
				}
				if ($i == count($rows)){		// 送信元が見つからない場合は1番目のページサブIDを使用
					$gEnvManager->setCurrentPageSubId($rows[0]['pd_sub_id']);
				} else {
					$gEnvManager->setCurrentPageSubId($rows[$i]['pd_sub_id']);// 存在するときは見つかったページサブIDで更新
				}
			}
		} else {		// 対象のウィジェットが見つからない場合は、互換ウィジェットを探す
			$widgetId = $this->db->getCompatibleWidgetId($targetWidgetId);
			if (!empty($widgetId)){
				$targetWidgetId = $widgetId;
				
				// 対象のウィジェットのページサブIDを取得
				$ret = $this->db->getSubPageId($targetWidgetId, $pageId, $rows);
				if ($ret){// データが存在する
					if (empty($rows[0]['pd_sub_id'])){		// 共通ウィジェットのときは、送信元にあわせる
						$ret = $this->db->getSubPageId($fromWidgetId, $pageId, $rows2);
						if ($ret){// データが存在する
							if (empty($rows2[0]['pd_sub_id'])){		// 送信元が共通ウィジェットのときは、既に設定されているページサブIDを使用
							} else {
								$gEnvManager->setCurrentPageSubId($rows2[0]['pd_sub_id']);
							}
						}
					} else {
						// 送信元があるか順にチェック
						for ($i = 0; $i < count($rows); $i++){
							$ret = $this->db->isExistsWidgetOnPage($pageId, $rows[$i]['pd_sub_id'], $fromWidgetId);
							if ($ret){	
								break;
							}
						}
						if ($i == count($rows)){		// 送信元が見つからない場合は1番目のページサブIDを使用
							$gEnvManager->setCurrentPageSubId($rows[0]['pd_sub_id']);
						} else {
							$gEnvManager->setCurrentPageSubId($rows[$i]['pd_sub_id']);// 存在するときは見つかったページサブIDで更新
						}
					}
				}
			}
		}
		// ページサブIDが見つからないときは、既に設定されている値を使用
		// 既に設定されている値は、URL「sub」パラメータで指定されている値か
		// 設定されていない場合はデフォルトのサブページID
		
		// ********** 指定ページへリダイレクト ***********
		// 実行パラメータ取得
		$todo = $request->trimValueOf(M3_REQUEST_PARAM_OPERATION_TODO);
		$this->redirect('?sub=' . $gEnvManager->getCurrentPageSubId() . '&' . $todo);
	}
	/**
	 * 最終HTML出力処理
	 *
	 * テンプレートの出力が完了した後、HTMLとして出力する最後の出力を行う
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 * @return string        				最終HTML
	 */
	function getLastContents($request)
	{
		global $gEnvManager;

		$contents = '';
		$pageId		= $gEnvManager->getCurrentPageId();
		$pageSubId	= $gEnvManager->getCurrentPageSubId();
		$cmd = $request->trimValueOf(M3_REQUEST_PARAM_OPERATION_COMMAND);
		
		if ($gEnvManager->isAdminDirAccess()){		// 管理画面へのアクセスのとき
			if ($gEnvManager->isSystemAdmin()){		// 管理者権限がある場合のみ有効
				// ウィジェットレイアウト用カーソル
				if ($cmd == M3_REQUEST_CMD_SHOW_POSITION_WITH_WIDGET){		// ウィジェット付きポジション表示
					//$contents .= '<div class="clear"></div>' . M3_NL;
					/*
					$contents .= '<div id="m3_overlay">' . M3_NL;		// ロード中表示
					$contents .= '<div id="preloader"><img src="' . $gEnvManager->getImagesUrl() . '/system/loader.gif" alt="" /></div>' . M3_NL;
					$contents .= '</div>' . M3_NL;*/
				
					$contents .= '<script type="text/javascript">' . M3_NL;
					$contents .= '<!--' . M3_NL;
				
					// テンプレート上のポジション名
					if (count($this->viewPosId) > 0){
						$posArrayStr = '[';
						for ($i = 0; $i < count($this->viewPosId); $i++){
							$posArrayStr .= '\'#' . $this->viewPosId[$i] . '\'';
							//$posArrayStr .= '\'' . $this->viewPosId[$i] . '\'';
							if ($i < count($this->viewPosId) - 1) $posArrayStr .= ',';
						}
						$posArrayStr .= ']';
						$contents .= 'var M3_POSITIONS=' . $posArrayStr . ';' . M3_NL;
					}
					// 画面定義のリビジョン番号
					$contents .= 'var M3_REVISION=' . $this->pageDefRev . ';' . M3_NL;
				
					// 更新用関数追加
					$contents .= 'function m3UpdateByConfig(serial){' . M3_NL;
					$contents .= '    window.m3.m3UpdateByConfig(serial);' . M3_NL;
					$contents .= '}' . M3_NL;
					
					// 携帯用テンプレートのときは、プレビューウィンドウの更新を通知する
					/*if ($gEnvManager->getCurrentTemplateType() == 1){
						$contents .= 'if (window.parent.m3UpdateByChildWindow) window.parent.m3UpdateByChildWindow();' . M3_NL;
					}*/
					$contents .= '// -->' . M3_NL;
					$contents .= '</script>' . M3_NL;
				}
				// ヘルプシステムの組み込み
				// ヘルプシステムは、「span」タグで埋め込み、「title」属性を使用する
				if ($this->useHelp){			// ヘルプ表示のとき
					$contents .= '<script type="text/javascript">' . M3_NL;
					$contents .= '<!--' . M3_NL;
					$contents .= '$(document).ready(function(){' . M3_NL;
					$contents .= '$(\'span.m3help\').cluetip({splitTitle: \'|\'});' . M3_NL;
					$contents .= '});' . M3_NL;
					$contents .= '//-->' . M3_NL;
					$contents .= '</script>' . M3_NL;
				} else {			// ヘルプ非表示のときは、title属性をクリアする
					$contents .= '<script type="text/javascript">' . M3_NL;
					$contents .= '<!--' . M3_NL;
					$contents .= 'var elems = document.getElementsByTagName("span");' . M3_NL;
					$contents .= 'for (var i = 0; i < elems.length; i++)' . M3_NL;
					$contents .= '{' . M3_NL;
					$contents .= '    elems[i].title = "";// ツールチップクリア' . M3_NL;
					$contents .= '}' . M3_NL;
					$contents .= '//-->' . M3_NL;
					$contents .= '</script>' . M3_NL;
				}
			}
		} else {		// 通常画面のとき
		}
		// Magic3出力コメント
		$contents .= '<!-- created by ' . M3_SYSTEM_NAME . ' v' . M3_SYSTEM_VERSION . ' - http://www.magic3.org -->' . M3_NL;
		$contents .= '<!-- convert time: ' . sprintf('%01.03f', microtime(true) - M3_MTIME) . ' -->' . M3_NL;
		return $contents;
	}
	/**
	 * Widget単体起動用のHTMLのヘッダ部(headタグ)出力
	 *
	 * Widgetの出力方法は、以下のパターンがある
	 *  ・HTMLヘッダ付加 - Widget単体で画面出力するためにHTMLヘッダを付加するパターン
	 *  ・HTMLヘッダなし - Wiget単体のタグ出力のみのパターン
	 *
	 * @param string $cmd		起動コマンド
	 */
	function startWidget($cmd)
	{
		global $gEnvManager;
		global $gRequestManager;
				
		// ウィジェット単体表示のときのみ出力
		if ($this->showWidget){
			// XHTML形式ではヘルプ機能にエラーが出るので一時的に戻す
			echo '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.1//EN" "http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd">' . M3_NL;
			echo '<html xmlns="http://www.w3.org/1999/xhtml" xml:lang="ja" lang="ja">' . M3_NL;
			echo '<head>' . M3_NL;
			
			// HTMLのヘッダ部(headタグ内)出力
			$this->getHeader();

			// ウィジェットのJavascriptを読み込む
			$widgetId = $gEnvManager->getCurrentWidgetId();		// カレントのウィジェットID
			
			// テンプレートの設定
			if ($cmd == M3_REQUEST_CMD_CONFIG_WIDGET){	// ウィジェット詳細設定画面のとき
				// 管理用テンプレート
				echo '<link href="' . $gEnvManager->getTemplatesUrl() . '/_admin/css/style.css" rel="stylesheet" type="text/css"/>' . M3_NL;
				echo '<!--[if IE]><link rel="stylesheet" type="text/css" media="screen" href="' . $gEnvManager->getTemplatesUrl() . '/_admin/css/iestyles.css" /><![endif]-->' . M3_NL;
			} else if ($cmd == M3_REQUEST_CMD_SHOW_WIDGET){		// ウィジェットの単体表示のとき
				// ウィジェットが管理用ウィジェットかどうかチェック
				if ($widgetId == self::ADMIN_WIDGET_ID){			// 管理用ウィジェットの場合
					// 管理用テンプレート
					echo '<link href="' . $gEnvManager->getTemplatesUrl() . '/_admin/css/style.css" rel="stylesheet" type="text/css"/>' . M3_NL;
					echo '<!--[if IE]><link rel="stylesheet" type="text/css" media="screen" href="' . $gEnvManager->getTemplatesUrl() . '/_admin/css/iestyles.css" /><![endif]-->' . M3_NL;
				} else {
					$curTemplate = $this->db->getDefaultTemplateId();
					$templateFile = $gEnvManager->getTemplatesUrl() . '/' . $curTemplate . self::DEFAULT_TEMPLATE_CSS_FILE;
					echo '<link href="' . $templateFile . '" rel="stylesheet" type="text/css"/>' . M3_NL;
				}
			}
			// ウィジェット情報取得
			$ret = $this->db->getWidgetInfo($widgetId, $row);

			// CSS読み込みが指定されていて、ディレクトリがあるときはディレクトリ内読み込み
			if ($row['wd_read_css']){
				$searchPath = $gEnvManager->getWidgetsPath() . '/' . $widgetId . '/' . M3_DIR_NAME_CSS;
				if (is_dir($searchPath)){
					$dir = dir($searchPath);
					while (($file = $dir->read()) !== false){
						$filePath = $searchPath . '/' . $file;
						if ($file != '.' && $file != '..' && is_file($filePath)
							&& strncmp($file, '_', 1) != 0){		// 「_」で始まる名前のファイルは読み込まない
							
							// CSSへのURLを作成
							$cssURL = $gEnvManager->getWidgetsUrl() . '/' . $widgetId . '/' . M3_DIR_NAME_CSS . '/' . $file;
							echo '<link rel="stylesheet" type="text/css" href="' . $cssURL . '" />' . M3_NL;
						}
					}
					$dir->close();
				}
			}
			
			// スクリプト読み込みが指定されていて、ディレクトリがあるときはディレクトリ内読み込み
			if ($row['wd_read_scripts']){
				$searchPath = $gEnvManager->getWidgetsPath() . '/' . $widgetId . '/' . M3_DIR_NAME_SCRIPTS;
				if (is_dir($searchPath)){
					$dir = dir($searchPath);
					while (($file = $dir->read()) !== false){
						$filePath = $searchPath . '/' . $file;
						if ($file != '.' && $file != '..' && is_file($filePath)
							&& strncmp($file, '_', 1) != 0){		// 「_」で始まる名前のファイルは読み込まない
							
							// スクリプトへのURLを作成
							$scriptURL = $gEnvManager->getWidgetsUrl() . '/' . $widgetId . '/' . M3_DIR_NAME_SCRIPTS . '/' . $file;
							
							// スクリプトをキャッシュ保存しない場合は、パラメータを付加
							if (!$this->hasScriptCache) $scriptURL .= $this->getCacheParam();
							echo '<script type="text/javascript" src="' . $scriptURL . '"></script>' . M3_NL;
						}
					}
					$dir->close();
				}
			}
			
			// ウィジェットのタイトルを設定
			$title = $row['wd_name'];
			echo '<title>' . $title . '</title>' . M3_NL;
			echo '</head>' . M3_NL;
			echo '<body>' . M3_NL;
			if ($cmd == M3_REQUEST_CMD_SHOW_WIDGET ||		// ウィジェットの単体表示のとき
				$cmd == M3_REQUEST_CMD_CONFIG_WIDGET){	// ウィジェット詳細設定画面のとき
				// 別ウィンドウで表示のときは、「閉じる」ボタンを表示
				$openBy = $gRequestManager->trimValueOf(M3_REQUEST_PARAM_OPEN_BY);		// ウィンドウオープンタイプ
				if (!empty($openBy)){
					echo '<br>' . M3_NL;
					echo '<div align="right"><input type="button" class="button" value="閉じる" onclick="window.close();" />&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;</div>' . M3_NL;
				}
			}
		//	echo '<br>' . M3_NL;
//			echo '<div align="center">' . M3_NL;
			echo '<!-- Widget Start -->' . M3_NL;
		}
	}
	/**
	 * Widget単体起動用のタグを閉じる
	 *
	 * @param string $cmd		起動コマンド
	 */
	function endWidget($cmd)
	{
		// ウィジェット単体表示のときのみ出力
		if ($this->showWidget){
			echo '<!-- Widget End -->' . M3_NL;
//			echo '</div>' . M3_NL;
			if ($cmd == M3_REQUEST_CMD_CONFIG_WIDGET){	// ウィジェット詳細設定画面のとき
				if ($this->updateParentWindow){			// 親ウィンドウ再描画のとき
					echo '<script type="text/javascript">' . M3_NL;
					echo '<!--' . M3_NL;
					echo '$(function(){' . M3_NL;
					//echo 'm3UpdateParentWindow();' . M3_NL;
					echo 'm3UpdateParentWindowByConfig(' . $this->updateDefSerial . ');' . M3_NL;// 更新する項目のページ定義シリアル番号
					echo '});' . M3_NL;
					echo '// -->' . M3_NL;
					echo '</script>' . M3_NL;
				}
			}
			echo '</body>' . M3_NL;
			echo '</html>' . M3_NL;
		}
	}
	/**
	 * HTMLのヘッダ部(headタグ内)出力
	 *
	 * システムに共通な定義をHTMLのheadタグ内に出力する
	 * mosFunc.phpからも実行されるので、このメソッドは引数なしに固定。
	 * この関数は、以下の「形式1」または「形式2」でheadタグ内に記述する
	 *
	 * 形式1:            <!-- m3:HTMLHeader -->
	 * 形式2(old style): <?php mosShowHead(); ?>
	 */
	function getHeader()
	{
		global $gEnvManager;
		global $gRequestManager;
		
		$this->outputHead = true;				// HTMLヘッダ出力を行ったかどうか
		
		// 実行コマンドを取得
		$cmd = $gRequestManager->trimValueOf(M3_REQUEST_PARAM_OPERATION_COMMAND);
			
		// ######### 携帯用サイトの場合は別にヘッダを作成する #########
		if ($gEnvManager->getIsMobileSite()){
			// キャラクターセット
			echo '<meta http-equiv="Content-Type" content="text/html; charset=Shift_JIS" />' . M3_NL;
		
			// キャッシュを保存させない
			echo '<meta http-equiv="Cache-Control" content="no-cache" />' . M3_NL;
		} else {
			// ********** メタタグの設定 **********
			echo '<meta http-equiv="content-script-type" content="text/javascript" />' . M3_NL;
			echo '<meta http-equiv="content-style-type" content="text/css" />' . M3_NL;
		
			// キャラクターセット
			echo '<meta http-equiv="content-type" content="text/html; charset=' . M3_HTML_CHARSET .'" />' . M3_NL;
			// 基準ディレクトリの指定
			if ($cmd == M3_REQUEST_CMD_SHOW_POSITION ||				// 表示位置を表示するとき
				$cmd == M3_REQUEST_CMD_SHOW_POSITION_WITH_WIDGET){	// 表示位置を表示するとき(ウィジェット付き)
				echo '<base href="' . $gEnvManager->getRootUrl() . '/" />' . M3_NL;
			}
		}
		echo self::HEAD_TAGS;			// HTMLヘッダの埋め込みデータ
	}
	/**
	 * HTMLヘッダ出力文字列の取得
	 *
	 * @return string		HTMLヘッダ出力文字列
	 */
	function getHeaderOutput()
	{
		global $gEnvManager;
		global $gRequestManager;
		global $mosConfig_live_site, $mosConfig_absolute_path, $mosConfig_favicon;
		
		// 実行コマンドを取得
		$cmd = $gRequestManager->trimValueOf(M3_REQUEST_PARAM_OPERATION_COMMAND);
		
		$replaceStr = '';		// 変換文字列
			
		// タイトルの設定
		if (!$this->showWidget){		// 単体実行以外のとき
			// 画面タイトル
			$titleItemCount = 0;		// タイトル項目数
			$defaultTitle = trim($this->db->getSiteDefValue(M3_TB_FIELD_SITE_TITLE));
			if (!empty($defaultTitle)) $titleItemCount++;
			if (!empty($this->headSubTitle)){		// サブタイトルが設定されているとき
				$titleItemCount += count($this->headSubTitle);
			}
			// タイトルフォーマットを取得
			$title = '';
			if ($titleItemCount > 0){
				$format = $this->db->getSystemConfig(self::CONFIG_KEY_HEAD_TITLE_FORMAT);
				if (empty($format)){
					$title = htmlspecialchars($defaultTitle);
				} else {
					$formats = explode(';', $format);
					$titleItemCount = ($titleItemCount > count($formats)) ? count($formats) : $titleItemCount;
					$title = $formats[$titleItemCount -1];
					$number = 1;
					if (!empty($defaultTitle)){
						$title = str_replace('$1', htmlspecialchars($defaultTitle), $title);
						$number++;
					}
					for ($i = 0; $i < count($this->headSubTitle); $i++){
						$key = '$' . $number;
						$value = htmlspecialchars(trim($this->headSubTitle[$i]));
						$title = str_replace($key, $value, $title);
						$number++;
					}
				}
			}
			if (!empty($title)) $replaceStr .= '<title>' . $title . '</title>' . M3_NL;

			// サイトの説明
			if (!empty($this->headDescription)) $replaceStr .= '<meta name="description" content="' . htmlspecialchars($this->headDescription) . '" />' . M3_NL;
			
			// 検索エンジン用キーワード
			if (!empty($this->headKeywords)) $replaceStr .= '<meta name="keywords" content="' . htmlspecialchars($this->headKeywords) . '" />' . M3_NL;
		}
		
		// ##### PC用URLと携帯用URLのアクセス別に処理 #####
		if ($gEnvManager->getIsMobileSite()){		// 携帯用URLのとき
		} else {			// PC用URLのとき
			// 検索ロボットへの指示
			$robots = htmlspecialchars(trim($this->db->getSiteDefValue(M3_TB_FIELD_SITE_ROBOTS)));
			if (!empty($robots)){
				$replaceStr .= '<meta name="robots" content="' . $robots . '" />' . M3_NL;
			}
		
			// サイト構築エンジン
			$replaceStr .= '<meta name="generator" content="' . M3_SYSTEM_NAME . ' ver.' . M3_SYSTEM_VERSION . ' - ' . M3_SYSTEM_DESCRIPTION . '" />' . M3_NL;		

			// アイコンの指定
			if ( !$mosConfig_favicon ) {
				$mosConfig_favicon = 'favicon.ico';
			}
			$icon = $mosConfig_absolute_path .'/images/'. $mosConfig_favicon;
			if ( !file_exists( $icon ) ) {
				$icon = $mosConfig_live_site .'/images/favicon.ico';
			} else {
				$icon = $mosConfig_live_site .'/images/' .$mosConfig_favicon;
			}
			$replaceStr .= '<link rel="shortcut icon" href="' . $icon .'" />' . M3_NL;
		
			// ##### 共通ライブラリの読み込み #####
			$this->db->getWidgetsIdWithLib($gEnvManager->getCurrentPageId(), $gEnvManager->getCurrentPageSubId(), $rows);
			for ($i = 0; $i < count($rows); $i++){
				$libArray = explode(',', trim($rows[$i]['wd_add_script_lib']));		// 通常画面への追加スクリプト
				for ($j = 0; $j < count($libArray); $j++){
					$lib = strtolower(trim($libArray[$j]));// 小文字に変換
					if (strcmp($lib, 'jquery') == 0){// jQuery本体のとき
						$this->addScriptFile(self::JQUERY_FILENAME);		// JQueryスクリプト追加
					} else if (isset($this->libFiles[$lib])){		// ライブラリが存在するとき
						// JQueryのプラグイン、JQuery UIの場合はJQueryを追加
						if (strncmp($lib, 'jquery.', 7) == 0 || strcmp($lib, self::LIB_JQUERY_UI) == 0){
							$this->addScriptFile(self::JQUERY_FILENAME);		// JQueryスクリプト追加
						}
					
						if (isset($this->libFiles[$lib]['script'])){
							$scriptFiles = $this->libFiles[$lib]['script'];
							for ($k = 0; $k < count($scriptFiles); $k++){
								$this->addScriptFile($scriptFiles[$k]);		// スクリプト追加
							}
						}
						if (isset($this->libFiles[$lib]['css'])){
							$cssFiles = $this->libFiles[$lib]['css'];
							for ($k = 0; $k < count($cssFiles); $k++){
								$this->addCssFile($cssFiles[$k]);		// CSS追加
							}
						}					
					}
				}
			}

			// ##### 共通CSS読み込み #####
			if ($gEnvManager->isAdminDirAccess()){		// 管理画面へのアクセスのとき
				if ($gEnvManager->isSystemAdmin()){		// 管理者権限がある場合のみ有効
					// 管理機能用共通ライブラリのCSSの読み込み
					$count = count($this->defaultAdminCssFiles);
					for ($i = 0; $i < $count; $i++){
						// CSSへのURLを作成
						$cssURL = $gEnvManager->getScriptsUrl() . '/' . $this->defaultAdminCssFiles[$i];
						$replaceStr .=  '<link rel="stylesheet" type="text/css" href="' . $cssURL . '" />' . M3_NL;
					}
				}
			} else {
				// 共通ライブラリのCSSの読み込み
				$count = count($this->defaultCssFiles);
				for ($i = 0; $i < $count; $i++){
					// CSSへのURLを作成
					$cssURL = $gEnvManager->getScriptsUrl() . '/' . $this->defaultCssFiles[$i];
					$replaceStr .=  '<link rel="stylesheet" type="text/css" href="' . $cssURL . '" />' . M3_NL;
				}
			}
			
			// ##### 表示モードによるCSS読み込み #####
			// ウィジェット付きポジション画面は管理画面のアクセスではない
			if ($cmd == M3_REQUEST_CMD_SHOW_POSITION_WITH_WIDGET){		// ウィジェット付きポジション表示
				// ウィジェット操作用CSS
				$cssURL = $gEnvManager->getScriptsUrl() . self::M3_ADMIN_WIDGET_CSS_FILE;
				$replaceStr .=  '<link rel="stylesheet" type="text/css" href="' . $cssURL . '" />' . M3_NL;
			}
			
			// ##### ウィジェットごとのCSS読み込み #####
			// CSSがあるウィジェットを取得
			$this->db->getWidgetsIdWithCss($gEnvManager->getCurrentPageId(), $gEnvManager->getCurrentPageSubId(), $rows);
			for ($i = 0; $i < count($rows); $i++){
				$searchPath = $gEnvManager->getWidgetsPath() . '/' . $rows[$i]['wd_id'] . '/' . M3_DIR_NAME_CSS;
				// ディレクトリがあるときはディレクトリ内読み込み
				if (is_dir($searchPath)){
					$dir = dir($searchPath);
					while (($file = $dir->read()) !== false){
						$filePath = $searchPath . '/' . $file;
						if ($file != '.' && $file != '..' && is_file($filePath)
							&& strncmp($file, '_', 1) != 0){		// 「_」で始まる名前のファイルは読み込まない
					
							// CSSへのURLを作成
							$cssURL = $gEnvManager->getWidgetsUrl() . '/' . $rows[$i]['wd_id'] . '/' . M3_DIR_NAME_CSS . '/' . $file;
							$replaceStr .=  '<link rel="stylesheet" type="text/css" href="' . $cssURL . '" />' . M3_NL;
						}
					}
					$dir->close();
				}
			}
			// ウィジェットからの追加のCSS読み込み
			$count = count($this->headCssFiles);
			for ($i = 0; $i < $count; $i++){
				$cssUrl = $this->headCssFiles[$i];
				$replaceStr .=  '<link rel="stylesheet" type="text/css" href="' . $cssUrl . '" />' . M3_NL;
			}

			// ##### 共通Javascriptの読み込み #####
			if ($gEnvManager->isAdminDirAccess()){		// 管理画面へのアクセスのとき
				if ($gEnvManager->isSystemAdmin()){		// 管理者権限がある場合のみ有効
					// 管理画面用の共通スクリプトを読み込む
					$count = count($this->defaultAdminScriptFiles);
					for ($i = 0; $i < $count; $i++){
						$scriptFilename = $this->defaultAdminScriptFiles[$i];

						// スクリプトをキャッシュ保存しない場合は、パラメータを付加
						$scriptURL = $gEnvManager->getScriptsUrl() . '/' . $scriptFilename;
						if (!$this->hasScriptCache) $scriptURL .= $this->getCacheParam();
						$replaceStr .=  '<script type="text/javascript" src="' . $scriptURL . '"></script>' . M3_NL;
					}
					if ($cmd == M3_REQUEST_CMD_SHOW_POSITION_WITH_WIDGET){		// ウィジェット付きポジション表示のときは、ウィジェット操作ライブラリを読み込む
						// wigetのドラッグドロップ用
						$scriptURL = $gEnvManager->getScriptsUrl() . '/' . self::M3_ADMIN_WIDGET_SCRIPT_FILENAME;
						if (!$this->hasScriptCache) $scriptURL .= $this->getCacheParam();// スクリプトをキャッシュ保存しない場合は、パラメータを付加
						$replaceStr .=  '<script type="text/javascript" src="' . $scriptURL . '"></script>' . M3_NL;
						// キー割付用
						$scriptURL = $gEnvManager->getScriptsUrl() . '/' . self::JQUERY_JSHOTKEYS_FILENAME;
						if (!$this->hasScriptCache) $scriptURL .= $this->getCacheParam();// スクリプトをキャッシュ保存しない場合は、パラメータを付加
						$replaceStr .=  '<script type="text/javascript" src="' . $scriptURL . '"></script>' . M3_NL;
					}
				}
			} else {			// 通常画面
				// Ajaxライブラリを使用しているウィジェットがあるときは追加
				if ($this->db->isExistsWidgetWithAjax($gEnvManager->getCurrentPageId(), $gEnvManager->getCurrentPageSubId())){
					$this->addScriptFile(self::JQUERY_FILENAME);		// デフォルトAjaxライブラリ追加
					$this->addScriptFile(self::M3_OPTION_SCRIPT_FILENAME);	// Magic3のオプションライブラリ追加
					//$this->addScriptFile(self::M3_AJAX_SCRIPT_FILENAME);	// Magic3のAjaxライブラリ追加
				}
				
				$count = count($this->defaultScriptFiles);
				for ($i = 0; $i < $count; $i++){
					$scriptURL = $gEnvManager->getScriptsUrl() . '/' . $this->defaultScriptFiles[$i];
			
					// スクリプトをキャッシュ保存しない場合は、パラメータを付加
					if (!$this->hasScriptCache) $scriptURL .= $this->getCacheParam();
					$replaceStr .=  '<script type="text/javascript" src="' . $scriptURL . '"></script>' . M3_NL;
				}
			}
			// ##### ウィジェットごとのJavaScript読み込み #####
			// スクリプトがあるウィジェットを取得
			$this->db->getWidgetsIdWithScript($gEnvManager->getCurrentPageId(), $gEnvManager->getCurrentPageSubId(), $rows);
			for ($i = 0; $i < count($rows); $i++){
				$searchPath = $gEnvManager->getWidgetsPath() . '/' . $rows[$i]['wd_id'] . '/' . M3_DIR_NAME_SCRIPTS;
			
				// ディレクトリがあるときはディレクトリ内読み込み
				if (is_dir($searchPath)){
					$dir = dir($searchPath);
					while (($file = $dir->read()) !== false){
						$filePath = $searchPath . '/' . $file;
						if ($file != '.' && $file != '..' && is_file($filePath)
							&& strncmp($file, '_', 1) != 0){		// 「_」で始まる名前のファイルは読み込まない
						
							// スクリプトへのURLを作成
							$scriptURL = $gEnvManager->getWidgetsUrl() . '/' . $rows[$i]['wd_id'] . '/' . M3_DIR_NAME_SCRIPTS . '/' . $file;
							// スクリプトをキャッシュ保存しない場合は、パラメータを付加
							if (!$this->hasScriptCache) $scriptURL .= $this->getCacheParam();
							$replaceStr .=  '<script type="text/javascript" src="' . $scriptURL . '"></script>' . M3_NL;
						}
					}
					$dir->close();
				}
			}
			// ウィジェットからの追加のCSS読み込み
			$count = count($this->headScriptFiles);
			for ($i = 0; $i < $count; $i++){
				$scriptUrl = $this->headScriptFiles[$i];
		
				// スクリプトをキャッシュ保存しない場合は、パラメータを付加
				if (!$this->hasScriptCache) $scriptUrl .= $this->getCacheParam();
				$replaceStr .=  '<script type="text/javascript" src="' . $scriptUrl . '"></script>' . M3_NL;
			}
			
			// JavaScriptグローバル変数の設定
			$replaceStr .= '<script type="text/javascript">' . M3_NL;
			$replaceStr .= '<!--' . M3_NL;
			$replaceStr .= '// Magic3 Global values' . M3_NL;
			$replaceStr .= 'var M3_ROOT_URL="' . $gEnvManager->getRootUrl() . '";' . M3_NL;		// システムルートURL
			if ($cmd == M3_REQUEST_CMD_SHOW_POSITION_WITH_WIDGET){		// ウィジェット付きポジション表示
				$pageId = $gRequestManager->trimValueOf('page');		// ページID
				$pageSubId = $gRequestManager->trimValueOf('sub');// ページサブID
				$task = $gRequestManager->trimValueOf('task');
			
				$replaceStr .= 'var M3_DEFAULT_ADMIN_URL="' . $gEnvManager->getDefaultAdminUrl() . '";' . M3_NL;		// 管理機能URL
				if (!empty($task)){		// 戻りタスクが設定されているときのみ最大化可能
					$replaceStr .= 'function gobackPagedef(){' . M3_NL;
					$replaceStr .= '    window.location.href = "' . $gEnvManager->getDefaultAdminUrl() . '?pageid=' . $pageId . '&pagesubid=' . $pageSubId . '&task=' . $task . '";' . M3_NL;
					$replaceStr .= '}' . M3_NL;
					$replaceStr .= '$(document).ready(function(){' . M3_NL;
					$replaceStr .= '    jQuery.hotkeys.add("esc", function(){' . M3_NL;
					$replaceStr .= '        gobackPagedef();' . M3_NL;
					$replaceStr .= '    });' . M3_NL;
					$replaceStr .= '});' . M3_NL;
				}
			}
			// ポップアップメッセージがある場合は表示
			if (!empty($this->popupMsg)){
				$replaceStr .=  'alert("' . $this->popupMsg . '");' . M3_NL;
			}
			$replaceStr .= '// -->' . M3_NL;
			$replaceStr .= '</script>' . M3_NL;
			
			// HEADタグに埋め込むCSS,JavaScript,任意文字列
			if (count($this->headCss) > 0){
				// CSSの場合は全体をstyleタグで囲む
				$replaceStr .= '<style type="text/css">' . M3_NL;
				$replaceStr .= '<!--' . M3_NL;
				for ($i = 0; $i < count($this->headCss); $i++){
					$replaceStr .= $this->headCss[$i];
				}
				$replaceStr .= M3_NL . '//-->' . M3_NL;
				$replaceStr .= '</style>' . M3_NL;
			}
			if (count($this->headScript) > 0){
				// JavaScriptの場合は全体をscriptタグで囲む
				$replaceStr .= '<script type="text/javascript">' . M3_NL;
				$replaceStr .= '<!--' . M3_NL;
				for ($i = 0; $i < count($this->headScript); $i++){
					$replaceStr .= $this->headScript[$i];
				}
				$replaceStr .= M3_NL . '//-->' . M3_NL;
				$replaceStr .= '</script>' . M3_NL;
			}
			if (count($this->headString) > 0){
				// 任意文字列の場合はそのまま追加
				for ($i = 0; $i < count($this->headString); $i++){
					$replaceStr .= $this->headString[$i];
				}
			}
		}
		return $replaceStr;
	}
	/**
	 * 各部品のHTML出力
	 *
	 * @param string $position		HTMLテンプレート上の書き出し位置
	 * @param int    $style			ウィジェットの表示スタイル(Joomla!v1.5テンプレート用)
	 * @param bool   $echo			echo出力するかどうか
	 * @return string				出力コンテンツ
	 */
	function getContents($position, $style = '', $echo = true)
	{
		global $gRequestManager;
		global $gEnvManager;
		global $gDesignManager;
		
		// ファイル名、ページ名を取得
		$filename	= $gEnvManager->getCurrentPageId();
		$subId		= $gEnvManager->getCurrentPageSubId();
		if (empty($subId)) $subId = $gEnvManager->getDefaultPageSubId();

		// ポジション名表示モードに応じて出力を作成
		$contents = '';		// 出力コンテンツ
		switch ($this->showPositionMode){
			case 0:		// 通常画面
				// ページ定義を取得。同じポジションが続く場合は最初の一度だけ定義を取得
				if (empty($this->pageDefPosition) || $position != $this->pageDefPosition){		// ポジションが異なる場合
					$ret = $this->db->getPageDef($filename, $subId, $position, $rows);
					if ($ret){	// 1行以上データが取得できたとき
						$this->pageDefRows = $rows;			// ページ定義レコード
						$this->pageDefPosition = $position;
					} else {
						$this->pageDefRows = array();
						$this->pageDefPosition = '';
					}
				}
				// ウィジェットを実行
				$count = count($this->pageDefRows);
				
				if ($echo){			// echo出力のとき
					for ($i = 0; $i < $count; $i++){
						$ret = $this->pageDefLoop($i, $this->pageDefRows[$i]);
						if (!$ret) return;		// 処理中断のときは終了
					}
				} else {
					for ($i = 0; $i < $count; $i++){
						ob_clean();
						$ret = $this->pageDefLoop($i, $this->pageDefRows[$i]);
						$widgetContent = ob_get_contents();

						// Joomla!ウィジェットの出力に埋め込む
						if (strcasecmp($style, 'none') == 0){
							$contents .= $widgetContent;
						} else {
							$title = $this->pageDefRows[$i]['pd_title'];
							$contents .= JModuleRender::getModuleContents($style, $widgetContent, $title);
						}
						if (!$ret) return;		// 処理中断のときは終了
					}
				}
			
				if ($position == 'main'){		// メイン部のときは、メッセージを出力
					/*if (strlen($this->popupMsg) > 0){
						echo "\n<script language=\"javascript\">alert('" . addslashes($this->popupMsg) . "');</script>";
					}*/
				} else if ($position == 'debug'){		// デバッグ文出力
				} else {

				}
				break;
			case 1:		// ポジション表示
				$contents .= '<div style="background-color:#eee;margin:2px;padding:10px;border:3px solid #f00;color:#700;">ポジション名: ';
				$contents .= '<b>' . $position . '</b>';
				$contents .= '</div>';
				break;
			case 2:		// ポジション表示(ウィジェット付き)
				$rev = '555';			// データのリビジョン番号
				// ポジションのHTMLタグIDを作成
				$num = 0;
				$posId = '';
				for ($i = 0; $i < 100; $i++){
					$posId = $position . '_' . $num;
					$viewPosId = self::POSITION_TAG_HEAD . $posId;
					if (!in_array($viewPosId, $this->viewPosId)) break;
					$num++;
				}
				$this->viewPosId[] = $viewPosId;// IDを保存
				
				// ページ定義を取得。同じポジションが続く場合は最初の一度だけ定義を取得
				if (empty($this->pageDefPosition) || $position != $this->pageDefPosition){		// ポジションが異なる場合
					$ret = $this->db->getPageDef($filename, $subId, $position, $rows);
					if ($ret){	// 1行以上データが取得できたとき
						$this->pageDefRows = $rows;			// ページ定義レコード
						$this->pageDefPosition = $position;
					} else {
						$this->pageDefRows = array();
						$this->pageDefPosition = '';
					}
				}
				$contents .= '<div id="' . $viewPosId . '" class="m3_widgetpos_box" m3="pos:' . $position . ';rev:' . $rev . ';">' . M3_NL;		// リビジョン番号を付加
				$contents .= '<h2>' . $position . '</h2>' . M3_NL;
				
				// ウィジェットイメージを表示
				$widgetTagHead = self::WIDGET_TAG_HEAD . $posId;
				$contents .= $this->getWidgetList($widgetTagHead, $this->pageDefRows);

				$contents .= '</div>' . M3_NL;
				break;
			default:
				$contents .= '<div style="background-color:#eee;margin:2px;padding:10px;border:1px solid #f00;color:#700;">param error</div>';
				break;
		}
		// ポジションを保存
		$this->viewPositions[] = $position;
		
		if ($echo){
			echo $contents;
			return '';
		} else {
			return $contents;
		}
	}
	/**
	 * 各部品のHTML出力
	 *
	 * @param string $position		HTMLテンプレート上の書き出し位置
	 * @return int   				コンテンツの数
	 */
	function getWidgetsCount($position)
	{
		global $gRequestManager;
		global $gEnvManager;

		// 実行コマンドを取得
		$cmd = $gRequestManager->trimValueOf(M3_REQUEST_PARAM_OPERATION_COMMAND);
		if ($cmd == M3_REQUEST_CMD_SHOW_POSITION_WITH_WIDGET){		// ウィジェット付きポジション表示
			return 1;		// ウィジェットが設定されていないポジション名を表示するために固定で値を返す
		}
		
		// ファイル名、ページ名を取得
		$filename	= $gEnvManager->getCurrentPageId();
		$subId		= $gEnvManager->getCurrentPageSubId();
		if (empty($subId)) $subId = $gEnvManager->getDefaultPageSubId();

		// 取得しようとするページ定義のポジションが既に取得しているポジションと異なるときはデータを取得
		if (empty($this->pageDefPosition) || $position != $this->pageDefPosition){		// 現在取得しているページ定義のポジション
			$ret = $this->db->getPageDef($filename, $subId, $position, $rows);
			if ($ret){	// 1行以上データが取得できたとき
				$this->pageDefRows = $rows;			// ページ定義レコード
				$this->pageDefPosition = $position;
			} else {
				$this->pageDefRows = array();
				$this->pageDefPosition = '';
			}
		}
		return count($this->pageDefRows);
	}
	/**
	 * ウィジェット情報取得
	 *
	 * 画面作成機能でウィジェット情報を取得するためのAjaxインターフェイス
	 *
	 * @param RequestManager $request		HTTPリクエスト処理クラス
	 */
	function getWidgetInfoByAjax($request)
	{
		global $gEnvManager;
		global $gDesignManager;
		
		// アクセスするページIDからPC用か携帯用かを判断
		$widgetForPc = true;
		$pageId		= $request->trimValueOf(M3_REQUEST_PARAM_PAGE_ID);
		$pageSubId	= $request->trimValueOf(M3_REQUEST_PARAM_PAGE_SUB_ID);
		$mobilePageIdPrefix = M3_DIR_NAME_MOBILE . '_';
		if (strncmp($pageId, $mobilePageIdPrefix, strlen($mobilePageIdPrefix)) == 0) $widgetForPc = false;

		$task = $request->trimValueOf(M3_REQUEST_PARAM_OPERATION_TASK);
		if ($task == 'list'){
			if ($widgetForPc){		// PC用画面作成のとき
				$ret = $this->db->getAvailableWidgetList($rows);
			} else {	// 携帯画面作成のとき
				$ret = $this->db->getAvailableMobileWidgetList($rows);
			}
			if ($ret){
				for ($i = 0; $i < count($rows); $i++){
					$widgetId = $rows[$i]['wd_id'];
					$desc = $rows[$i]['wd_description'];
					$widgetTag = self::WIDGET_TYPE_TAG_HEAD . $widgetId;
					$image = $gDesignManager->getWidgetIconUrl($widgetId, self::WIDGET_ICON_IMG_SIZE);
					$imageTag = '<img class="' . self::WIDTET_CLASS_NAME . '" src="' . $image . '" ';
					$imageTag .= 'width="' . self::WIDGET_ICON_IMG_SIZE . '"';
					$imageTag .= ' height="' . self::WIDGET_ICON_IMG_SIZE . '"';
					$imageTag .= ' />';
					echo '<dl class="m3_widgetlist_item" id="' . $widgetTag . '">' . M3_NL;
					echo '<dt>' . $rows[$i]['wd_name'] . '</dt>' . M3_NL;			// ウィジェット名
					echo '<dd><table width="100%"><tr valign="top"><td width="35">' . $imageTag . '</td><td>' . $desc . '</td></tr></table></dd>' . M3_NL;
					echo '</dl>' . M3_NL;
				}
			}
		} else if ($task == 'wget' || $task == 'wdelete' || $task == 'wtoggle' || $task == 'wadd' || $task == 'wmove'){	// ウィジェット再取得、ウィジェット削除,ウィジェット共通属性変更、ウィジェット追加、ウィジェット移動のとき
			$rev	= $request->trimValueOf('rev');			// リビジョン
			$serial = $request->trimValueOf('serial');
			$position = $request->trimValueOf('pos');
			$widgetsStr = $request->trimValueOf('widgets');
			if (empty($widgetsStr)){
				$widgets = array();
			} else {
				$widgets = explode(',', $widgetsStr);
			}
			$shared = $request->trimValueOf('shared');
			$updatepos = explode(',', $request->trimValueOf('updatepos'));
			$index = $request->trimValueOf('index');
			
			// 処理ごとのパラメータ
			if ($task == 'wmove'){
				$positions = explode(',', $position);
				if (count($positions) >= 2){
					$position = $positions[0];
					$position2 = $positions[1];
				} else {
					$position = $positions[0];
					$position2 = '';
				}
			}
			// ##### エラーチェック #####
			$isErr = false;
			// リビジョンのエラーチェック
			$rev = '111';			// データのリビジョン番号

			// 変更前データ取得
			$ret = $this->db->getPageDef($pageId, $pageSubId, $position, $rows);	// 0レコードでも正常とする
			
			// 変更前のウィジェットのシリアル番号をチェック
			if (count($widgets) == count($rows)){
				if (!($task == 'wmove' && empty($position2))){			// 同一ブロック内の移動の場合はチェックなし
					for ($i = 0; $i < count($rows); $i++){
						if ($widgets[$i] != $rows[$i]['pd_serial']){// シリアル番号
							$isErr = true;
							break;
						}
					}
				}
			} else {
				$isErr = true;
			}

			// データの更新
			if (!$isErr){		// エラーなしのとき
				if ($task == 'wdelete'){
					$ret = $this->db->deleteWidget($serial);
				} else if ($task == 'wtoggle'){
					$newShared = 0;
					if (empty($shared)) $newShared = 1;
					$ret = $this->db->toggleSharedWidget($pageId, $pageSubId, $serial, $newShared);
				} else if ($task == 'wadd'){	// ウィジェットの追加
					$widget = $request->trimValueOf('widget');
					
					// エラーチェック
					if (empty($widget)) $isErr = true;
					
					// ウィジェットを追加
					if (!$isErr) $this->db->addWidget($pageId, $pageSubId, $position, $widget, $index);
				} else if ($task == 'wmove'){
					// ウィジェットを移動
					if (!$isErr) $this->db->moveWidget($pageId, $pageSubId, $position, $serial, $index);
				}
			}
			// 再表示データ取得
			$ret = $this->db->getPageDef($pageId, $pageSubId, $position, $rows);// 0レコードでも正常とする
			
			// 移動のときは、移動元と移動先の再表示データを取得
			if ($task == 'wmove' && !empty($position2)){
				$ret = $this->db->getPageDef($pageId, $pageSubId, $position2, $rows2);// 0レコードでも正常とする
			}

			// 更新データを作成
			// 更新対象のポジションブロック
			echo '<div>' . M3_NL;
			for ($i = 0; $i < count($updatepos); $i++){
				// ウィジェットIDヘッダ作成
				$widgetTagHead = str_replace(self::POSITION_TAG_HEAD, self::WIDGET_TAG_HEAD, $updatepos[$i]);
					
				// ポジション名取得
				$posName = str_replace(self::POSITION_TAG_HEAD, '', substr($updatepos[$i], 0, strlen($updatepos[$i]) - strlen(strrchr($updatepos[$i], "_"))));
				if ($task == 'wmove' && $posName == $position2){
					// ウィジェット一覧外枠
					echo '<div id="' . $updatepos[$i] . '" class="m3_widgetpos_box" m3="pos:' . $position2 . ';rev:' . $rev . ';">' . M3_NL;		// リビジョン番号を付加
					echo '<h2>' . $position2 . '</h2>' . M3_NL;
				
					// ウィジェット一覧出力
					echo $this->getWidgetList($widgetTagHead, $rows2);
				} else {
					// ウィジェット一覧外枠
					echo '<div id="' . $updatepos[$i] . '" class="m3_widgetpos_box" m3="pos:' . $position . ';rev:' . $rev . ';">' . M3_NL;		// リビジョン番号を付加
					echo '<h2>' . $position . '</h2>' . M3_NL;
				
					// ウィジェット一覧出力
					echo $this->getWidgetList($widgetTagHead, $rows);
				}
				// ウィジェット一覧外枠
				echo '</div>' . M3_NL;
			}
			echo '</div>' . M3_NL;
		}
	}
	/**
	 * 画面作成用ウィジェット一覧データ出力
	 *
	 * @param string $widgetTagHead		HTMLウィジェットID用のヘッダ
	 * @param array $rows				ウィジェット一覧データ
	 * @return string					ウィジェット一覧出力
	 */
	function getWidgetList($widgetTagHead, $rows)
	{
		global $gDesignManager;
		global $gEnvManager;
		
		$contents = '';		// ウィジェット一覧出力コンテンツ
		for ($i = 0; $i < count($rows); $i++){
			$widgetId = $rows[$i]['wd_id'];
			$desc = $rows[$i]['wd_description'];
			$widgetIndex = $rows[$i]['pd_index'];		// 表示順
			$configId = $rows[$i]['pd_config_id'];		// 定義ID
			$serial = $rows[$i]['pd_serial'];		// シリアル番号
			$widgetTag = $widgetTagHead . '_' . $i;				// ウィジェット識別用ユニークタグ
			$image = $gDesignManager->getWidgetIconUrl($widgetId, self::WIDGET_ICON_IMG_SIZE);
			$imageTag = '<img class="' . self::WIDTET_CLASS_NAME . '" src="' . $image . '" ';
			$imageTag .= 'width="' . self::WIDGET_ICON_IMG_SIZE . '"';
			$imageTag .= ' height="' . self::WIDGET_ICON_IMG_SIZE . '"';
			$imageTag .= ' />';
			// 定義名
			if (empty($rows[$i]['pd_config_name'])){
				if ($rows[$i]['wd_use_instance_def'] && $configId == 0){		// インスタンス定義が必要で未定義のとき
					$configName = '<span style="color:red;">[未設定]</span>';
				} else {
					$configName = '';
				}
			} else {
				$configName = '[' . $rows[$i]['pd_config_name'] . ']';
			}
			$hasAdmin = '0';		// 管理画面があるかどうか
			$configImg = '';	// 設定アイコンの表示
			if ($rows[$i]['wd_has_admin']){
				$hasAdmin = '1';
				$configImg = '<a href="#" onclick="m3ShowConfigWindow(\'' . $widgetId . '\', \'' . $configId . '\', \'' . $serial . '\');return false;" >' .
								'<img src="' . $gEnvManager->getRootUrl() . '/images/system/config.png" /></a>&nbsp;';
			}
			$shared = '0';		// 共通属性があるかどうか
			$sharedClassName = '';
			if (empty($rows[$i]['pd_sub_id'])){
				$shared = '1';	// 共通ウィジェットのとき
				$sharedClassName = 'm3_widget_shared';			// 共通ウィジェットのクラス
			}
			$m3Option = 'm3="widgetid:' . $widgetId . '; serial:' . $serial . '; configid:' . $configId . '; useconfig:' . $hasAdmin . '; shared:' . $shared . '"';
			$contents .= '<dl class="m3_widget_sortable" id="' . $widgetTag . '" ' . $m3Option . ' >' . M3_NL;
			$contents .= '<dt class="m3_widget_with_check_box ' . $sharedClassName . '">' . $rows[$i]['wd_name'] . '</dt>' . M3_NL;
			$contents .= '<dd><table width="100%"><tr valign="top"><td width="35">' . $imageTag . '</td><td>' . $desc . '</td></tr><table>' . M3_NL;
			$contents .= '<table width="100%"><tr><td>' . $configName . '</td><td align="right">' . $configImg . $widgetIndex . '</td></tr></table></dd>' . M3_NL;
			$contents .= '</dl>' . M3_NL;
		}
		return $contents;
	}
	/**
	 * 取得した画面定義情報を処理
	 *
	 * @param int $index			行番号
	 * @param array $fetchedRow		fetch取得した行
	 * @param array $param    		任意パラメータ
	 * @return bool					処理を中断するかどうか
	 */
	function pageDefLoop($index, $fetchedRow)
	{
		global $gEnvManager;
		global $gDesignManager;
		global $gErrorManager;

		// ページ作成中断のときは終了
		if ($this->isAbort) return false;
		
		// ウィジェットID取得
		$widgetId = $fetchedRow['wd_id'];		
		$widgetIndexFile = $gEnvManager->getWidgetsPath() . '/' . $widgetId . '/index.php';

		// その他パラメータ取得
		$configId = $fetchedRow['pd_config_id'];		// ウィジェット定義ID
		if ($configId == 0) $configId = '';
		$prefix = $fetchedRow['wd_suffix'];		// サフィックス文字列
		$serial = $fetchedRow['pd_serial'];		// シリアル番号
		$style	= $fetchedRow['pd_style'];		// CSS
		$lateLaunchIndex = $fetchedRow['wd_launch_index'];		// 遅延実行インデックス
		
		// ウィジェットが遅延実行に指定されている場合は、ウィジェットのタグのみ埋め込む
		if ($lateLaunchIndex > 0){		// 遅延実行のとき
			// 遅延実行ウィジェットリストに追加
			if (!isset($this->lateLaunchWidgetList[$widgetId])){
				$this->lateLaunchWidgetList[$widgetId] = (int)$lateLaunchIndex;
			}
			// 実行パラメータ保存
			$maxNo = 0;		// 最大シリアル番号
			$count = count($this->latelaunchWidgetParam);
			for ($i = 0; $i < $count; $i++){
				list($wId, $index, $tmp1, $tmp2, $tmp3, $tmp4) = $this->latelaunchWidgetParam[$i];
				if ($wId == $widgetId) $maxNo = $index + 1;
			}
			$this->latelaunchWidgetParam[] = array($widgetId, $maxNo, $configId, $prefix, $serial, $style);
			
			// 遅延実行用タグを埋め込む
			echo self::WIDGET_ID_TAG_START . $widgetId . self::WIDGET_ID_SEPARATOR . $maxNo . self::WIDGET_ID_TAG_END;
		} else {
			// ウィジェットが存在する場合は実行
			if (!file_exists($widgetIndexFile)) {
				echo 'widget not found error: ' . $widgetId;
			} else {
				// パラメータ初期化
				$this->lastHeadCss = '';			// 最後に設定したHTMLヘッダにCSS出力する文字列
				$this->lastHeadScript = '';			// 最後に設定したHTMLヘッダにJavascript出力する文字列
				$this->lastHeadString = '';			// 最後に設定したHTMLヘッダに出力する任意文字列
	
				// ウィジェット親のCSS定義があるときは、タグを追加
				if (!empty($style)) echo '<div style="' . $style . '">' . M3_NL;
					
				// ウィジェットの前出力
				echo $gDesignManager->getAdditionalWidgetOutput(true);
				
				// 作業中のウィジェットIDを設定
				$gEnvManager->setCurrentWidgetId($widgetId);
	
				// ウィジェット定義IDを設定
				$gEnvManager->setCurrentWidgetConfigId($configId);
			
				// ページ定義のシリアル番号を設定
				$gEnvManager->setCurrentPageDefSerial($fetchedRow['pd_serial']);
					
				// パラメータを設定
				$gEnvManager->setCurrentWidgetPrefix($prefix);		// プレフィックス文字列
			
				// 実行ログを残す
				$this->db->writeWidgetLog($widgetId, 0/*ページ実行*/);
				
				// ウィジェットを実行
				//require_once($widgetIndexFile);
				// ウィジェットの呼び出しは、複数回存在する可能性があるのでrequire_once()で呼び出さない
				$msg = 'widget-start(' . $widgetId . ')';
				$gErrorManager->writeDebug(__METHOD__, $msg);		// 時間計測用
				require($widgetIndexFile);
				$msg = 'widget-end(' . $widgetId . ')';
				$gErrorManager->writeDebug(__METHOD__, $msg);		// 時間計測用

				// 作業中のウィジェットIDを解除
				$gEnvManager->setCurrentWidgetId('');
			
				// ウィジェット定義IDを解除
				$gEnvManager->setCurrentWidgetConfigId('');
				
				// ページ定義のシリアル番号を解除
				$gEnvManager->setCurrentPageDefSerial(0);
			
				// パラメータを解除
				$gEnvManager->setCurrentWidgetPrefix('');				// プレフィックス文字列

				// ウィジェットの後出力
				echo $gDesignManager->getAdditionalWidgetOutput(false);
//				echo "<!-- ".time()." -->";

				// ウィジェット親のCSS定義があるときは、タグを追加
				if (!empty($style)) echo '</div>' . M3_NL;
			}
		}
		return true;		// 処理継続
	}
	/**
	 * インナーウィジェットの操作
	 *
	 * @param int $cmdNo				実行コマンド(0=初期化、1=更新、2=計算、10=コンテンツ取得)
	 * @param string $id				ウィジェットID+インナーウィジェットID
	 * @param string $configId			インナーウィジェット定義ID
	 * @param object $paramObj			インナーウィジェットに渡すパラメータオブジェクト
	 * @param object $optionParamObj	インナーウィジェットに渡すパラメータオブジェクト(オプション)
	 * @param object $resultObj			インナーウィジェットから返る結果オブジェクト
	 * @param string $content			出力内容
	 * @param bool $isAdmin				管理者機能(adminディレクトリ以下)かどうか
	 * @return bool						true=成功、false=失敗
	 */
	function commandIWidget($cmdNo, $id, $configId, &$paramObj, &$optionParamObj, &$resultObj, &$content, $isAdmin = false)
	{
		global $gEnvManager;
		global $gCmdParamManager;
		
		// ウィジェットIDとインナーウィジェットIDを取り出す
		list($widgetId, $iWidgetId) = explode(M3_WIDGET_ID_SEPARATOR, $id);

		// インナーウィジェットに渡すパラメータを設定
		switch ($cmdNo){
			case 0:
				$cmd = self::IWIDTET_CMD_INIT;			// 初期化
				$gCmdParamManager->setParam($id . M3_WIDGET_ID_SEPARATOR . $configId, $cmd, $paramObj, $optionParamObj);
				break;
			case 1:
				$cmd = self::IWIDTET_CMD_UPDATE;		// 更新
				$gCmdParamManager->setParam($id . M3_WIDGET_ID_SEPARATOR . $configId, $cmd);		// 設定されているパラメータの更新は行わない
				break;
			case 2:
				$cmd = self::IWIDTET_CMD_CALC;			// 計算
				$gCmdParamManager->setParam($id . M3_WIDGET_ID_SEPARATOR . $configId, $cmd, $paramObj, $optionParamObj);
				break;
			case 10:
				$cmd = self::IWIDTET_CMD_CONTENT;			// コンテンツ取得
				$gCmdParamManager->setParam($id . M3_WIDGET_ID_SEPARATOR . $configId, $cmd);		// 設定されているパラメータの更新は行わない
				break;
			default:
				$cmd = '';
				$gCmdParamManager->setParam($id . M3_WIDGET_ID_SEPARATOR . $configId, $cmd, $paramObj, $optionParamObj);
				break;
		}

		// インナーウィジェットのエラー出力は呼び出しウィジェット側で受けるため、出力を抑止する
		ob_start();// バッファ作成

		// ウィジェット実行ファイル取得
		if (empty($widgetId)){		// ウィジェットIDが指定されていないときは共通ディレクトリ
		//$gEnvManager->getIWidgetsPath() . '/' . $iWidgetId . '/' . $containerClass . '.php';
		} else {
			if ($isAdmin){
				$widgetIndexFile = $gEnvManager->getWidgetsPath() . '/' . $widgetId . '/include/iwidgets/' . $iWidgetId . '/' . M3_DIR_NAME_ADMIN . '/index.php';
			} else {
				$widgetIndexFile = $gEnvManager->getWidgetsPath() . '/' . $widgetId . '/include/iwidgets/' . $iWidgetId . '/index.php';
			}
		}

		// 作業中のインナーウィジェットIDを設定
		$gEnvManager->setCurrentIWidgetId($id);		// インナーウィジェットを設定

		// インナーウィジェット定義IDを設定
		$gEnvManager->setCurrentIWidgetConfigId($configId);
		
		// ウィジェットを実行
		// ウィジェットの呼び出しは、複数回存在する可能性があるのでrequire_once()で呼び出さない
		if (file_exists($widgetIndexFile)){
			require($widgetIndexFile);
		} else {
			echo 'file not found error: ' . $widgetIndexFile;
		}
		
		// 作業中のウィジェットIDを解除
		$gEnvManager->setCurrentIWidgetId('');
	
		// インナーウィジェット定義IDを解除
		$gEnvManager->setCurrentIWidgetConfigId('');
		
		// 現在のバッファ内容を取得し、バッファを破棄
		$content = ob_get_contents();
		ob_end_clean();
		
		// 値を再取得
		if ($gCmdParamManager->getParam($id . M3_WIDGET_ID_SEPARATOR . $configId, $cmd, $obj, $option)){
			$paramObj = $obj;
			$optionParamObj = $option;
		}

		// 結果を取得
		if ($gCmdParamManager->getResult($id . M3_WIDGET_ID_SEPARATOR . $configId, $resObj)) $resultObj = $resObj;
		return true;
	}
	/**
	 * コールバックメソッドを実行し、出力を返す
	 *
	 * @param array $callback		実行するコールバック
	 * @return string				出力結果
	 */
	function getOutputByCallbackMethod($callback)
	{
		ob_start();// バッファ作成
		
		// コールバック関数を実行
		if (is_callable($callback)) call_user_func($callback);
		
		// 現在のバッファ内容を取得し、バッファを破棄
		$content = ob_get_contents();
		ob_end_clean();
		return $content;
	}
	/**
	 * 指定URLへリダイレクト
	 *
	 * @param string $url			遷移先URL。未指定の場合は現在のスクリプト。URLでないときは、現在のスクリプトに付加。
	 * @return 						なし
	 */
	function redirect($url = '')
	{
		global $gEnvManager;

		$toUrl = $url;
		if (empty($toUrl)){
			$toUrl = $gEnvManager->getCurrentScriptUrl();
		} else if (strncmp($toUrl, 'http://', 7) != 0 && strncmp($toUrl, 'https://', 8) != 0){	// URL以外のときはクエリー文字列と判断する
			$toUrl = $gEnvManager->getCurrentScriptUrl() . $toUrl;
		}
		
		ob_end_clean();
		header('HTTP/1.1 301 Moved Permanently');
		header('Location: ' . $toUrl);
//		exit();
	}
	/**
	 * 指定URLのディレクトリへリダイレクト
	 *
	 * @return 						なし
	 */
	function redirectToDirectory()
	{
		global $gEnvManager;

		// ファイル名を削除
		$dirPath = dirname($gEnvManager->getCurrentScriptUrl()) . '/';
		$this->redirect($dirPath);
	}
	/**
	 * インストール用URLへリダイレクト
	 *
	 * @return 						なし
	 */
	function redirectToInstall()
	{
		global $gEnvManager;
		
		$sytemRootUrl = $gEnvManager->calcSystemRootUrl();
		self::redirect($sytemRootUrl . '/admin/install.php');
	}
	/**
	 * ログイン、ログアウト処理を行った後、リダイレクト
	 *
	 * @param RequestManager	$request		HTTPリクエスト処理クラス
	 * @param bool				$success		ログインの成否
	 * @return int								0=何も処理しなかった場合、1=ログイン処理、2=ログアウト処理
	 */
	function standardLoginLogoutRedirect($request, &$success)
	{
		global $gAccessManager;
		
		// ログイン、ログアウト処理、および、コンテナの起動
		$cmd = $request->trimValueOf(M3_REQUEST_PARAM_OPERATION_COMMAND);
		if ($cmd == M3_REQUEST_CMD_LOGIN){			// ログインの場合
			// ユーザ認証
			if ($gAccessManager->userLogin($request) == true){
				// この画面にリダイレクト
				$this->redirect();
				$success = true;		// ログイン成功
			} else {	// ログイン失敗のとき
				$success = false;		// ログイン失敗
			}
			return 1;
		} else if ($cmd == M3_REQUEST_CMD_LOGOUT){		// ログアウト
			$gAccessManager->userLogout(true);
			
			// この画面にリダイレクト
			$this->redirect();
			return 2;
		} else {
			return 0;
		}
	}
	/**
	 * ウィジェットがセットされているページサブIDを取得
	 *
	 * @param string $pageId		ページID
	 * @param string $widgetId		対象ウィジェットID
	 * @return string				ページサブID
	 */
	function getPageSubIdByWidget($pageId, $widgetId)
	{
		$pageSubId = '';
		
		// 対象のウィジェットのページサブIDを取得
		$ret = $this->db->getSubPageId($widgetId, $pageId, $rows);
		if ($ret){// データが存在する
			if (!empty($rows[0]['pd_sub_id'])) $pageSubId = $rows[0]['pd_sub_id'];
		}
		return $pageSubId;
	}
	/**
	 * URLで示されるページ上のウィジェットをウィジェット種別で取得
	 *
	 * @param string $url			URL
	 * @param string $widgetType	対象ウィジェットタイプ
	 * @return string				ウィジェットID
	 */
	function getWidgetIdByWidgetType($url, $widgetType)
	{
		$ret = $this->getPageIdFromUrl($url, $pageId, $pageSubId, $param);
		
		// ウィジェットIDを取得
		if ($ret) $widgetId = $this->db->getWidgetIdByType($pageId, $pageSubId, $widgetType);
		return $widgetId;
	}
	/**
	 * URLを解析して、ページID,サブページIDを取得
	 *
	 * @param string $url			URL
	 * @param string $pageId		ページID
	 * @param string $pageSubId		ページサブID
	 * @param array  $urlParam		URLパラメータ
	 * @return bool					true=成功、false=失敗
	 */
	function getPageIdFromUrl($url, &$pageId, &$pageSubId, &$urlParam)
	{
		global $gEnvManager;
			
		// ページID、ページサブIDを求める
		list($page, $other) = explode('?', str_replace($gEnvManager->getRootUrl(), '', $url));
		parse_str($other, $urlParam);
		
		$pageId = str_replace('/', '_', trim($page, '/'));
		$pageId = basename($pageId, '.php');

		// ページサブIDを取得
		$pageSubId = $param['sub'];
		if (empty($pageSubId)) $pageSubId = $gEnvManager->getDefaultPageSubId();
		return true;
	}
	/**
	 * ウィジェットが組み込まれているページのURLを生成
	 *
	 * @param string $widgetId	送信元ウィジェットID
	 * @param string $param			実行パラメータ
	 * @return string				生成したURL
	 */
	function getDefaultPageUrlByWidget($widgetId, $param)
	{
		global $gEnvManager;

		// このウィジェットがマップされているページサブIDを取得
		$sub = '';
		$subPageId = $this->getPageSubIdByWidget($gEnvManager->getDefaultPageId(), $widgetId);
		if (empty($subPageId)){
			// 見つからないときは互換ウィジェットを探す
			$targetWidgetId = $this->db->getCompatibleWidgetId($widgetId);
			if (!empty($targetWidgetId)){
				$subPageId = $this->getPageSubIdByWidget($gEnvManager->getDefaultPageId(), $targetWidgetId);
			}
		}
		if (!empty($subPageId)) $sub = 'sub=' . $subPageId . '&';
		return $gEnvManager->getDefaultUrl() . '?' . $sub . $param;
	}
	/**
	 * ウィジェット指定呼び出し用のURLを生成
	 *
	 * @param string $pageId		ページID(空のときは現在のURL)
	 * @param string $toWidgetId	送信先ウィジェットID
	 * @param string $fromWidgetId	送信元ウィジェットID
	 * @param string $todo			実行パラメータ
	 * @return string				生成したURL
	 */
	function createWidgetCmdUrl($pageId, $toWidgetId, $fromWidgetId, $todo)
	{
		global $gEnvManager;
		
		$url = '';
		if (empty($pageId)){	
			$url .= $gEnvManager->getCurrentScriptUrl();
		} else {
			// URLを設定
			$pathArray = explode('_', $pageId);
			for ($i = 0; $i < count($pathArray); $i++){
				$path .= '/' . $pathArray[$i];
			}
			$url = $gEnvManager->getRootUrl() . $path . '.php';
		}
		
		$url .= '?' . M3_REQUEST_PARAM_OPERATION_COMMAND . '=' . M3_REQUEST_CMD_FIND_WIDGET;
		$url .= '&' . M3_REQUEST_PARAM_WIDGET_ID . '=' . $toWidgetId;
		if (!empty($fromWidgetId)) $url .= '&' . M3_REQUEST_PARAM_FROM . '=' . $fromWidgetId;		// 送信元
		if (!empty($todo)) $url .= '&' . M3_REQUEST_PARAM_OPERATION_TODO . '=' . urlencode($todo);
		return $url;
	}
	/**
	 * キャッシュ制御用の付加パラメータを取得
	 *
	 * @return string				パラメータ
	 */	
	function getCacheParam()
	{
		$addParam = '?' . date('YmdHis');
		return $addParam;
	}
	/**
	 * 現在のURLにアクセス可能かチェック
	 *
	 * @return bool				true=可能、false=不可
	 */	
	function canAccessUrl()
	{
		global $gEnvManager;
		
		// 現在設定されているページIDを取得
		$pageId		= $gEnvManager->getCurrentPageId();
		$pageSubId	= $gEnvManager->getCurrentPageSubId();
		return $this->db->canAccessPage($pageId, $pageSubId);
	}
}
?>
