<?php
/**
 * 各種テキスト変換マネージャー
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2008 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: textConvManager.php 1326 2008-12-07 10:03:12Z fishbone $
 * @link       http://www.magic3.org
 */
require_once(M3_SYSTEM_INCLUDE_PATH . '/common/htmlEdit.php');
require_once(M3_SYSTEM_INCLUDE_PATH . '/data/emojiMap.php');// 絵文字マップ情報
require_once(M3_SYSTEM_INCLUDE_PATH . '/common/core.php');

class TextConvManager extends Core
{
	private $db;						// DBオブジェクト
	
	/**
	 * コンストラクタ
	 */
	function __construct()
	{
		//global $gInstanceManager;
		// 親クラスを呼び出す
		parent::__construct();
		
		// システムDBオブジェクト取得
		$this->db = $this->gInstance->getSytemDbObject();
	}
	/**
	 * キー値テーブルの値を使用してテキストを変換
	 *
	 * @param string $src		変換するデータ
	 * @param string $dest      変換後データ
	 * @param bool $convBr      キーワード変換部分の改行コードをBRタグに変換するかどうか
	 * @return bool				true=成功、false=失敗
	 */
	function convByKeyValue($src, &$dest, $convBr=false)
	{
		// データをコピー
		$dest = $src;

		// キーワードを取得
		$keywords = array();
		$matches = array();
		$pattern = '/(' . preg_quote(M3_TAG_START) . '([A-Z0-9_]+)' . preg_quote(M3_TAG_END) . ')/u';
		preg_match_all($pattern, $src, $matches, PREG_SET_ORDER);
		for ($i = 0; $i < count($matches); $i++){
			$value = $matches[$i][2];
			if (!in_array($value, $keywords)) $keywords[] = $value;
		}
		// キーワードを変換
		for ($i = 0; $i < count($keywords); $i++){
			$key = $keywords[$i];
			$value = $this->db->getKeyValue($key, $tmp);
			if ($convBr){// 改行コード変換の場合
				$value = HtmlEdit::convLineBreakToBr($value);
			}
			$dest = str_replace(M3_TAG_START . $key . M3_TAG_END, $value, $dest);
		}
		return true;
	}
	/**
	 * 絵文字画像をMagic3絵文字タグに変換
	 *
	 * @param string $src		変換するデータ
	 * @param string $dest      変換後データ
	 * @return bool				true=成功、false=失敗
	 */
	function convToEmojiTag($src, &$dest)
	{
		global $gEnvManager;
		
		$emojiUrl = $gEnvManager->getEmojiImagesUrl();
		$emojiUrl = str_replace('/', '\/', $emojiUrl);		// 「/」を正規表現用に「\/」に変換
		
		$str = '/<img[^<]*?src="' . $emojiUrl . '\/.*?\.gif\?code=(\d+?)"[^>]*?\/>/';
        $dest = preg_replace_callback($str, array($this, "_replace_to_emoji_tag_callback"), $src);
		return true;
	}
	/**
	 * Magic3絵文字タグ変換コールバック関数
	 *
	 * @param array $matchData		検索マッチデータ
	 * @return string				変換後データ
	 */
    function _replace_to_emoji_tag_callback($matchData)
	{
		global $EMOJI;		// 絵文字マップ情報
		
		// 絵文字コードよりMagic3絵文字タグを作成
		$code = intval($matchData[1]);		// 絵文字コード
		if ($code < 0 || count($EMOJI) <= $code) return '';		// エラーチェック
		
        $code++;		// 絵文字コード(内部コードは1以上)
		$tag = M3_TAG_START . M3_TAG_MACRO_EMOJI_CODE . ':' . $code . M3_TAG_END;
		return $tag;
    }
	/**
	 * Magic3絵文字タグを絵文字画像に変換
	 *
	 * @param string $src		変換するデータ
	 * @param string $dest      変換後データ
	 * @return bool				true=成功、false=失敗
	 */
	function convFromEmojiTag($src, &$dest)
	{
		$startTag = str_replace('[', '\[', M3_TAG_START . M3_TAG_MACRO_EMOJI_CODE);		// 「[」を正規表現用に「\[」に変換
		$endTag = str_replace(']', '\]', M3_TAG_END);		// 「[」を正規表現用に「\[」に変換
		$str = '/' . $startTag . ':(\d+?)' . $endTag . '/';
        $dest = preg_replace_callback($str, array($this, "_replace_from_emoji_tag_callback"), $src);
		return true;
	}
	/**
	 * Magic3絵文字タグ変換コールバック関数
	 *
	 * @param array $matchData		検索マッチデータ
	 * @return string				変換後データ
	 */
    function _replace_from_emoji_tag_callback($matchData)
	{
		global $gEnvManager;
		global $EMOJI;		// 絵文字マップ情報
		
		$emojiUrl = $gEnvManager->getEmojiImagesUrl();
				
		// 絵文字コードよりMagic3絵文字タグを作成
		// 配列のインデックスは、「絵文字コード-1」で指定
        $index = intval($matchData[1]) -1;		// 絵文字コード
		if ($index < 0 || count($EMOJI) <= $index) return '';// エラーチェック
		
		$emojiImageUrl = $emojiUrl . '/' . $EMOJI[$index]['filename'] . '.gif?code=' . $index;
		$tag = '<img src="' . $emojiImageUrl . '" />';
		return $tag;
    }
	/**
	 * Magic3絵文字タグを携帯端末に合わせて変換
	 *
	 * @param string $src		変換するデータ
	 * @param string $dest      変換後データ
	 * @return bool				true=成功、false=失敗
	 */
	function convEmoji($src, &$dest)
	{
		$startTag = str_replace('[', '\[', M3_TAG_START . M3_TAG_MACRO_EMOJI_CODE);		// 「[」を正規表現用に「\[」に変換
		$endTag = str_replace(']', '\]', M3_TAG_END);		// 「[」を正規表現用に「\[」に変換
		$str = '/' . $startTag . ':(\d+?)' . $endTag . '/';
        $dest = preg_replace_callback($str, array($this, "_replace_emoji_callback"), $src);
		return true;
	}
	/**
	 * Magic3絵文字タグ変換コールバック関数
	 *
	 * @param array $matchData		検索マッチデータ
	 * @return string				変換後データ
	 */
    function _replace_emoji_callback($matchData)
	{
		global $gEnvManager;
		global $gInstanceManager;
		global $EMOJI;		// 絵文字マップ情報
		
		// 絵文字変換ライブラリ読み込み
		require_once($gEnvManager->getLibPath() . '/MobilePictogramConverter-1.2.0/MobilePictogramConverter.php');
		
		$agent = $gInstanceManager->getMobileAgent();
		$emojiUrl = $gEnvManager->getEmojiImagesUrl();
				
		// 絵文字コードよりMagic3絵文字タグを作成
		// 配列のインデックスは、「絵文字コード-1」で指定
        $index = intval($matchData[1]) -1;		// 絵文字コード
		if ($index < 0 || count($EMOJI) <= $index) return '';// エラーチェック
		
		$emojiData = pack('H*', $EMOJI[$index]['i']);		// imodeSJISデータ
		$mpc = MobilePictogramConverter::factory($emojiData, MPC_FROM_FOMA, MPC_FROM_CHARSET_SJIS);

		if ($agent->isDoCoMo()){	// ドコモ端末のとき
			$dest = $mpc->Convert(MPC_TO_FOMA, MPC_TO_OPTION_RAW);
		} else if ($agent->isEZweb()){	// au端末のとき
			$dest = $mpc->Convert(MPC_TO_EZWEB, MPC_TO_OPTION_RAW);
		} else if ($agent->isVodafone()){	// ソフトバンク端末のとき
			$dest = $mpc->Convert(MPC_TO_SOFTBANK, MPC_TO_OPTION_RAW);
		} else {		// その他の端末のとき
			$emojiImageUrl = $emojiUrl . '/' . $EMOJI[$index]['filename'] . '.gif?code=' . $index;
			$dest = '<img src="' . $emojiImageUrl . '" />';
		}
		return $dest;
    }
	
	/**
	 * HTMLをテキストに変換
	 *
	 * @param string $src		変換するデータ
	 * @return string			変換後データ
	 */
	function htmlToText($src)
	{
		$search = array("'<script[^>]*?>.*?</script>'si",	// javascriptを削除
						"'<[\/\!]*?[^<>]*?>'si",  // htmlタグを削除
						"'([\r\n])[\s]+'",  // 空白文字を削除
						"'&(quot|#34);'i",  // htmlエンティティを置換
						"'&(amp|#38);'i",
						"'&(lt|#60);'i",
						"'&(gt|#62);'i",
						"'&(nbsp|#160);'i",
						"'&(iexcl|#161);'i",
						"'&(cent|#162);'i",
						"'&(pound|#163);'i",
						"'&(copy|#169);'i",
						"'&#(\d+);'e");  // phpとして評価

		$replace = array("",
							"",
							"\\1",
							"\"",
							"&",
							"<",
							">",
							" ",
							chr(161),
							chr(162),
							chr(163),
							chr(169),
							"chr(\\1)");

		return preg_replace($search, $replace, $src);
	}
}
?>
