<?php
/**
 * DBクラス
 *
 * PHP versions 5
 *
 * LICENSE: This source file is licensed under the terms of the GNU General Public License.
 *
 * @package    Magic3 Framework
 * @author     平田直毅(Naoki Hirata) <naoki@aplo.co.jp>
 * @copyright  Copyright 2006-2007 Magic3 Project.
 * @license    http://www.gnu.org/copyleft/gpl.html  GPL License
 * @version    SVN: $Id: bbs_categoryDb.php 2 2007-11-03 04:59:01Z fishbone $
 * @link       http://www.magic3.org
 */
require_once($gEnvManager->getDbPath() . '/baseDb.php');

class bbs_categoryDb extends BaseDb
{
	/**
	 * すべての言語を取得
	 *
	 * @param function	$callback			コールバック関数
	 * @return			true=取得、false=取得せず
	 */
	function getAllLang($callback)
	{
		$queryStr = 'SELECT * FROM _language ORDER BY ln_priority';
		$this->selectLoop($queryStr, array(), $callback, null);
	}
	/**
	 * BBSカテゴリー一覧を取得
	 *
	 * @param function	$callback			コールバック関数
	 * @param string	$lang				言語
	 * @return 			なし
	 */
	function getAllCategory($callback, $lang)
	{
		$queryStr = 'SELECT * FROM bbs_category LEFT JOIN _login_user ON sr_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=  'WHERE sr_language_id = ? ';
		$queryStr .=    'AND sr_deleted = false ';		// 削除されていない
		$queryStr .=  'ORDER BY sr_sort_order';
		$this->selectLoop($queryStr, array($lang), $callback, null);
	}
	/**
	 * BBSカテゴリーの対応言語を取得
	 *
	 * @param int		$id			カテゴリーID
	 * @return bool					true=取得、false=取得せず
	 */
	function getLangByCategoryId($id, &$rows)
	{
		$queryStr = 'SELECT ln_id, ln_name, ln_name_en FROM bbs_category LEFT JOIN _language ON sr_language_id = ln_id ';
		$queryStr .=  'WHERE sr_deleted = false ';	// 削除されていない
		$queryStr .=    'AND sr_id = ? ';
		$queryStr .=  'ORDER BY sr_id, ln_priority';
		$retValue = $this->selectRecords($queryStr, array($id), $rows);
		return $retValue;
	}
	/**
	 * BBSカテゴリーの新規追加
	 *
	 * @param int	  $id	カテゴリーID
	 * @param string  $lang			言語ID
	 * @param string  $name			名前
	 * @param int     $index		表示順
	 * @param bool    $visible		表示、非表示
	 * @param int     $userId		更新者ユーザID
	 * @param int     $newSerial	新規シリアル番号
	 * @return bool					true = 成功、false = 失敗
	 */
	function addCategory($id, $lang, $name, $index, $visible, $userId, &$newSerial)
	{	
		// トランザクション開始
		$this->startTransaction();
		
		if ($id == 0){		// IDが0のときは、カテゴリーIDを新規取得
			// コンテンツIDを決定する
			$queryStr = 'select max(sr_id) as mid from bbs_category ';
			$ret = $this->selectRecord($queryStr, array(), $row);
			if ($ret){
				$cId = $row['mid'] + 1;
			} else {
				$cId = 1;
			}
		} else {
			$cId = $id;
		}
		
		// 前レコードの削除状態チェック
		$historyIndex = 0;
		$queryStr = 'SELECT * FROM bbs_category ';
		$queryStr .=  'WHERE sr_id = ? ';
		$queryStr .=    'AND sr_language_id = ? ';
		$queryStr .=  'ORDER BY sr_history_index DESC ';
		$queryStr .=    'LIMIT 1';
		$ret = $this->selectRecord($queryStr, array($cId, $lang), $row);
		if ($ret){
			if (!$row['sr_deleted']){		// レコード存在していれば終了
				$this->endTransaction();
				return false;
			}
			$historyIndex = $row['sr_history_index'] + 1;
		}
		
		// データを追加
		$queryStr = 'INSERT INTO bbs_category ';
		$queryStr .=  '(sr_id, sr_language_id, sr_history_index, sr_name, sr_sort_order, sr_visible, sr_create_user_id, sr_create_dt) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, ?, ?, ?, ?, now())';
		$this->execStatement($queryStr, array($cId, $lang, $historyIndex, $name, $index, $visible, $userId));
		
		// 新規のシリアル番号取得
		$queryStr = 'select max(sr_serial) as ns from bbs_category ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret) $newSerial = $row['ns'];
			
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * スレッドカテゴリーをシリアル番号で取得
	 *
	 * @param int		$serial				シリアル番号
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getCategoryBySerial($serial, &$row)
	{
		$queryStr  = 'select * from bbs_category LEFT JOIN _login_user ON sr_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=   'WHERE sr_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		return $ret;
	}	
	/**
	 * スレッドカテゴリーをカテゴリーIDで取得
	 *
	 * @param int		$id					カテゴリーID
	 * @param string	$langId				言語ID
	 * @param array     $row				レコード
	 * @return bool							取得 = true, 取得なし= false
	 */
	function getCategoryByCategoryId($id, $langId, &$row)
	{
		$queryStr  = 'SELECT * FROM bbs_category LEFT JOIN _login_user ON sr_create_user_id = lu_id AND lu_deleted = false ';
		$queryStr .=   'WHERE sr_deleted = false ';	// 削除されていない
		$queryStr .=   'AND sr_id = ? ';
		$queryStr .=   'AND sr_language_id = ? ';
		$ret = $this->selectRecord($queryStr, array($id, $langId), $row);
		return $ret;
	}
	/**
	 * スレッドカテゴリーの更新
	 *
	 * @param int     $serial		シリアル番号
	 * @param string  $name			名前
	 * @param int     $index		表示順
	 * @param bool    $visible		表示、非表示
	 * @param int     $userId		更新者ユーザID
	 * @param int     $newSerial	新規シリアル番号
	 * @return bool					true = 成功、false = 失敗
	 */
	function updateCategory($serial, $name, $index, $visible, $userId, &$newSerial)
	{	
		$now = date("Y/m/d H:i:s");	// 現在日時
				
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のシリアルNoのレコードが削除状態でないかチェック
		$historyIndex = 0;		// 履歴番号
		$queryStr  = 'select * from bbs_category ';
		$queryStr .=   'where sr_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		if ($ret){		// 既に登録レコードがあるとき
			if ($row['sr_deleted']){		// レコードが削除されていれば終了
				$this->endTransaction();
				return false;
			}
			$historyIndex = $row['sr_history_index'] + 1;
		} else {		// 存在しない場合は終了
			$this->endTransaction();
			return false;
		}
		
		// 古いレコードを削除
		$queryStr  = 'UPDATE bbs_category ';
		$queryStr .=   'SET sr_deleted = true, ';	// 削除
		$queryStr .=     'sr_update_user_id = ?, ';
		$queryStr .=     'sr_update_dt = ? ';
		$queryStr .=   'WHERE sr_serial = ?';
		$this->execStatement($queryStr, array($userId, $now, $serial));
		
		// 新規レコード追加
		$queryStr = 'INSERT INTO bbs_category ';
		$queryStr .=  '(sr_id, sr_language_id, sr_history_index, sr_name, sr_sort_order, sr_visible, sr_create_user_id, sr_create_dt) ';
		$queryStr .=  'VALUES ';
		$queryStr .=  '(?, ?, ?, ?, ?, ?, ?, ?)';
		$this->execStatement($queryStr, array($row['sr_id'], $row['sr_language_id'], $historyIndex, $name, $index, $visible, $userId, $now));

		// 新規のシリアル番号取得
		$queryStr = 'select max(sr_serial) as ns from bbs_category ';
		$ret = $this->selectRecord($queryStr, array(), $row);
		if ($ret) $newSerial = $row['ns'];
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * カテゴリーの削除
	 *
	 * @param int $serialNo			シリアルNo
	 * @param int $userId			ユーザID(データ更新者)
	 * @return						true=成功、false=失敗
	 */
	function delCategory($serialNo, $userId)
	{
		// トランザクション開始
		$this->startTransaction();
		
		// 指定のシリアルNoのレコードが削除状態でないかチェック
		$queryStr  = 'select * from bbs_category ';
		$queryStr .=   'where sr_deleted = false ';		// 未削除
		$queryStr .=     'and sr_serial = ? ';
		$ret = $this->isRecordExists($queryStr, array($serialNo));
		// 存在しない場合は、既に削除されたとして終了
		if (!$ret){
			$this->endTransaction();
			return false;
		}
		
		// レコードを削除
		$queryStr  = 'UPDATE bbs_category ';
		$queryStr .=   'SET sr_deleted = true, ';	// 削除
		$queryStr .=     'sr_update_user_id = ?, ';
		$queryStr .=     'sr_update_dt = now() ';
		$queryStr .=   'WHERE sr_serial = ?';
		$this->execStatement($queryStr, array($userId, $serialNo));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
	/**
	 * カテゴリーIDで削除
	 *
	 * @param int $serial			シリアルNo
	 * @param int $userId			ユーザID(データ更新者)
	 * @return						true=成功、false=失敗
	 */
	function delCategoryById($serial, $userId)
	{
		// トランザクション開始
		$this->startTransaction();
		
		// コンテンツIDを取得
		$queryStr  = 'select * from bbs_category ';
		$queryStr .=   'where sr_deleted = false ';		// 未削除
		$queryStr .=     'and sr_serial = ? ';
		$ret = $this->selectRecord($queryStr, array($serial), $row);
		if ($ret){		// 既に登録レコードがあるとき
			if ($row['sr_deleted']){		// レコードが削除されていれば終了
				$this->endTransaction();
				return false;
			}
		} else {		// 存在しない場合は終了
			$this->endTransaction();
			return false;
		}
		$contId = $row['sr_id'];
		
		// レコードを削除
		$queryStr  = 'UPDATE bbs_category ';
		$queryStr .=   'SET sr_deleted = true, ';	// 削除
		$queryStr .=     'sr_update_user_id = ?, ';
		$queryStr .=     'sr_update_dt = now() ';
		$queryStr .=   'WHERE sr_id = ?';
		$this->execStatement($queryStr, array($userId, $contId));
		
		// トランザクション確定
		$ret = $this->endTransaction();
		return $ret;
	}
}
?>
