-- --------------------------------------------------------------------------------------------------
-- 標準テーブル作成スクリプト
-- システムの標準構成で必要なテーブルの作成、初期データの登録を行う
-- --------------------------------------------------------------------------------------------------

-- 国マスター
DROP TABLE IF EXISTS country;
CREATE TABLE country (
    ct_id                VARCHAR(3)     DEFAULT ''                    NOT NULL,      -- 国ID(ISO 3文字コード)
    ct_language_id       VARCHAR(2)     DEFAULT ''                    NOT NULL,      -- 言語ID
    ct_name              VARCHAR(20)    DEFAULT ''                    NOT NULL,      -- 国名称
    ct_name_short        VARCHAR(15)    DEFAULT ''                    NOT NULL,      -- 国名称略称
    ct_iso_code_2        VARCHAR(2)     DEFAULT ''                    NOT NULL,      -- ISO 2文字コード
    ct_index             INT            DEFAULT 0                     NOT NULL,      -- ソート用
    PRIMARY KEY          (ct_id,        ct_language_id)
) TYPE=innodb;
INSERT INTO country (ct_id, ct_language_id, ct_name, ct_name_short, ct_iso_code_2, ct_index) VALUES ('JPN', 'ja', '日本',           '日本',     'JP', 0);
INSERT INTO country (ct_id, ct_language_id, ct_name, ct_name_short, ct_iso_code_2, ct_index) VALUES ('USA', 'ja', 'アメリカ合衆国', 'アメリカ', 'US', 1);
INSERT INTO country (ct_id, ct_language_id, ct_name, ct_name_short, ct_iso_code_2, ct_index) VALUES ('GBR', 'ja', 'イギリス',       'イギリス', 'GB', 2);
INSERT INTO country (ct_id, ct_language_id, ct_name, ct_name_short, ct_iso_code_2, ct_index) VALUES ('DEU', 'ja', 'ドイツ',         'ドイツ',   'DE', 3);
INSERT INTO country (ct_id, ct_language_id, ct_name, ct_name_short, ct_iso_code_2, ct_index) VALUES ('FRA', 'ja', 'フランス',       'フランス', 'FR', 4);
INSERT INTO country (ct_id, ct_language_id, ct_name, ct_name_short, ct_iso_code_2, ct_index) VALUES ('CHN', 'ja', '中華人民共和国', '中国',     'CN', 5);
INSERT INTO country (ct_id, ct_language_id, ct_name, ct_name_short, ct_iso_code_2, ct_index) VALUES ('KOR', 'ja', '大韓民国',       '韓国',     'KR', 6);

-- 通貨マスター
DROP TABLE IF EXISTS currency;
CREATE TABLE currency (
    cu_id                VARCHAR(3)     DEFAULT ''                    NOT NULL,      -- 通貨ID
    cu_language_id       VARCHAR(2)     DEFAULT ''                    NOT NULL,      -- 言語ID
    cu_name              VARCHAR(20)    DEFAULT ''                    NOT NULL,      -- 名称
    cu_description       VARCHAR(80)    DEFAULT ''                    NOT NULL,      -- 説明
    cu_symbol            VARCHAR(20)    DEFAULT ''                    NOT NULL,      -- 表示記号
    cu_post_symbol       VARCHAR(20)    DEFAULT ''                    NOT NULL,      -- 後表示記号
    cu_decimal_place     INT            DEFAULT 0                     NOT NULL,      -- 小数以下桁数
    cu_index             INT            DEFAULT 0                     NOT NULL,      -- ソート用
    PRIMARY KEY          (cu_id,        cu_language_id)
) TYPE=innodb;
INSERT INTO currency (cu_id, cu_language_id, cu_name, cu_description, cu_symbol, cu_post_symbol, cu_decimal_place, cu_index) VALUES ('JPY', 'ja', '円',         '日本円',               '￥', '円',         0, 1);
INSERT INTO currency (cu_id, cu_language_id, cu_name, cu_description, cu_symbol, cu_post_symbol, cu_decimal_place, cu_index) VALUES ('USD', 'ja', 'ドル',       'アメリカドル',         '$',  '',           2, 2);
INSERT INTO currency (cu_id, cu_language_id, cu_name, cu_description, cu_symbol, cu_post_symbol, cu_decimal_place, cu_index) VALUES ('EUR', 'ja', 'ユーロ',     '欧州ユーロ',           '',   '',           2, 3);
INSERT INTO currency (cu_id, cu_language_id, cu_name, cu_description, cu_symbol, cu_post_symbol, cu_decimal_place, cu_index) VALUES ('GBP', 'ja', 'ポンド',     'イギリスポンド',       '￡', '',           2, 4);
INSERT INTO currency (cu_id, cu_language_id, cu_name, cu_description, cu_symbol, cu_post_symbol, cu_decimal_place, cu_index) VALUES ('CNY', 'ja', '元',         '中国人民元',           '',   '元',         0, 5);
INSERT INTO currency (cu_id, cu_language_id, cu_name, cu_description, cu_symbol, cu_post_symbol, cu_decimal_place, cu_index) VALUES ('KRW', 'ja', 'ウォン',     '韓国ウォン',           '',   'ウォン',     0, 6);
INSERT INTO currency (cu_id, cu_language_id, cu_name, cu_description, cu_symbol, cu_post_symbol, cu_decimal_place, cu_index) VALUES ('TWD', 'ja', 'ドル',       '台湾ドル',             '',   'ドル',       2, 7);
INSERT INTO currency (cu_id, cu_language_id, cu_name, cu_description, cu_symbol, cu_post_symbol, cu_decimal_place, cu_index) VALUES ('THB', 'ja', 'バーツ',     'タイバーツ',           '',   'バーツ',     0, 8);
INSERT INTO currency (cu_id, cu_language_id, cu_name, cu_description, cu_symbol, cu_post_symbol, cu_decimal_place, cu_index) VALUES ('CAD', 'ja', 'ドル',       'カナダドル',           '',   'ドル',       2, 9);
INSERT INTO currency (cu_id, cu_language_id, cu_name, cu_description, cu_symbol, cu_post_symbol, cu_decimal_place, cu_index) VALUES ('SGD', 'ja', 'ドル',       'シンガポールドル',     '',   'ドル',       2, 10);
INSERT INTO currency (cu_id, cu_language_id, cu_name, cu_description, cu_symbol, cu_post_symbol, cu_decimal_place, cu_index) VALUES ('MYR', 'ja', 'リンギット', 'マレーシアリンギット', '',   'リンギット', 0, 11);
INSERT INTO currency (cu_id, cu_language_id, cu_name, cu_description, cu_symbol, cu_post_symbol, cu_decimal_place, cu_index) VALUES ('CHF', 'ja', 'フラン',     'スイスフラン',         '',   'フラン',     0, 12);
INSERT INTO currency (cu_id, cu_language_id, cu_name, cu_description, cu_symbol, cu_post_symbol, cu_decimal_place, cu_index) VALUES ('IDR', 'ja', 'ルピア',     'インドネシアルピア',   '',   'ルピア',     0, 13);
INSERT INTO currency (cu_id, cu_language_id, cu_name, cu_description, cu_symbol, cu_post_symbol, cu_decimal_place, cu_index) VALUES ('INR', 'ja', 'ルピー',     'インドルピー',         '',   'ルピー',     0, 14);
INSERT INTO currency (cu_id, cu_language_id, cu_name, cu_description, cu_symbol, cu_post_symbol, cu_decimal_place, cu_index) VALUES ('PHP', 'ja', 'ペソ',       'フィリピンペソ',       '',   'ペソ',       0, 15);
INSERT INTO currency (cu_id, cu_language_id, cu_name, cu_description, cu_symbol, cu_post_symbol, cu_decimal_place, cu_index) VALUES ('NZD', 'ja', 'ドル',       'ニュージーランドドル', '',   'ドル',       2, 16);
INSERT INTO currency (cu_id, cu_language_id, cu_name, cu_description, cu_symbol, cu_post_symbol, cu_decimal_place, cu_index) VALUES ('HKD', 'ja', 'ドル',       '香港ドル',             '',   'ドル',       2, 17);
INSERT INTO currency (cu_id, cu_language_id, cu_name, cu_description, cu_symbol, cu_post_symbol, cu_decimal_place, cu_index) VALUES ('TRL', 'ja', 'リラ',       'トルコリラ',           '',   'リラ',       0, 18);

-- メニューマスター
DROP TABLE IF EXISTS menu;
CREATE TABLE menu (
    me_serial            INT            AUTO_INCREMENT,                              -- レコードシリアル番号
    me_id                VARCHAR(20)    DEFAULT ''                    NOT NULL,      -- メニューID
    me_language_id       VARCHAR(2)     DEFAULT ''                    NOT NULL,      -- 言語ID
    me_name              VARCHAR(30)    DEFAULT ''                    NOT NULL,      -- 名前
    me_type              INT            DEFAULT 0                     NOT NULL,      -- メニューのタイプ(0=テーブル,1=リスト)
    me_show_name         BOOLEAN        DEFAULT true                  NOT NULL,      -- 名前を表示するかどうか
    me_params            TEXT           DEFAULT ''                    NOT NULL,      -- 各種パラメータ
    me_style             TEXT           DEFAULT ''                    NOT NULL,      -- HTMLスタイル属性

    me_checked_out       BOOLEAN        DEFAULT false                 NOT NULL,      -- チェックアウト中かどうか
    me_checked_out_dt    TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- チェックアウト日時
    me_update_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード更新者
    me_update_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード更新日時
    PRIMARY KEY          (me_serial),
    UNIQUE               (me_id,        me_language_id)
) TYPE=innodb;

-- メニュー項目マスター
DROP TABLE IF EXISTS menu_item;
CREATE TABLE menu_item (
    mi_serial            INT            AUTO_INCREMENT,                              -- レコードシリアル番号
    mi_id                INT            DEFAULT 0                     NOT NULL,      -- 項目ID
    mi_language_id       VARCHAR(2)     DEFAULT ''                    NOT NULL,      -- 言語ID
    mi_menu_id           VARCHAR(20)    DEFAULT ''                    NOT NULL,      -- 項目が所属するメニューID
    mi_index             INT            DEFAULT 0                     NOT NULL,      -- メニュー上での項目番号(0～)
    mi_submenu_id        VARCHAR(20)    DEFAULT ''                    NOT NULL,      -- サブメニューID
    mi_name              VARCHAR(30)    DEFAULT ''                    NOT NULL,      -- 名前
    mi_type              INT            DEFAULT 0                     NOT NULL,      -- 項目タイプ(0=通常リンク、1=JavaScript)
    mi_link_type         INT            DEFAULT 0                     NOT NULL,      -- リンクタイプ(0=同ウィンドウ、1=別ウィンドウ)
    mi_link_url          TEXT           DEFAULT ''                    NOT NULL,      -- リンク先
    mi_param             TEXT           DEFAULT ''                    NOT NULL,      -- その他パラメータ
    mi_style             TEXT           DEFAULT ''                    NOT NULL,      -- CSS
    mi_select_param      TEXT           DEFAULT ''                    NOT NULL,      -- 選択状態を判断するためのパラメータ
    mi_visible           BOOLEAN        DEFAULT true                  NOT NULL,      -- 表示するかどうか
    mi_enable            BOOLEAN        DEFAULT true                  NOT NULL,      -- 使用可能かどうか
    mi_show_name         BOOLEAN        DEFAULT true                  NOT NULL,      -- 名前を表示するかどうか
    mi_access_id         INT            DEFAULT 0                     NOT NULL,      -- アクセス頻度調査用

    mi_update_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード更新者
    mi_update_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード更新日時
    PRIMARY KEY          (mi_serial),
    UNIQUE               (mi_id,        mi_language_id)
) TYPE=innodb;

-- コンテンツマスター
DROP TABLE IF EXISTS content;
CREATE TABLE content (
    cn_serial            INT            AUTO_INCREMENT,                              -- レコードシリアル番号
    cn_type              VARCHAR(20)    DEFAULT ''                    NOT NULL,      -- コンテンツタイプ
    cn_id                INT            DEFAULT 0                     NOT NULL,      -- コンテンツID
    cn_language_id       VARCHAR(2)     DEFAULT ''                    NOT NULL,      -- 言語ID
    cn_history_index     INT            DEFAULT 0                     NOT NULL,      -- 履歴管理用インデックスNo(0～)
    cn_name              VARCHAR(40)    DEFAULT ''                    NOT NULL,      -- コンテンツ名
    cn_description       VARCHAR(100)   DEFAULT ''                    NOT NULL,      -- 説明
    cn_html              TEXT           DEFAULT ''                    NOT NULL,      -- コンテンツHTML
    cn_disp_type         SMALLINT       DEFAULT 0                     NOT NULL,      -- 表示タイプ(0=プレーン、1=インナーフレーム)
    cn_visible           BOOLEAN        DEFAULT true                  NOT NULL,      -- 表示可否
    cn_active_start_dt   TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- 表示可能期間(開始)
    cn_active_end_dt     TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- 表示可能期間(終了)
    cn_default           BOOLEAN        DEFAULT false                 NOT NULL,      -- デフォルトフラグ
    cn_key               VARCHAR(20)    DEFAULT ''                    NOT NULL,      -- 外部からの参照用キー
    
    cn_check_out_user_id INT            DEFAULT 0                     NOT NULL,      -- チェックアウトユーザID(0のときはチェックイン状態)
    cn_check_out_dt      TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- チェックアウト日時
    cn_create_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード作成者
    cn_create_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード作成日時
    cn_update_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード更新者
    cn_update_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード更新日時
    cn_deleted           BOOLEAN        DEFAULT false                 NOT NULL,      -- レコード削除状態
    PRIMARY KEY          (cn_serial),
    UNIQUE               (cn_type,      cn_id,        cn_language_id,               cn_history_index)
) TYPE=innodb;

-- ブログ設定マスター
DROP TABLE IF EXISTS blog_config;
CREATE TABLE blog_config (
    bg_id                VARCHAR(30)    DEFAULT ''                    NOT NULL,      -- ID(Key)
    bg_value             TEXT           DEFAULT ''                    NOT NULL,      -- 値
    bg_name              VARCHAR(50)    DEFAULT ''                    NOT NULL,      -- 名称
    bg_description       VARCHAR(80)    DEFAULT ''                    NOT NULL,      -- 説明
    bg_index             INT            DEFAULT 0                     NOT NULL,      -- ソート用
    PRIMARY KEY          (bg_id)
) TYPE=innodb;
INSERT INTO blog_config (bg_id,              bg_value,    bg_name,                     bg_index)
VALUES                 ('receive_comment',   '0',         'コメントを受け付け',        1);
INSERT INTO blog_config (bg_id,              bg_value,    bg_name,                     bg_index)
VALUES                 ('receive_trackback', '0',         'トラックバックの受け付け',  2);
INSERT INTO blog_config (bg_id,              bg_value,    bg_name,                     bg_index)
VALUES                 ('entry_view_count',  '10',        '記事表示数',                3);
INSERT INTO blog_config (bg_id,              bg_value,    bg_name,                     bg_index)
VALUES                 ('entry_view_order',  '1',        '記事表示順',                 4);

-- ブログカテゴリマスター
DROP TABLE IF EXISTS blog_category;
CREATE TABLE blog_category (
    bc_serial            INT            AUTO_INCREMENT,                              -- レコードシリアル番号
    bc_id                INT            DEFAULT 0                     NOT NULL,      -- カテゴリID
    bc_language_id       VARCHAR(2)     DEFAULT ''                    NOT NULL,      -- 言語ID
    bc_history_index     INT            DEFAULT 0                     NOT NULL,      -- 履歴管理用インデックスNo(0～)

    bc_name              VARCHAR(30)    DEFAULT ''                    NOT NULL,      -- カテゴリ名称
    bc_parent_id         INT            DEFAULT 0                     NOT NULL,      -- 親カテゴリID
    bc_sort_order        INT            DEFAULT 0                     NOT NULL,      -- ソート用
    bc_visible           BOOLEAN        DEFAULT true                  NOT NULL,      -- 表示するかどうか

    bc_create_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード作成者
    bc_create_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード作成日時
    bc_update_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード更新者
    bc_update_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード更新日時
    bc_deleted           BOOLEAN        DEFAULT false                 NOT NULL,      -- レコード削除状態
    PRIMARY KEY          (bc_serial),
    UNIQUE               (bc_id,        bc_language_id,               bc_history_index)
) TYPE=innodb;

-- ブログエントリー(記事)マスター
DROP TABLE IF EXISTS blog_entry;
CREATE TABLE blog_entry (
    be_serial            INT            AUTO_INCREMENT,                              -- レコードシリアル番号
    be_id                INT            DEFAULT 0                     NOT NULL,      -- エントリーID
    be_language_id       VARCHAR(2)     DEFAULT ''                    NOT NULL,      -- 言語ID
    be_history_index     INT            DEFAULT 0                     NOT NULL,      -- 履歴管理用インデックスNo(0～)
    
    be_name              VARCHAR(40)    DEFAULT ''                    NOT NULL,      -- エントリータイトル
    be_html              VARCHAR(2000)  DEFAULT ''                    NOT NULL,      -- エントリー本文HTML
    be_html_ext          VARCHAR(2000)  DEFAULT ''                    NOT NULL,      -- エントリー本文HTML(続き)
    be_description       VARCHAR(100)   DEFAULT ''                    NOT NULL,      -- 概要
    be_status            SMALLINT       DEFAULT 0                     NOT NULL,      -- エントリー状態(0=未設定、1=編集中、2=公開、3=非公開)
    be_search_tag        VARCHAR(100)   DEFAULT ''                    NOT NULL,      -- 検索用タグ(「,」区切り)
    be_regist_user_id    INT            DEFAULT 0                     NOT NULL,      -- エントリー作者
    be_regist_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- 投稿日時
    be_active_start_dt   TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- 公開期間(開始)
    be_active_end_dt     TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- 公開期間(終了)

    be_check_out_user_id INT            DEFAULT 0                     NOT NULL,      -- チェックアウトユーザID(0のときはチェックイン状態)
    be_check_out_dt      TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- チェックアウト日時
    be_create_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード作成者
    be_create_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード作成日時
    be_update_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード更新者
    be_update_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード更新日時
    be_deleted           BOOLEAN        DEFAULT false                 NOT NULL,      -- レコード削除状態
    PRIMARY KEY          (be_serial),
    UNIQUE               (be_id,        be_language_id,               be_history_index)
) TYPE=innodb;

-- ブログ記事とブログ記事カテゴリーの対応付けマスター
DROP TABLE IF EXISTS blog_entry_with_category;
CREATE TABLE blog_entry_with_category (
    bw_serial            INT            AUTO_INCREMENT,                              -- レコードシリアル番号
    bw_entry_id          INT            DEFAULT 0                     NOT NULL,      -- ブログ記事ID
    bw_language_id       VARCHAR(2)     DEFAULT ''                    NOT NULL,      -- 言語ID
    bw_history_index     INT            DEFAULT 0                     NOT NULL,      -- 履歴管理用インデックスNo(0～)
    bw_index             INT            DEFAULT 0                     NOT NULL,      -- インデックス番号
    bw_category_id       INT            DEFAULT 0                     NOT NULL,      -- ブログ記事カテゴリーID
    bw_deleted           BOOLEAN        DEFAULT false                 NOT NULL,      -- レコード削除状態
    PRIMARY KEY          (bw_serial),
    UNIQUE               (bw_entry_id,        bw_language_id,       bw_history_index,     bw_index)
) TYPE=innodb;

-- ブログコメントトラン
DROP TABLE IF EXISTS blog_comment;
CREATE TABLE blog_comment (
    bo_serial            INT            AUTO_INCREMENT,                              -- レコードシリアル番号
    bo_entry_id          INT            DEFAULT 0                     NOT NULL,      -- エントリーID
    bo_user_id           INT            DEFAULT 0                     NOT NULL,      -- ユーザID
    bo_regist_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- 投稿日時
    
    bo_title             VARCHAR(50)    DEFAULT ''                    NOT NULL,      -- タイトル
    bo_html              VARCHAR(200)   DEFAULT ''                    NOT NULL,      -- 本文HTML
    bo_url               TEXT           DEFAULT ''                    NOT NULL,      -- 参照用URL
    bo_user_name         VARCHAR(40)    DEFAULT ''                    NOT NULL,      -- ユーザ名
    bo_email             VARCHAR(40)    DEFAULT ''                    NOT NULL,      -- Eメールアドレス
    
    bo_update_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード更新者
    bo_update_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード更新日時
    bo_deleted           BOOLEAN        DEFAULT false                 NOT NULL,      -- レコード削除状態
    PRIMARY KEY          (bo_serial)
) TYPE=innodb;

-- BBS設定マスター
DROP TABLE IF EXISTS bbs_config;
CREATE TABLE bbs_config (
    sf_id                VARCHAR(30)    DEFAULT ''                    NOT NULL,      -- ID(Key)
    sf_value             TEXT           DEFAULT ''                    NOT NULL,      -- 値
    sf_name              VARCHAR(50)    DEFAULT ''                    NOT NULL,      -- 名称
    sf_description       VARCHAR(80)    DEFAULT ''                    NOT NULL,      -- 説明
    sf_index             INT            DEFAULT 0                     NOT NULL,      -- ソート用
    PRIMARY KEY          (sf_id)
) TYPE=innodb;
INSERT INTO bbs_config (sf_id,                     sf_value,    sf_name,                        sf_index)
VALUES                 ('use_email',               '1',         'Eメール送信機能',              1);
INSERT INTO bbs_config (sf_id,                     sf_value,    sf_name,                        sf_index)
VALUES                 ('send_password_on_regist', '0',         '会員登録時のEメール自動送信',  2);
INSERT INTO bbs_config (sf_id,                     sf_value,    sf_name,                        sf_index)
VALUES                 ('can_edit_thread',         '0',         '投稿記事編集許可',             3);
INSERT INTO bbs_config (sf_id,                     sf_value,    sf_name,                        sf_index)
VALUES                 ('admin_name',              'BBS管理者', '管理者名',                     4);
INSERT INTO bbs_config (sf_id,                     sf_value,    sf_name,                        sf_index)
VALUES                 ('auto_email_sender',       '',          '自動送信メール送信元アドレス', 5);

-- BBSカテゴリマスター
DROP TABLE IF EXISTS bbs_category;
CREATE TABLE bbs_category (
    sr_serial            INT            AUTO_INCREMENT,                              -- レコードシリアル番号
    sr_id                INT            DEFAULT 0                     NOT NULL,      -- カテゴリID
    sr_language_id       VARCHAR(2)     DEFAULT ''                    NOT NULL,      -- 言語ID
    sr_history_index     INT            DEFAULT 0                     NOT NULL,      -- 履歴管理用インデックスNo(0～)

    sr_name              VARCHAR(30)    DEFAULT ''                    NOT NULL,      -- カテゴリ名称
    sr_sort_order        INT            DEFAULT 0                     NOT NULL,      -- ソート用
    sr_visible           BOOLEAN        DEFAULT true                  NOT NULL,      -- 表示するかどうか

    sr_create_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード作成者
    sr_create_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード作成日時
    sr_update_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード更新者
    sr_update_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード更新日時
    sr_deleted           BOOLEAN        DEFAULT false                 NOT NULL,      -- レコード削除状態
    PRIMARY KEY          (sr_serial),
    UNIQUE               (sr_id,        sr_language_id,               sr_history_index)
) TYPE=innodb;

-- BBSグループマスター
DROP TABLE IF EXISTS bbs_group;
CREATE TABLE bbs_group (
    sg_serial            INT            AUTO_INCREMENT,                              -- レコードシリアル番号
    sg_id                INT            DEFAULT 0                     NOT NULL,      -- グループID
    sg_language_id       VARCHAR(2)     DEFAULT ''                    NOT NULL,      -- 言語ID
    sg_history_index     INT            DEFAULT 0                     NOT NULL,      -- 履歴管理用インデックスNo(0～)

    sg_name              VARCHAR(30)    DEFAULT ''                    NOT NULL,      -- グループ名称
    sg_sort_order        INT            DEFAULT 0                     NOT NULL,      -- ソート用
    sg_editable          BOOLEAN        DEFAULT true                  NOT NULL,      -- データの編集許可

    sg_create_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード作成者
    sg_create_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード作成日時
    sg_update_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード更新者
    sg_update_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード更新日時
    sg_deleted           BOOLEAN        DEFAULT false                 NOT NULL,      -- レコード削除状態
    PRIMARY KEY          (sg_serial),
    UNIQUE               (sg_id,        sg_language_id,               sg_history_index)
) TYPE=innodb;
INSERT INTO `bbs_group` (`sg_id`, `sg_language_id`, `sg_name`, `sg_sort_order`, `sg_editable`, `sg_create_dt`) VALUES
(1,    'ja',           'ゲスト',        1,   false,   now());

-- BBS記事マスター
DROP TABLE IF EXISTS bbs_thread;
CREATE TABLE bbs_thread (
    se_serial            INT            AUTO_INCREMENT,                              -- レコードシリアル番号
    se_id                INT            DEFAULT 0                     NOT NULL,      -- エントリーID
    se_language_id       VARCHAR(2)     DEFAULT ''                    NOT NULL,      -- 言語ID
    se_history_index     INT            DEFAULT 0                     NOT NULL,      -- 履歴管理用インデックスNo(0～)
    
    se_name              VARCHAR(40)    DEFAULT ''                    NOT NULL,      -- タイトル
    se_html              VARCHAR(2000)  DEFAULT ''                    NOT NULL,      -- 本文HTML
    se_status            SMALLINT       DEFAULT 0                     NOT NULL,      -- 記事状態(0=未設定、1=保留、2=表示、3=非表示)
    se_closed            BOOLEAN        DEFAULT false                 NOT NULL,      -- 投稿終了状態
    se_level             INT            DEFAULT 0                     NOT NULL,      -- 階層レベル
    se_max_sort_order    INT            DEFAULT 0                     NOT NULL,      -- 同スレッド内のソート順最大値
    se_root_id           INT            DEFAULT 0                     NOT NULL,      -- ルートスレッドID(ルートの場合は自ID)
    se_parent_id         INT            DEFAULT 0                     NOT NULL,      -- 親スレッドID
    se_sort_order        INT            DEFAULT 0                     NOT NULL,      -- 同スレッド内のソート順
    se_category_id       INT            DEFAULT 0                     NOT NULL,      -- 所属カテゴリー
    se_regist_user_id    INT            DEFAULT 0                     NOT NULL,      -- 投稿者
    se_regist_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- 投稿日時
    se_log_serial        INT            DEFAULT 0                     NOT NULL,      -- アクセスログシリアル番号

    se_check_out_user_id INT            DEFAULT 0                     NOT NULL,      -- チェックアウトユーザID(0のときはチェックイン状態)
    se_check_out_dt      TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- チェックアウト日時
    se_create_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード作成者
    se_create_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード作成日時
    se_update_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード更新者
    se_update_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード更新日時
    se_deleted           BOOLEAN        DEFAULT false                 NOT NULL,      -- レコード削除状態
    PRIMARY KEY          (se_serial),
    UNIQUE               (se_id,        se_language_id,               se_history_index)
) TYPE=innodb;

-- BBSグループアクセス権マスター
DROP TABLE IF EXISTS bbs_group_access;
CREATE TABLE bbs_group_access (
    so_group_id          INT            DEFAULT 0                     NOT NULL,      -- グループID(0はゲスト(ログインなし)グループ)
    so_category_id       INT            DEFAULT 0                     NOT NULL,      -- カテゴリーID
    
    so_read              BOOLEAN        DEFAULT false                 NOT NULL,      -- 読み込み権限
    so_write             BOOLEAN        DEFAULT false                 NOT NULL,      -- 書き込み権限

    so_update_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード更新者
    so_update_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード更新日時
    PRIMARY KEY          (so_group_id,  so_category_id)
) TYPE=innodb;

-- BBS会員情報マスター
DROP TABLE IF EXISTS bbs_member;
CREATE TABLE bbs_member (
    sv_serial            INT            AUTO_INCREMENT,                              -- レコードシリアル番号
    sv_id                INT            DEFAULT 0                     NOT NULL,      -- 会員ID
    sv_history_index     INT            DEFAULT 0                     NOT NULL,      -- 履歴管理用インデックスNo(0～)
    
    sv_language_id       VARCHAR(2)     DEFAULT ''                    NOT NULL,      -- 対応言語ID
    sv_type              SMALLINT       DEFAULT 0                     NOT NULL,      -- 会員種別(0=仮会員、1=正会員)
    sv_name              VARCHAR(40)    DEFAULT ''                    NOT NULL,      -- 会員名
    sv_login_user_id     INT            DEFAULT 0                     NOT NULL,      -- ログインユーザID
    sv_group             TEXT           DEFAULT ''                    NOT NULL,      -- 所属グループ
    sv_regist_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- 登録日時
    sv_avatar            VARCHAR(40)    DEFAULT ''                    NOT NULL,      -- アバターファイル名
    sv_signature         VARCHAR(200)   DEFAULT ''                    NOT NULL,      -- 署名
    sv_url               TEXT           DEFAULT ''                    NOT NULL,      -- ホームーページ
    sv_recv_mailnews     BOOLEAN        DEFAULT false                 NOT NULL,      -- 新着情報メールを受信するかどうか

    sv_create_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード作成者
    sv_create_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード作成日時
    sv_update_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード更新者
    sv_update_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード更新日時
    sv_deleted           BOOLEAN        DEFAULT false                 NOT NULL,      -- レコード削除状態
    PRIMARY KEY          (sv_serial),
    UNIQUE               (sv_id,        sv_history_index)
) TYPE=innodb;

-- BBS投稿ログトラン
DROP TABLE IF EXISTS bbs_post_log;
CREATE TABLE bbs_post_log (
    sl_user_id        	 INT            DEFAULT 0                     NOT NULL,      -- 投稿ユーザID
    sl_count             INT            DEFAULT 0                     NOT NULL,      -- 投稿回数
    sl_update_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- 最終投稿日時
    PRIMARY KEY  (sl_user_id)
) TYPE=innodb;

-- BBS投稿参照トラン
DROP TABLE IF EXISTS bbs_view_count;
CREATE TABLE bbs_view_count (
    su_serial            INT            AUTO_INCREMENT,                              -- レコードシリアル番号
    su_thread_id         INT            DEFAULT 0                     NOT NULL,      -- スレッドID
    su_language_id       VARCHAR(2)     DEFAULT ''                    NOT NULL,      -- 言語ID
    su_date              DATE           DEFAULT '0000-00-00'          NOT NULL,      -- 日付
    su_hour              SMALLINT       DEFAULT 0                     NOT NULL,      -- 時間
    su_count             INT            DEFAULT 0                     NOT NULL,      -- 参照数
    PRIMARY KEY          (su_serial),
    UNIQUE               (su_thread_id,   su_language_id,         su_date,       su_hour)
) TYPE=innodb;

-- アクセスカウンターウィジェット用
-- アクセス時間管理テーブル
DROP TABLE IF EXISTS ac_access;
CREATE TABLE ac_access (
    ac_ssid              VARCHAR(32)    DEFAULT ''                    NOT NULL,      -- セッションID
    ac_time              INT            DEFAULT 0                     NOT NULL,      -- 最終アクセス時間
    PRIMARY KEY  (ac_ssid)
) TYPE=innodb;

-- アクセス数管理テーブル
DROP TABLE IF EXISTS ac_count;
CREATE TABLE ac_count (
    co_date              DATE           DEFAULT '0000-00-00'          NOT NULL,      -- 日付
    co_count             INT            DEFAULT 0                     NOT NULL,      -- アクセス回数
    PRIMARY KEY  (co_date)
) TYPE=innodb;

-- バナーウィジェット用
-- バナー表示定義
DROP TABLE IF EXISTS bn_def;
CREATE TABLE bn_def (
    bd_serial            INT            AUTO_INCREMENT,                              -- レコードシリアル番号
    bd_id                INT            DEFAULT 0                     NOT NULL,      -- バナーID
    bd_item_id           TEXT           DEFAULT ''                    NOT NULL,      -- 対応バナー項目(「,」区切りで複数指定可)
    bd_name              VARCHAR(50)    DEFAULT ''                    NOT NULL,      -- バナー名
    bd_disp_type         SMALLINT       DEFAULT 0                     NOT NULL,      -- 表示形式(0=サイクリック,1=ランダム)
    bd_first_item_index  INT            DEFAULT 0                     NOT NULL,      -- バナー項目の読み込み位置インデックス(サイクリック時に使用)
    bd_disp_item_count   SMALLINT       DEFAULT 0                     NOT NULL,      -- 同時に表示する項目数
    bd_disp_direction    SMALLINT       DEFAULT 0                     NOT NULL,      -- 表示方向(0=縦,1=横)
    bd_disp_align        SMALLINT       DEFAULT 0                     NOT NULL,      -- 表示位置アラインメント(0=指定なし,1=left,2=center,3=right)
    
    bd_update_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード更新者
    bd_update_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード更新日時
    PRIMARY KEY          (bd_serial),
    UNIQUE               (bd_id)
) TYPE=innodb;

-- バナー項目
DROP TABLE IF EXISTS bn_item;
CREATE TABLE bn_item (
    bi_serial            INT            AUTO_INCREMENT,                              -- レコードシリアル番号
    bi_id                INT            DEFAULT 0                     NOT NULL,      -- バナー項目ID
    bi_history_index     INT            DEFAULT 0                     NOT NULL,      -- 履歴管理用インデックスNo(0～)
    bi_group             INT            DEFAULT 0                     NOT NULL,      -- グルーピング用
    bi_name              VARCHAR(40)    DEFAULT ''                    NOT NULL,      -- バナー名
    bi_description       VARCHAR(100)   DEFAULT ''                    NOT NULL,      -- 説明
    bi_admin_note        VARCHAR(100)   DEFAULT ''                    NOT NULL,      -- 管理者用備考
    bi_image_url         TEXT           DEFAULT ''                    NOT NULL,      -- 表示画像
    bi_link_url          TEXT           DEFAULT ''                    NOT NULL,      -- リンク先
    bi_image_width       VARCHAR(10)    DEFAULT ''                    NOT NULL,      -- 画像幅
    bi_image_height      VARCHAR(10)    DEFAULT ''                    NOT NULL,      -- 画像高さ
    bi_image_alt         VARCHAR(40)    DEFAULT ''                    NOT NULL,      -- 画像代替テキスト
    bi_image_title       VARCHAR(40)    DEFAULT ''                    NOT NULL,      -- 画像ツールチップ
    bi_html              VARCHAR(200)   DEFAULT ''                    NOT NULL,      -- HTML
    bi_visible           BOOLEAN        DEFAULT true                  NOT NULL,      -- 表示可否
    bi_active_start_dt   TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- 表示可能期間(開始)
    bi_active_end_dt     TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- 表示可能期間(終了)

    bi_check_out_user_id INT            DEFAULT 0                     NOT NULL,      -- チェックアウトユーザID(0のときはチェックイン状態)
    bi_check_out_dt      TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- チェックアウト日時
    bi_create_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード作成者
    bi_create_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード作成日時
    bi_update_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード更新者
    bi_update_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード更新日時
    bi_deleted           BOOLEAN        DEFAULT false                 NOT NULL,      -- レコード削除状態
    PRIMARY KEY          (bi_serial),
    UNIQUE               (bi_id,        bi_history_index)
) TYPE=innodb;

-- バナー項目参照ログ
DROP TABLE IF EXISTS bn_item_view;
CREATE TABLE bn_item_view (
    bv_serial            INT            AUTO_INCREMENT,                              -- レコードシリアル番号
    bv_public_key        CHAR(32)       DEFAULT ''                    NOT NULL,      -- 公開発行キー
    bv_item_serial       INT            DEFAULT 0                     NOT NULL,      -- バナー項目シリアル番号
    bv_log_serial        INT            DEFAULT 0                     NOT NULL,      -- アクセスログシリアル番号
    bv_dt                TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- 参照日時
    PRIMARY KEY          (bv_serial)
) TYPE=innodb;

-- バナー項目クリックログ
DROP TABLE IF EXISTS bn_item_access;
CREATE TABLE bn_item_access (
    ba_serial            INT            AUTO_INCREMENT,                              -- レコードシリアル番号
    ba_public_key        CHAR(32)       DEFAULT ''                    NOT NULL,      -- 公開発行キー
    ba_redirect_url      TEXT           DEFAULT ''                    NOT NULL,      -- 遷移先URL
    ba_log_serial        INT            DEFAULT 0                     NOT NULL,      -- アクセスログシリアル番号
    ba_dt                TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- 参照日時
    PRIMARY KEY          (ba_serial)
) TYPE=innodb;

-- 予約リソースマスター
DROP TABLE IF EXISTS reserve_resource;
CREATE TABLE reserve_resource (
    rr_id                INT            DEFAULT 0                     NOT NULL,      -- リソースID
    rr_type              INT            DEFAULT 0                     NOT NULL,      -- リソースタイプ(0=常設、1=週間、2=スポット)
    rr_config_id         INT            DEFAULT 0                     NOT NULL,      -- 設定ID
    rr_name              VARCHAR(50)    DEFAULT ''                    NOT NULL,      -- 名称
    rr_description       VARCHAR(80)    DEFAULT ''                    NOT NULL,      -- 説明
    rr_visible           BOOLEAN        DEFAULT true                  NOT NULL,      -- 表示可否
    rr_active_start_dt   TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- 有効期間(開始)
    rr_active_end_dt     TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- 有効期間(終了)
    rr_sort_order        INT            DEFAULT 0                     NOT NULL,      -- 表示ソート用
    
    rr_update_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード更新者
    rr_update_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード更新日時
    rr_deleted           BOOLEAN        DEFAULT false                 NOT NULL,      -- レコード削除状態
    PRIMARY KEY          (rr_id)
) TYPE=innodb;
INSERT INTO reserve_resource (rr_id, rr_type, rr_config_id, rr_name, rr_sort_order)
VALUES                       (1,     0,       0,            'デフォルト', 1);

-- 予約設定マスター
DROP TABLE IF EXISTS reserve_config;
CREATE TABLE reserve_config (
    rc_id                INT            DEFAULT 0                     NOT NULL,      -- 定義ID
    rc_key               VARCHAR(30)    DEFAULT ''                    NOT NULL,      -- キー
    rc_value             TEXT           DEFAULT ''                    NOT NULL,      -- 値
    rc_name              VARCHAR(50)    DEFAULT ''                    NOT NULL,      -- 名称
    rc_description       VARCHAR(80)    DEFAULT ''                    NOT NULL,      -- 説明
    rc_index             INT            DEFAULT 0                     NOT NULL,      -- ソート用
    PRIMARY KEY          (rc_id,        rc_key)
) TYPE=innodb;
INSERT INTO reserve_config (rc_id,   rc_key,                  rc_value, rc_name,        rc_index)
VALUES                     (0, 'unit_interval_minute', '15',     '単位時間(分)', 0);
INSERT INTO reserve_config (rc_id,   rc_key,                  rc_value, rc_name,                 rc_index)
VALUES                     (0, 'max_count_per_unit',      '3',      '1単位あたりの最大登録数', 1);
INSERT INTO reserve_config (rc_id,   rc_key,                  rc_value, rc_name,                 rc_index)
VALUES                     (0, 'view_day_start',      '0',      '先頭に表示する日付', 2);
INSERT INTO reserve_config (rc_id,   rc_key,                  rc_value, rc_name,                 rc_index)
VALUES                     (0, 'view_day_range',      '10',      '一覧表示日数', 3);
INSERT INTO reserve_config (rc_id,   rc_key,                  rc_value, rc_name,                 rc_index)
VALUES                     (0, 'max_user_reserve_count',      '1',      'ユーザの最大予約可能数', 4);
INSERT INTO reserve_config (rc_id,   rc_key,                  rc_value, rc_name,                 rc_index)
VALUES                     (0, 'default_resource_id',      '1',      'デフォルトのリソースID', 5);
INSERT INTO reserve_config (rc_id,   rc_key,                  rc_value, rc_name,                 rc_index)
VALUES                     (0, 'cancel_available_day',      '1',    '予約キャンセル可能な日数', 6);
INSERT INTO reserve_config (rc_id,   rc_key,                  rc_value, rc_name,                 rc_index)
VALUES                     (0, 'show_new_reserve_field',      '1',    '新規予約フィールドを表示', 7);

-- 予約カレンダーマスター
DROP TABLE IF EXISTS reserve_calendar;
CREATE TABLE reserve_calendar (
    ra_serial            INT            AUTO_INCREMENT,                              -- レコードシリアル番号
    ra_config_id         INT            DEFAULT 0                     NOT NULL,      -- 設定ID
    ra_usual             BOOLEAN        DEFAULT false                 NOT NULL,      -- 通常あるいは特定日の区別
    ra_specify_type      INT            DEFAULT 0                     NOT NULL,      -- 属性指定方法(0=デフォルト値、1=曜日指定、2=毎月、3=毎年)
    ra_day_attribute     INT            DEFAULT 0                     NOT NULL,      -- 日にち属性(0=未設定、1～7=日曜～土曜、8=祝日)
    ra_date              DATE           DEFAULT '0000-00-00'          NOT NULL,      -- 日にち指定の場合の日にち
    ra_start_time        INT            DEFAULT 0                     NOT NULL,      -- 日にち時間範囲指定の場合の開始時間(hhmm)
    ra_end_time          INT            DEFAULT 0                     NOT NULL,      -- 日にち時間範囲指定の場合の終了時間(hhmm)
    ra_available         BOOLEAN        DEFAULT false                 NOT NULL,      -- 利用可能かどうか
    PRIMARY KEY          (ra_serial)
) TYPE=innodb;
INSERT INTO reserve_calendar (ra_config_id, ra_usual, ra_specify_type, ra_day_attribute, ra_start_time, ra_end_time, ra_available)
VALUES                       (0,            true,     1,               2,                900,           1200,        true);
INSERT INTO reserve_calendar (ra_config_id, ra_usual, ra_specify_type, ra_day_attribute, ra_start_time, ra_end_time, ra_available)
VALUES                       (0,            true,     1,               2,                1300,          1730,        true);
INSERT INTO reserve_calendar (ra_config_id, ra_usual, ra_specify_type, ra_day_attribute, ra_start_time, ra_end_time, ra_available)
VALUES                       (0,            true,     1,               3,                900,           1200,        true);
INSERT INTO reserve_calendar (ra_config_id, ra_usual, ra_specify_type, ra_day_attribute, ra_start_time, ra_end_time, ra_available)
VALUES                       (0,            true,     1,               3,                1300,          1730,        true);
INSERT INTO reserve_calendar (ra_config_id, ra_usual, ra_specify_type, ra_day_attribute, ra_start_time, ra_end_time, ra_available)
VALUES                       (0,            true,     1,               4,                900,           1200,        true);
INSERT INTO reserve_calendar (ra_config_id, ra_usual, ra_specify_type, ra_day_attribute, ra_start_time, ra_end_time, ra_available)
VALUES                       (0,            true,     1,               4,                1300,          1730,        true);
INSERT INTO reserve_calendar (ra_config_id, ra_usual, ra_specify_type, ra_day_attribute, ra_start_time, ra_end_time, ra_available)
VALUES                       (0,            true,     1,               5,                900,           1200,        true);
INSERT INTO reserve_calendar (ra_config_id, ra_usual, ra_specify_type, ra_day_attribute, ra_start_time, ra_end_time, ra_available)
VALUES                       (0,            true,     1,               5,                1300,          1730,        true);
INSERT INTO reserve_calendar (ra_config_id, ra_usual, ra_specify_type, ra_day_attribute, ra_start_time, ra_end_time, ra_available)
VALUES                       (0,            true,     1,               6,                900,           1200,        true);
INSERT INTO reserve_calendar (ra_config_id, ra_usual, ra_specify_type, ra_day_attribute, ra_start_time, ra_end_time, ra_available)
VALUES                       (0,            true,     1,               6,                1300,          1730,        true);
INSERT INTO reserve_calendar (ra_config_id, ra_usual, ra_specify_type, ra_day_attribute, ra_start_time, ra_end_time, ra_available)
VALUES                       (0,            true,     1,               7,                900,           1200,        true);

-- 予約状況トラン
DROP TABLE IF EXISTS reserve_status;
CREATE TABLE reserve_status (
    rs_serial            INT            AUTO_INCREMENT,                              -- レコードシリアル番号
    rs_resource_id       INT            DEFAULT 0                     NOT NULL,      -- リソースID
    rs_user_id           INT            DEFAULT 0                     NOT NULL,      -- ユーザID
    rs_status            INT            DEFAULT 0                     NOT NULL,      -- 状態(1=予約、2=キャンセル)
    rs_start_dt          TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- 開始日時
    rs_note              VARCHAR(100)   DEFAULT ''                    NOT NULL,      -- 備考
    
    rs_create_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード作成者
    rs_create_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード作成日時
    rs_update_user_id    INT            DEFAULT 0                     NOT NULL,      -- レコード更新者
    rs_update_dt         TIMESTAMP      DEFAULT '0000-00-00 00:00:00' NOT NULL,      -- レコード更新日時
    rs_deleted           BOOLEAN        DEFAULT false                 NOT NULL,      -- レコード削除状態
    PRIMARY KEY          (rs_serial)
) TYPE=innodb;

-- 更新データ
-- テーブルのバージョン
DELETE FROM _version WHERE vs_id = 'standard_table';
INSERT INTO _version (vs_id,                         vs_value, vs_name)
VALUES               ('standard_table',     '2008032401',     '標準テーブルのバージョン');

-- メール内容
DELETE FROM _mail_form WHERE mf_id = 'regist_member';
INSERT INTO _mail_form (mf_id,           mf_language_id, mf_subject,         mf_content,                                                                 mf_create_dt) 
VALUES                 ('regist_member', 'ja',           '会員登録',         'ご登録ありがとうございました。\nパスワードを送信します。\nこのパスワードでログインし、パスワードを再設定してください。\n\n[#URL#]\n\nパスワード　[#PASSWORD#]', now());
DELETE FROM _mail_form WHERE mf_id = 'send_password';
INSERT INTO _mail_form (mf_id,           mf_language_id, mf_subject,         mf_content,                                                                 mf_create_dt) 
VALUES                 ('send_password', 'ja',           'パスワード再送信', 'パスワードを再送信します。\nこのパスワードでログインし、パスワードを再設定してください。\n\n[#URL#]\n\nパスワード　[#PASSWORD#]',                               now());
DELETE FROM _mail_form WHERE mf_id = 'send_password_simple';
INSERT INTO _mail_form (mf_id,           mf_language_id, mf_subject,         mf_content,                                                                 mf_create_dt) 
VALUES                 ('send_password_simple', 'ja',    'パスワード再送信', 'パスワードを再送信します。\nこのパスワードでログインし、パスワードを再設定してください。\n\nパスワード　[#PASSWORD#]',                               now());
DELETE FROM _mail_form WHERE mf_id = 'contact_us';
INSERT INTO _mail_form (mf_id,           mf_language_id, mf_subject,         mf_content,                                                                 mf_create_dt) 
VALUES                 ('contact_us', 'ja',              'お問い合わせ',     '以下のお問い合わせがありました。\n\n[#BODY#]',                             now());

-- 管理画面メニューデータ
DELETE FROM _nav_item WHERE ni_id >= 500 AND ni_id < 1000;
INSERT INTO _nav_item
(ni_id, ni_parent_id, ni_index, ni_nav_id,    ni_task_id,                     ni_group_id, ni_param, ni_name,    ni_help_title, ni_help_body, ni_visible) VALUES
(500,   0,            3,        'admin_menu', '_widget',                      '',          '',       'ウィジェット設定', 'ウィジェット設定', '個別のウィジェットの設定を行います。', true),
(501,   500,          0,        'admin_menu', 'configwidget_default_mainmenu','',          '',          'メインメニュー', 'メインメニュー', 'Webサイトで常時表示するメニューを作成します。', true),
(502,   500,          1,        'admin_menu', 'configwidget_default_content', '',          '',       'コンテンツビュー', 'デフォルトコンテンツビュー', 'Webサイトのコンテンツを管理する設定画面です。', true),
(503,   500,          2,        'admin_menu', 'configwidget_ec_main',         '',          '',       'Eコマースメイン', 'Eコマースメイン', 'Eコマース機能のメインとなる設定画面です。', true),
(504,   500,          3,        'admin_menu', 'configwidget_blog_main',       '',          '',       'ブログメイン', 'ブログメイン', 'ブログ機能のメインとなる設定画面です。', true),
(505,   500,          4,        'admin_menu', 'configwidget_bbs_main',        '',          '',       '掲示板メイン', '掲示板メイン', '掲示板機能のメインとなる設定画面です。', true),
(506,   500,          5,        'admin_menu', 'configwidget_reserve_main',        '',          '',   '予約メイン', '予約メイン', '汎用予約機能のメインとなる設定画面です。', true);

-- ウィジェット情報
DELETE FROM _widgets WHERE wd_id = 'default_mainmenu';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('default_mainmenu',   'デフォルトメインメニュー',   'DMEN', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, '常に表示し共通で使用するメインメニュー。',         false,           false,       true,         true,        true,         false,               false,true,           0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'nav_menu';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('nav_menu',   'ナビゲーションメニュー',   'NAVM', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'ページのヘッダ下や、フッタ部に表示するメニュー。',           false,           false,       true,         true,        true,         false,               false,true,           0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'default_footer';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('default_footer',     'デフォルトフッタ',           'DFOT', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'フッタ部分に表示し、著作権の表示を行う。',          false,           false,       true,         true,        false,        false,               false,true,           0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'custom_header';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('custom_header',     'カスタムヘッダ',           'DFHT', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'ヘッダ部分で画像やサイトのタイトル文字列をカスタマイズ。',          false,           false,       true,         true,        true,        false,               false,true,           0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'ec_login_box';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('ec_login_box',       'Eコマース - ログイン',       'ELOG', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'Eコマースの会員ログイン用ボックス。',          false,           false,       true,         true,        false,        false,               false,true,           100, now(), now());
DELETE FROM _widgets WHERE wd_id = 'custom_footer';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('custom_footer',     'カスタムフッタ',           'CSFT', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'フッタ部に表示し、著作権等の表示を行う。',          false,           false,       true,         true,        true,        false,               false,true,           0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'separator';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('separator',          'セパレータ',                 'SEPT', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'ウィジェット間の区切り線。',          false,           false,       true,         true,        false,         false,               false,true,          0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'templateChanger';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('templateChanger',    'テンプレートチェンジャー',   'TCHA', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'エンドユーザからテンプレートの変更を可能にする。',          false,           false,       true,         true,        false,        false,               false,true,           0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'date';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('date',               '現在日時',                   'DTST', '1.0.0',  '',  'Naoki Hirata', 'Magic3.org', 'GPL', 10, '現在の日付時刻を表示。',      false,           false,       true,         true,        false,        false,               false,true,           0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'ajax_sample1';
INSERT INTO _widgets 
(wd_id,          wd_name,                  wd_type, wd_version, wd_params, wd_author,      wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_use_ajax, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('ajax_sample1', 'Ajaxサンプルプログラム', 'AJSP',  '1.0.0',    '',        'Naoki Hirata', 'Magic3.org', 'GPL',      10, 'Ajaxサンプルプログラム。サーバと通信し、サーバの時刻を取得。',                true,            true,        true,        true,         true,               false,        true,                false,               true,              0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'access_count';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('access_count',       'アクセスカウンター',         'ACCT', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'サイトのアクセス数を表示。',          false,           false,       true,         true,        false,        false,               false,true,           0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'clock';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('clock',              '時計',                       'CLCK', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'アナログ時計。',          true,            false,       true,         true,        false,        false,               false,true,           0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'dg_clock';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('dg_clock',           'デジタル時計',               'DGCK', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'デジタル時計。',          false,           false,       true,         true,        false,        false,               false,true,           0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'default_content';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('default_content',    'デフォルトコンテンツビュー', 'DCON', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, '画像やテキストからできた、様々なコンテンツを表示。',          false,           false,       true,         true,        true,         false,               false,true,           0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'news';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('news',    'お知らせ', 'NEWS', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'お知らせを表示。',          false,           false,       true,         true,        true,         false,               true,true,           0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'blogparts_box';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('blogparts_box',    'ブログパーツ', 'BPAR', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'ブログパーツを表示',          false,           false,       true,         true,        true,         false,               true,true,           0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'banner';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('banner',             'バナー表示',                 'BANR', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'バナー画像を表示。',          false,           false,       true,         true,        true,         true,               true,true,            0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'ec_main';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('ec_main',            'Eコマース - メイン',         'ECOM', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'Eコマースのメインプログラム。商品の説明や一覧を表示したり、会員登録処理や商品購入処理などを行う。',          false,           false,       true,         true,        true,         true,               false,true,           0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'ec_menu';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('ec_menu',   'Eコマース - 商品メニュー',   'ECMN', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'Eコマースの商品メニュー。カテゴリで取得した商品一覧をEコマースメインウィジェットに表示させる。',          false,           false,       true,         true,        true,         false,               false,true,           0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'ec_category_menu';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('ec_category_menu',   'Eコマース - カテゴリーメニュー', 'ECCM', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'Eコマースカテゴリメニュー',          false,           false,       true,         true,        true,         false,               false,true,           0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'ec_cart_box';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('ec_cart_box',        'Eコマース - カート',     'ECCT', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'Eコマースでカート内の商品を表示するボックス。',         false,           false,       true,         true,        false,         false,               false,true,           100, now(),      now());
DELETE FROM _widgets WHERE wd_id = 'ec_product_header';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('ec_product_header',               'Eコマース - 商品ヘッダ',                   'ECHD', '1.0.0',  '',  'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'Eコマース商品メニューで商品一覧表示させるときに同時に商品のヘッダ画像を表示する',     false,           false,       true,         true,        false,        false,               false,true,           0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'ec_product_display';
INSERT INTO _widgets
(wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('ec_product_display',            'Eコマース - 新着おすすめ',         'ECPD', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, '商品のおすすめや新着などの一覧を表示する。',          false,           false,       true,         true,        true,         false,               true,true,           0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'ec_search_box';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('ec_search_box',   'Eコマース - 検索',            'ECSR', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'Eコマース商品検索ボックス。',         false,           false,       true,         true,        false,         false,               false,true,               0, now(),    now());
DELETE FROM _widgets WHERE wd_id = 'yahoo_weather';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('yahoo_weather',      'Yahooお天気',                'YWEA', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'Yahooのサイトからお天気情報を取得表示する。',         false,           false,       true,         true,        false,        false,               false,true,               0, now(),   now());
DELETE FROM _widgets WHERE wd_id = 'othello';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('othello',            'オセロ',                     'OTHE', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'オセロプログラム。',         false,           false,       true,         true,        false,        false,               false,true,               0, now(),   now());
DELETE FROM _widgets WHERE wd_id = 'blog_main';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('blog_main',          'ブログ - メイン',            'BGMA', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'ブログの記事の内容を表示する。',          false,           false,       true,         true,        true,        false,               false,true,               0, now(),    now());
DELETE FROM _widgets WHERE wd_id = 'blog_search_box';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('blog_search_box',   'ブログ - 検索',            'BGSR', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'ブログ記事を検索するためのボックス。',        false,           false,       true,         true,        false,         false,               false,true,               0, now(),    now());
DELETE FROM _widgets WHERE wd_id = 'blog_new_box';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('blog_new_box',   'ブログ - 新規',            'BGNW', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'ブログの新規一覧を表示。',         false,           false,       true,         true,        false,         false,               false,true,               0, now(),    now());
DELETE FROM _widgets WHERE wd_id = 'blog_calendar_box';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('blog_calendar_box',   'ブログ - カレンダー',            'BGCL', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'カレンダーからブログ記事にアクセスするためのボックス。',        false,           false,       true,         true,        false,         false,               false,true,               0, now(),    now());
DELETE FROM _widgets WHERE wd_id = 'bbs_main';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('bbs_main',          '掲示板 - メイン',            'BBMA', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, '掲示板の投稿記事を表示。',         false,           false,       true,         true,        true,        false,               false,true,               0, now(),    now());
DELETE FROM _widgets WHERE wd_id = 'bbs_login_box';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('bbs_login_box',         '掲示板 - ログイン',            'BBML', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, '掲示板の会員ログイン用ボックス。',         false,           false,       true,         true,        false,        false,               false,true,               100, now(),    now());
DELETE FROM _widgets WHERE wd_id = 'contactus';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('contactus',          'お問い合わせメール送信',            'CONU', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'お問い合わせの入力データをメールで送る。',          false,           false,       true,         true,        true,        false,               false,true,               0, now(),    now());
DELETE FROM _widgets WHERE wd_id = 'joomla_clock';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('joomla_clock',           'Joomla時計',               'JMCK', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'Joomlaロゴの時計。',          false,           false,       true,         true,        false,        false,               false,true,           0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'qrcode';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('qrcode',           'QRコード',               'QRVW', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'QRコードを作成表示。',         false,           false,       true,         true,        true,        true,               false,true,           0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'youtube';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('youtube',     'YouTube',           'YOUT', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'YouTubeの投稿動画を表示。',         false,           false,       true,         true,        true,        false,               false,true,           0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'image';
INSERT INTO _widgets
(wd_id,   wd_name, wd_type, wd_version, wd_params, wd_author,      wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('image', '画像',  'IMAG',  '1.0.0',    '',        'Naoki Hirata', 'Magic3.org', 'GPL',      10, '様々な画像を表示。',               false,           false,       true,         true,        true,         false,                                true,                true,              0,               now(),         now());
DELETE FROM _widgets WHERE wd_id = 'release_info';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('release_info',               'Magic3リリース情報',                   'RINF', '1.0.0',  '',  'Naoki Hirata', 'Magic3.org', 'GPL', 10, 'Magic3の最新リリース情報を表示。',      false,           false,       true,         true,        false,        false,               false,true,           0, now(), now());
DELETE FROM _widgets WHERE wd_id = 'reserve_main';
INSERT INTO _widgets (wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('reserve_main',            '予約 - メイン',         'RESM', '1.0.0',  '',        'Naoki Hirata', 'Magic3.org', 'GPL', 10, '予約管理のメインプログラム。',          false,           false,       true,         true,        true,         true,               false,true,           0, now(), now());

DELETE FROM _widgets WHERE wd_id = 'ec_product_random_box';
INSERT INTO _widgets
(wd_id, wd_name, wd_type, wd_version, wd_params, wd_author, wd_copyright, wd_license, wd_official_level, wd_description, wd_add_script_lib, wd_read_scripts, wd_read_css, wd_available, wd_editable, wd_has_admin, wd_enable_operation, wd_use_instance_def, wd_is_initialized, wd_launch_index, wd_install_dt, wd_create_dt) VALUES
('ec_product_random_box',               'Eコマース - 商品ランダム表示',                   'ECPR', '1.0.0',  '',  'Naoki Hirata', 'Magic3.org', 'GPL', 10, '商品をランダムに表示する。', 'jquery.easing,jquery.jcarousel',    false,           true,       true,         true,        true,        false,               false,true,           0, now(), now());

-- ウィジェット情報(携帯用)
DELETE FROM _widgets WHERE wd_id = 'm/sample';
INSERT INTO _widgets
(wd_id,      wd_name,          wd_type, wd_version, wd_params, wd_author,      wd_copyright, wd_license, wd_official_level, wd_description, wd_mobile, wd_has_admin, wd_is_initialized, wd_install_dt, wd_create_dt) VALUES
('m/sample', '携帯用サンプル', 'MSAM',  '1.0.0',    '',        'Naoki Hirata', 'Magic3.org', 'GPL',      10, '携帯用表示サンプルプログラム。日付を表示。',               true,      false,         true,              now(),         now());
DELETE FROM _widgets WHERE wd_id = 'm/sample_input';
INSERT INTO _widgets
(wd_id,      wd_name,          wd_type, wd_version, wd_params, wd_author,      wd_copyright, wd_license, wd_official_level, wd_description, wd_mobile, wd_has_admin, wd_is_initialized, wd_install_dt, wd_create_dt) VALUES
('m/sample_input', '携帯用サンプル入力', 'MSAI',  '1.0.0',    '',        'Naoki Hirata', 'Magic3.org', 'GPL',      10, '携帯用入力サンプルプログラム',               true,      false,         true,              now(),         now());
DELETE FROM _widgets WHERE wd_id = 'm/contactus';
INSERT INTO _widgets
(wd_id,      wd_name,          wd_type, wd_version, wd_params, wd_author,      wd_copyright, wd_license, wd_official_level, wd_description, wd_mobile, wd_has_admin, wd_is_initialized, wd_install_dt, wd_create_dt) VALUES
('m/contactus', 'お問い合わせメール送信', 'MCON',  '1.0.0',    '',        'Naoki Hirata', 'Magic3.org', 'GPL',      10, 'お問い合わせの入力データをメールで送る。',               true,      true,         true,              now(),         now());
DELETE FROM _widgets WHERE wd_id = 'm/custom_footer';
INSERT INTO _widgets
(wd_id,      wd_name,          wd_type, wd_version, wd_params, wd_author,      wd_copyright, wd_license, wd_official_level, wd_description, wd_mobile, wd_has_admin, wd_is_initialized, wd_install_dt, wd_create_dt) VALUES
('m/custom_footer', 'カスタムフッタ', 'MCSF',  '1.0.0',    '',        'Naoki Hirata', 'Magic3.org', 'GPL',      10, 'フッタ部に表示し、著作権等の表示を行う。',               true,      true,         true,              now(),         now());
DELETE FROM _widgets WHERE wd_id = 'm/mainmenu';
INSERT INTO _widgets
(wd_id,      wd_name,          wd_type, wd_version, wd_params, wd_author,      wd_copyright, wd_license, wd_official_level, wd_description, wd_mobile, wd_has_admin, wd_is_initialized, wd_install_dt, wd_create_dt) VALUES
('m/mainmenu', 'メインメニュー', 'MMEN',  '1.0.0',    '',        'Naoki Hirata', 'Magic3.org', 'GPL',      10, '常に表示するメインメニュー。',               true,      true,         true,              now(),         now());
DELETE FROM _widgets WHERE wd_id = 'm/content';
INSERT INTO _widgets
(wd_id,      wd_name,          wd_type, wd_version, wd_params, wd_author,      wd_copyright, wd_license, wd_official_level, wd_description, wd_mobile, wd_has_admin, wd_is_initialized, wd_install_dt, wd_create_dt) VALUES
('m/content', 'コンテンツ', 'MCON',  '1.0.0',    '',        'Naoki Hirata', 'Magic3.org', 'GPL',      10, 'コンテンツ表示。',               true,      true,         true,              now(),         now());

-- テンプレート情報
TRUNCATE TABLE _templates;
INSERT INTO _templates (tm_id,         tm_name,      tm_description, tm_default, tm_create_dt) 
VALUES                 ('a6greyflash', 'a6greyflash','',             true,      now());
INSERT INTO _templates (tm_id,         tm_name,      tm_description, tm_default, tm_create_dt) 
VALUES                 ('baki',        'baki',       '',             false,     now());
INSERT INTO _templates (tm_id,                    tm_name,                  tm_description, tm_default, tm_create_dt) 
VALUES                 ('jt_millennium_expresso', 'jt_millennium_expresso', '',             false,      now());
INSERT INTO _templates (tm_id,                    tm_name,                  tm_description, tm_default, tm_create_dt) 
VALUES                 ('jt_simplicity_red',      'jt_simplicity_red',      '',             false,      now());
INSERT INTO _templates (tm_id,                    tm_name,                  tm_description, tm_default, tm_create_dt) 
VALUES                 ('baladibol',              'baladibol',      '',             false,      now());
INSERT INTO _templates (tm_id,                    tm_name,                  tm_description, tm_default, tm_create_dt) 
VALUES                 ('eastertime-1.0',         'eastertime-1.0',      '',             false,      now());
INSERT INTO _templates (tm_id,                    tm_name,                  tm_description, tm_default, tm_create_dt) 
VALUES                 ('ellipse',                'ellipse',      '',             false,      now());
INSERT INTO _templates (tm_id,                    tm_name,                  tm_description, tm_default, tm_create_dt) 
VALUES                 ('freshgreen',             'freshgreen',      '',             false,      now());
INSERT INTO _templates (tm_id,                    tm_name,                  tm_description, tm_default, tm_create_dt) 
VALUES                 ('psychic',                'psychic',      '',             false,      now());
INSERT INTO _templates (tm_id,                    tm_name,                  tm_description, tm_default, tm_create_dt) 
VALUES                 ('rd_music',               'rd_music',      '',             false,      now());
INSERT INTO _templates (tm_id,                    tm_name,                  tm_description, tm_default, tm_create_dt) 
VALUES                 ('siteground44',           'siteground44',      '',             false,      now());
INSERT INTO _templates (tm_id,                    tm_name,                  tm_description, tm_default, tm_create_dt) 
VALUES                 ('browns',                 'browns',      '',             false,      now());
INSERT INTO _templates (tm_id,                    tm_name,                  tm_description, tm_default, tm_create_dt) 
VALUES                 ('red_rose_beauty',        'red_rose_beauty',      '',             false,      now());
INSERT INTO _templates (tm_id,                    tm_name,                  tm_description, tm_default, tm_create_dt) 
VALUES                 ('ec_simple_n2',           'ec_simple_n2',           '',             false,      now());
INSERT INTO _templates (tm_id,                    tm_name,                  tm_description, tm_default, tm_create_dt) 
VALUES                 ('ec_simple_n2_pink',      'ec_simple_n2_pink',      '',             false,      now());
INSERT INTO _templates (tm_id,                    tm_name,                  tm_description, tm_default, tm_create_dt) 
VALUES                 ('ec_simple_n3',           'ec_simple_n3',           '',             false,      now());
INSERT INTO _templates (tm_id,                    tm_name,                  tm_description, tm_default, tm_create_dt) 
VALUES                 ('ec_simple_n3_aqua',      'ec_simple_n3_aqua',      '',             false,      now());
INSERT INTO _templates (tm_id,                    tm_name,                  tm_description, tm_default, tm_create_dt) 
VALUES                 ('ec_simple_w2',           'ec_simple_w2',           '',             false,      now());
INSERT INTO _templates (tm_id,                    tm_name,                  tm_description, tm_default, tm_create_dt) 
VALUES                 ('ec_simple_w3',           'ec_simple_w3',           '',             false,      now());
INSERT INTO _templates (tm_id,                    tm_name,                  tm_description, tm_default, tm_mobile, tm_create_dt) 
VALUES                 ('m/default',             'm/default',             '',             true,       true,      now());

-- 基本テーブルの不要なデータを削除
-- 画面定義
DELETE FROM _page_def WHERE pd_id = 'index';

-- 画面定義(携帯用)
DELETE FROM _page_def WHERE pd_id = 'm_index';
INSERT INTO _page_def
(pd_id,         pd_sub_id, pd_position_id, pd_index, pd_widget_id, pd_config_id, pd_visible, pd_editable, pd_update_dt)
VALUES
('m_index',     'content', 'main',         1,        'm/sample',  0,            true,       true,        now());
INSERT INTO _page_def
(pd_id,         pd_sub_id, pd_position_id, pd_index, pd_widget_id, pd_config_id, pd_visible, pd_editable, pd_update_dt)
VALUES
('m_index',     'content', 'main',         3,        'm/sample_input',  0,            true,       true,        now());
