/* 
 * Copyright (c) 2006-2007 NTT DATA CORPORATION.
 * All rights reserved.
 *
 * derived from original pg_senna.c developed by Daisuke Maki
 * (See below for copyright notice)
 *
 */
/* 
 * Portions of this code were written based on pg_senna.c:
 * 
 * $Id: pg_senna.c 14 2006-01-02 11:03:54Z daisuke $
 *
 * Copyright (c) 2005 Daisuke Maki <dmaki@cpan.org>
 * All rights reserved.
 *
 * I used pg_rast.c as a sample to learn from. Much thanks goes to the
 * pg_rast developers (See below for copyright notice)
 *
 * Development was funded by Brazil, Ltd. <http://b.razil.jp>
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose, without fee, and without a
 * written agreement is hereby granted, provided that the above
 * copyright notice and this paragraph and the following two
 * paragraphs appear in all copies.
 *
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING
 * LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS
 * DOCUMENTATION, EVEN IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS
 * IS" BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE,
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 */
/* 
 * Portions of this code were written based on pg_rast.c:
 *
 * Copyright (c) 2005 Satoshi Nagayasu <snaga@snaga.org>
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose, without fee, and without a
 * written agreement is hereby granted, provided that the above
 * copyright notice and this paragraph and the following two
 * paragraphs appear in all copies.
 *
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING
 * LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS
 * DOCUMENTATION, EVEN IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS
 * IS" BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE,
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include "postgres.h"
#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>
#include <mb/pg_wchar.h>
#include "access/genam.h"
#include "access/heapam.h"
#include "pgsenna2.h"
#include "access/xlogutils.h"
#include "catalog/index.h"
#include "catalog/catalog.h"
#include "storage/smgr.h"
#include "storage/lock.h"
#include "commands/vacuum.h"
#include "executor/executor.h"
#include "miscadmin.h"
#include "optimizer/cost.h"
#include "optimizer/restrictinfo.h"
#include "nodes/relation.h"
#include "storage/itemptr.h"
#include "utils/guc.h"
#include "funcapi.h"
#include "utils/relcache.h"

#if defined(POSTGRES81) && defined(WIN32)
extern DLLIMPORT char *DataDir; 
extern DLLIMPORT double cpu_operator_cost;
#endif /* defined(POSTGRES81) && defined(WIN32) */

#if defined(POSTGRES82) || defined(POSTGRES83)
#include "access/reloptions.h"
PG_MODULE_MAGIC;
#endif

PG_FUNCTION_INFO_V1(pgs2insert);
PG_FUNCTION_INFO_V1(pgs2beginscan);
PG_FUNCTION_INFO_V1(pgs2gettuple);
PG_FUNCTION_INFO_V1(pgs2getmulti);
PG_FUNCTION_INFO_V1(pgs2rescan);
PG_FUNCTION_INFO_V1(pgs2endscan);
PG_FUNCTION_INFO_V1(pgs2markpos);
PG_FUNCTION_INFO_V1(pgs2restrpos);
PG_FUNCTION_INFO_V1(pgs2build);
PG_FUNCTION_INFO_V1(pgs2buildb);
PG_FUNCTION_INFO_V1(pgs2buildu);
PG_FUNCTION_INFO_V1(pgs2bulkdelete);
PG_FUNCTION_INFO_V1(pgs2vacuumcleanup);
PG_FUNCTION_INFO_V1(pgs2costestimate);
PG_FUNCTION_INFO_V1(pgs2contain);
PG_FUNCTION_INFO_V1(pgs2nop);
PG_FUNCTION_INFO_V1(pgs2getscore);
PG_FUNCTION_INFO_V1(pgs2getscore_simple);
PG_FUNCTION_INFO_V1(pgs2getnhits);
#if defined(POSTGRES82) || defined(POSTGRES83)
PG_FUNCTION_INFO_V1(pgs2options);
#endif
PG_FUNCTION_INFO_V1(pgs2indexcache);


inline static void *scan_stat_open(sen_records *records, index_info *ii);
inline static void scan_stat_close(void *opaque);

static index_info *index_cache;
static index_info *last_used_cache;
static scan_stat *curr_scan_stat;
static int last_nhits = 0;
static int max_n_index_cache = MAX_N_INDEX_CACHE;


static void
pgs2_logger_func(int level, const char *time, const char *title,
                 const char *msg, const char *location, void *func_arg)
{
  const char slev[] = " EACewnid-";
  elog(LOG, "pgsenna2: |%c|%s %s %s", *(slev + level), title, msg, location);
}

static sen_logger_info pgs2_sen_logger_info = {sen_log_warning,
                                               SEN_LOG_TIME | SEN_LOG_MESSAGE,
                                               pgs2_logger_func};

void __attribute__ ((constructor)) pgs2_init(void) {
  sen_rc rc;
  if (sen_logger_info_set(&pgs2_sen_logger_info)) {
    elog(INFO, "pgsenna2: cannot logger_info_set.");
  } else {
    elog(DEBUG1, "pgsenna2: logger_info_set.");
  }
  rc = sen_init();
  if (rc != sen_success) {
    elog(ERROR, "pgsenna2: sen_init() failed %d", rc);
  }
};

void
sen_check_init(void)
{
  static int initialized = 0;
  int i;
  index_info *ii;
  const char *option;

  if (initialized) { return ; }
#ifndef __GNUC__
  pgs2_init();
#endif /* __GNUC__ */
  option = GetConfigOption("ludia.max_n_index_cache");
  if (option) {
    max_n_index_cache = atoi(option);
    if(max_n_index_cache <= 0) {
      elog(ERROR, "pgsenna2: value of max_n_index_cache is invalid: %d", max_n_index_cache);
    }
  } else {
    elog(DEBUG1, "pgsenna2: value of max_n_index_cache = %d", max_n_index_cache);
  }
  index_cache = (index_info*)malloc(max_n_index_cache * sizeof(*index_cache));
  last_used_cache = NULL;
  for (i = max_n_index_cache, ii = index_cache; i; i--, ii++) {
    ii->namespace = 0;
    ii->relid = 0;
    ii->name[0] = '\0';
    ii->index = NULL;
    ii->using = 0;
    ii->s = NULL;
    ii->previous = NULL;
    ii->next = NULL;
  }
  initialized++;
}


inline static void
index_list_update(index_info *ii) {
  if (last_used_cache == ii) {
    return;
  }
  if (ii->previous) {
    ((index_info *)ii->previous)->next = ii->next;
  }
  if (ii->next) {
    ((index_info *)ii->next)->previous = ii->previous;
  }
  if (last_used_cache) {
    last_used_cache->previous = ii;
  }
  ii->next = last_used_cache;
  ii->previous = NULL;
  last_used_cache = ii;
}


/* todo: SizeOfIptrData may be more suitable.  */
#define KEY_SIZE (sizeof(ItemPointerData))

index_info *
index_info_open(Relation r, int createp, int flags)
{
  int i;
  index_info *ii, *cand;
  char path[MAXPGPATH];
  Oid namespace = RelationGetNamespace(r);
  Oid relid = RelationGetRelid(r);
  char *name = RelationGetRelationName(r);
  sen_check_init();
  ii = last_used_cache;
  for (i = max_n_index_cache - 1; i; i--) {
    if(!ii) {
      break;
    } else {
      if (ii->index && 
          namespace == ii->namespace &&
          relid == ii->relid && 
          !strcmp(ii->name, name)) {
        if (createp) { /* for REINDEX (and CLUSTER) */
          break;
        } else {
          index_list_update(ii);
          return ii;
        }
      } else {
        ii = ii->next;
      }
    }
  }
  cand = ii;  // cand is oldest or NULL
  if (!cand) {
    for (i = max_n_index_cache, ii = index_cache; i; i--, ii++) {
      if (!ii->using) {
        cand = ii;
        break;
      }
    }
  }
  if (!cand) {
    elog(ERROR, "pgsenna2: n of indices in use reached max(%d)", max_n_index_cache);
  }
  index_list_update(cand);

  sen_index_close(cand->index);
  if(cand->s && cand->s->opaque) {
    scan_stat_close(cand->s->opaque);
    cand->s->opaque = NULL;
    cand->s = NULL;
  }
  cand->namespace = namespace;
  cand->relid = relid;
  strcpy(cand->name, name);
  {
    char *pathname;
    RelationOpenSmgr(r);
    pathname = relpath(r->rd_smgr->smgr_rnode);
    snprintf(path, MAXPGPATH, "%s/%s", DataDir, pathname);
    RelationCloseSmgr(r);
    pfree(pathname);
  }
  if (createp) {
    int initial_n_segments = INITIAL_N_SEGMENTS;
    sen_encoding encoding;
    const char *option;
    switch (GetDatabaseEncoding()) {
    case PG_UTF8:
      encoding = sen_enc_utf8;
      break;
    case PG_EUC_JP:
      encoding = sen_enc_euc_jp;
      break;
    case PG_SJIS:
      encoding = sen_enc_sjis;
      break;
    default:
      encoding = sen_enc_default;
    }
    sen_index_remove(path);
    option = GetConfigOption("ludia.initial_n_segments");
    if (option) {
      initial_n_segments = atoi(option);
      if(initial_n_segments <= 0) {
        elog(ERROR, "pgsenna2: value of initial_n_segments is invalid: %d", initial_n_segments);
      }
    } else {
      elog(DEBUG1, "pgsenna2: value of initial_n_segments = %d", initial_n_segments);
    }
    cand->index = sen_index_create(path,
                                   KEY_SIZE,
                                   flags, initial_n_segments, encoding);
    if (!cand->index) {
      elog(ERROR, "pgsenna2: index create failed (%s)", path);
    }
  } else {
    cand->index = sen_index_open(path);
    if (!cand->index) {
      elog(ERROR, "pgsenna2: index open failed (%s)", path);
    }
  }
  cand->using = 1;
  return cand;
}


void
index_info_close(index_info *ii)
{
  /* ii->using = 0; */
}


inline static void *
scan_stat_open(sen_records *records, index_info *ii)
{
  scan_stat *ss = palloc(sizeof(scan_stat));
  ss->records = records;
  ss->index = ii;
  ss->mark = NULL;
  curr_scan_stat = ss;
  last_nhits = sen_records_nhits(records);
  return (void *) ss;
}


inline static void
scan_stat_close(void *opaque)
{
  scan_stat *ss = (scan_stat *) opaque;
  if (ss) {
    if (ss->records) { sen_records_close(ss->records); }
    if (ss->index) { index_info_close(ss->index); }
    pfree(ss);
  }
  curr_scan_stat = NULL;
}


inline static void
scan_stat_close_all()
{
  int i = 0;
  index_info *ii;
  for (i = max_n_index_cache, ii = index_cache; i; i--, ii++) {
    if (!ii->using) {
      return;
    }
    if(ii->s && ii->s->opaque) {
      scan_stat_close(ii->s->opaque);
      ii->s->opaque = NULL;
      ii->s = NULL;
    }
  }
}


inline static void
init_support_funcs(support_funcs *funcs, Relation index)
{
  fmgr_info_copy(&funcs->filterFn,
                 index_getprocinfo(index, 1, PGS2_FILTER_PROC),
                 CurrentMemoryContext);
}


inline static void
do_insert(Relation r, ItemPointer tid, index_info *ii,
          Datum *values, bool *isnull, support_funcs *funcs)
{
  char *c;
  text *t;
  sen_rc rc = sen_success;
  Datum value;
  int i, natts = RelationGetNumberOfAttributes(r);
  for (i = 0; i < natts; i ++) {
    if (!isnull[i]) {
      value = FunctionCall1(&funcs->filterFn, PointerGetDatum(values[i]));
      t = DatumGetTextP(value);
      c = text2cstr(t);
      if (c && *c) {
        LOCKTAG locktag;
        LockAcquireResult res = LOCKACQUIRE_OK;

        SET_LOCKTAG_RELATION(locktag, MyDatabaseId, ii->relid);
#if defined(POSTGRES82) || defined(POSTGRES83)
        res = LockAcquire(&locktag, ShareUpdateExclusiveLock, 0, 0);
#else
        res = LockAcquire(DEFAULT_LOCKMETHOD, &locktag, r->rd_istemp,
                          ShareUpdateExclusiveLock, 0, 0);
#endif
        if(res == LOCKACQUIRE_OK) {
          rc = sen_index_upd(ii->index, (void *)tid, NULL, 0, c, strlen(c));
        } else {
          elog(ERROR, "pgsenna2: cannot LockAcquire while do_insert (%d)", res);
        }
        if (rc != sen_success) {
          elog(ERROR, "pgsenna2: sen_index_upd failed while do_insert (%d)", rc);
        }
#if defined(POSTGRES82) || defined(POSTGRES83)
        LockRelease(&locktag, ShareUpdateExclusiveLock, 0);        
#else
        LockRelease(DEFAULT_LOCKMETHOD, &locktag,
                    ShareUpdateExclusiveLock, 0);
#endif
      } else {
        if (!(sen_sym_get(ii->index->keys, (void *)tid))) { rc = sen_other_error; }
      }
      pfree(c);
      if (t != (text *) value) {
        elog(DEBUG1, "pgsenna2: palloced when untoasted (%p)", t);
        pfree(t);
      }
      if (value != values[i]) {
        elog(DEBUG1, "pgsenna2: palloced in filter function (%p)", DatumGetPointer(value));
        pfree((text *)value);
      }
      if (rc) {
        elog(ERROR, "pgsenna2: insert failed (%d)", rc);
      }
    }
  }
}


static void
buildCallback(Relation index,
              HeapTuple htup,
              Datum *values,
              bool *isnull,
              bool tupleIsAlive,
              void *state)
{
  build_stat *bs = (build_stat *) state;
  do_insert(index, &htup->t_self, bs->index, values, isnull, &bs->funcs);
  bs->indtuples += 1;
}


Datum
pgs2build(PG_FUNCTION_ARGS)
{
  Relation  heap = (Relation) PG_GETARG_POINTER(0);
  Relation  index = (Relation) PG_GETARG_POINTER(1);
  IndexInfo  *indexInfo = (IndexInfo *) PG_GETARG_POINTER(2);
#if defined(POSTGRES82) || defined(POSTGRES83)
  IndexBuildResult *result;
#endif
  build_stat bs;
  double    reltuples;
  init_support_funcs(&bs.funcs, index);
  bs.indtuples = 0;
  bs.index = index_info_open(index, 1, SEN_INDEX_NORMALIZE);
  reltuples = IndexBuildHeapScan(heap, index, indexInfo,
                                 buildCallback, (void *) &bs);
#if defined(POSTGRES82) || defined(POSTGRES83)
  result = (IndexBuildResult *) palloc(sizeof(IndexBuildResult));
  result->heap_tuples = reltuples;
  result->index_tuples = bs.indtuples;
  index_info_close(bs.index);
  PG_RETURN_POINTER(result);
#else
  IndexCloseAndUpdateStats(heap, reltuples, index, bs.indtuples);
  index_info_close(bs.index);
  PG_RETURN_VOID();
#endif
}


Datum
pgs2buildb(PG_FUNCTION_ARGS)
{
  Relation  heap = (Relation) PG_GETARG_POINTER(0);
  Relation  index = (Relation) PG_GETARG_POINTER(1);
  IndexInfo  *indexInfo = (IndexInfo *) PG_GETARG_POINTER(2);
#if defined(POSTGRES82) || defined(POSTGRES83)
  IndexBuildResult *result;
#endif
  build_stat bs;
  double    reltuples;
  init_support_funcs(&bs.funcs, index);
  bs.indtuples = 0;
  bs.index = index_info_open(index, 1, SEN_INDEX_NGRAM|SEN_INDEX_NORMALIZE);
  reltuples = IndexBuildHeapScan(heap, index, indexInfo,
                                 buildCallback, (void *) &bs);
#if defined(POSTGRES82) || defined(POSTGRES83)
  result = (IndexBuildResult *) palloc(sizeof(IndexBuildResult));
  result->heap_tuples = reltuples;
  result->index_tuples = bs.indtuples;
  index_info_close(bs.index);
  PG_RETURN_POINTER(result);
#else
  IndexCloseAndUpdateStats(heap, reltuples, index, bs.indtuples);
  index_info_close(bs.index);
  PG_RETURN_VOID();
#endif
}


Datum
pgs2buildu(PG_FUNCTION_ARGS)
{
  Relation  heap = (Relation) PG_GETARG_POINTER(0);
  Relation  index = (Relation) PG_GETARG_POINTER(1);
  IndexInfo  *indexInfo = (IndexInfo *) PG_GETARG_POINTER(2);
#if defined(POSTGRES82) || defined(POSTGRES83)
  IndexBuildResult *result;
#endif
  build_stat bs;
  double    reltuples;
  const char *option;
  int flags;

  init_support_funcs(&bs.funcs, index);
  bs.indtuples = 0;
  option = GetConfigOption("ludia.sen_index_flags");
  if (option == NULL) {
    elog(ERROR, "pgsenna2: value of sen_index_flags is invalid: NULL.");
  }
  flags = atoi(option);
  if (flags < 0) {
    elog(ERROR, "pgsenna2: value of sen_index_flags is invalid: %d.", flags);
  }
  elog(DEBUG1, "pgsenna2: value of sen_index_flags = %d", flags);
  bs.index = index_info_open(index, 1, flags);
  reltuples = IndexBuildHeapScan(heap, index, indexInfo,
                                 buildCallback, (void *) &bs);
#if defined(POSTGRES82) || defined(POSTGRES83)
  result = (IndexBuildResult *) palloc(sizeof(IndexBuildResult));
  result->heap_tuples = reltuples;
  result->index_tuples = bs.indtuples;
  index_info_close(bs.index);
  PG_RETURN_POINTER(result);
#else
  IndexCloseAndUpdateStats(heap, reltuples, index, bs.indtuples);
  index_info_close(bs.index);
  PG_RETURN_VOID();
#endif
}


Datum
pgs2insert(PG_FUNCTION_ARGS)
{
  Relation  r = (Relation) PG_GETARG_POINTER(0);
  Datum     *values = (Datum *) PG_GETARG_POINTER(1);
  bool     *isnull = (bool *) PG_GETARG_POINTER(2);
  ItemPointer ht_ctid = (ItemPointer) PG_GETARG_POINTER(3);
  support_funcs    funcs;
  index_info *ii;
  init_support_funcs(&funcs, r);
  /*
   * currently, pgsenna2 is not marked "amconcurrent" in pg_am,
   * caller should have acquired exclusive lock on index
   * relation.
   */
  ii = index_info_open(r, 0, 0);
  do_insert(r, ht_ctid, ii, values, isnull, &funcs);
  index_info_close(ii);
  PG_RETURN_BOOL(true);
}


Datum
pgs2bulkdelete(PG_FUNCTION_ARGS)
{
#if defined(POSTGRES82) || defined(POSTGRES83)
  IndexVacuumInfo *info = (IndexVacuumInfo *) PG_GETARG_POINTER(0);
  //IndexBulkDeleteResult *stats = (IndexBulkDeleteResult *) PG_GETARG_POINTER(1);
  IndexBulkDeleteCallback callback = (IndexBulkDeleteCallback) PG_GETARG_POINTER(2);
  void *callback_state = (void *) PG_GETARG_POINTER(3);
  Relation r = info->index;
#else
  Relation r = (Relation) PG_GETARG_POINTER(0);
  IndexBulkDeleteCallback callback = (IndexBulkDeleteCallback) PG_GETARG_POINTER(1);
  void *callback_state = (void *) PG_GETARG_POINTER(2);
#endif
  double tuples_removed = 0;
  IndexBulkDeleteResult *result;
  index_info *ii = index_info_open(r, 0, 0);

  sen_id id;
  ItemPointerData tid;
  unsigned int nrecords = sen_sym_size(ii->index->keys);
  Relation hrel = heap_open(r->rd_index->indrelid, NoLock);
  HeapTupleData htup;
  Buffer hbuffer;
  IndexInfo* indexInfo = BuildIndexInfo(r);
  EState *estate = CreateExecutorState();
  TupleTableSlot *slot = MakeSingleTupleTableSlot(RelationGetDescr(hrel));

  elog(DEBUG1, "pgsenna2: pgs2bulkdelete n=%d", nrecords);
  
  if (indexInfo->ii_Expressions != NIL &&
      indexInfo->ii_ExpressionsState == NIL) {
    indexInfo->ii_ExpressionsState = (List *)ExecPrepareExpr((Expr *)indexInfo->ii_Expressions, estate);
  }

  for (id = 0;;) {
    if ((id = sen_sym_next(ii->index->keys, id)) == SEN_SYM_NIL) {
      break;
    }
    if (!sen_sym_key(ii->index->keys, id, (void *) &tid, KEY_SIZE)) {
      elog(ERROR, "pgsenna2: sen_sym_key failed.");
    }
    if (callback(&tid, callback_state)) {
      htup.t_self = tid;
      if (heap_fetch(hrel, SnapshotAny, &htup, &hbuffer, false, NULL)) {
        (GetPerTupleExprContext(estate))->ecxt_scantuple = slot;
        update_index_with_tuple(indexInfo, ii, estate, &htup, NULL);
      }
      ReleaseBuffer(hbuffer);
    }
  }
  FreeExecutorState(estate);
  ExecDropSingleTupleTableSlot(slot);
  heap_close(hrel, NoLock);

  result = (IndexBulkDeleteResult *) palloc0(sizeof(IndexBulkDeleteResult));
  result->num_pages = 1;
  result->num_index_tuples = sen_sym_size(ii->index->keys);
  result->tuples_removed = tuples_removed;
  PG_RETURN_POINTER(result);
}


Datum
pgs2vacuumcleanup(PG_FUNCTION_ARGS)
{
  /* IndexVacuumInfo *info = (IndexVacuumInfo *) PG_GETARG_POINTER(0); */
  IndexBulkDeleteResult *stats = (IndexBulkDeleteResult *) PG_GETARG_POINTER(1);

  /* under construction... */
  
  PG_RETURN_POINTER(stats);
}


Datum
pgs2beginscan(PG_FUNCTION_ARGS)
{
  Relation  r = (Relation) PG_GETARG_POINTER(0);
  int      nkeys = PG_GETARG_INT32(1);
  ScanKey    key = (ScanKey) PG_GETARG_POINTER(2);
  IndexScanDesc s;
  s = RelationGetIndexScan(r, nkeys, key);
  PG_RETURN_POINTER(s);
}


Datum
pgs2gettuple(PG_FUNCTION_ARGS)
{
  IndexScanDesc s = (IndexScanDesc) PG_GETARG_POINTER(0);
  ScanDirection dir = (ScanDirection) PG_GETARG_INT32(1);
  scan_stat *ss = s->opaque;
  sen_rc rc = sen_success;
  int res;

  if (dir != 1) {
    elog(WARNING, "pgsenna2: dir(%d) assigned in pgs2gettuple()", dir);
  }

  if (!ss) {
    scan_stat_close_all();
    elog(ERROR, "pgsenna2: inconsistent scan, or max_n_index_cache is too small");
  }
  if (!ss->records) {
    elog(ERROR, "pgsenna2: inconsistent scan");
  }

  if (s->kill_prior_tuple) {
#ifndef POSTGRES83
    rc = sen_index_del(ss->index->index, &(s->currentItemData));
#endif
    if (rc) {
      elog(WARNING, "pgsenna2: sen_index_del failed(%d)", rc);
    }
  }

  res = sen_records_next(ss->records, &s->xs_ctup.t_self, KEY_SIZE, NULL);
#ifndef POSTGRES83
  if (res) { memcpy(&s->currentItemData, &s->xs_ctup.t_self, KEY_SIZE); }
#endif
  PG_RETURN_BOOL(res ? true : false);
}


Datum
pgs2getmulti(PG_FUNCTION_ARGS)
{
  IndexScanDesc s = (IndexScanDesc) PG_GETARG_POINTER(0);
  ItemPointer tids = (ItemPointer) PG_GETARG_POINTER(1);
  int32 max_tids = PG_GETARG_INT32(2);
  int32 *returned_tids = (int32 *) PG_GETARG_POINTER(3);
  int32 i;
  scan_stat *ss = s->opaque;
  bool more = true;
  if (!ss) {
    elog(ERROR, "pgsenna2: inconsistent scan, or max_n_index_cache is too small");
  }
  if (!ss->records) {
    elog(ERROR, "pgsenna2: inconsistent scan");
  }
  for (i = 0; i < max_tids; i++) {
    if (!sen_records_next(ss->records, &tids[i], KEY_SIZE, NULL)) {
      more = false;
      break;
    }
#ifndef POSTGRES83
    memcpy(&s->currentItemData, &tids[i], KEY_SIZE);
#endif
  }
  *returned_tids = i;
  PG_RETURN_BOOL(more);
}


Datum
pgs2rescan(PG_FUNCTION_ARGS)
{
  IndexScanDesc s = (IndexScanDesc) PG_GETARG_POINTER(0);
  ScanKey    key = (ScanKey) PG_GETARG_POINTER(1);
  int i;
  sen_rc rc;
  sen_query *q;
  sen_records *r;
  index_info *ii;
  sen_encoding encoding;
  sen_sel_operator op;
  const char *option;
  int max_n_sort_result = MAX_N_SORT_RESULT;

  if (!key) { elog(ERROR, "pgsenna2: access method does not support scan without scankey."); }

#ifndef POSTGRES83
  ItemPointerSetInvalid(&s->currentItemData);
  ItemPointerSetInvalid(&s->currentMarkData);
#endif

  scan_stat_close(s->opaque);

  ii = index_info_open(s->indexRelation, 0, 0);
  rc = sen_index_info(ii->index, NULL, NULL, NULL, &encoding, NULL, NULL, NULL, NULL, NULL, NULL);
  if (rc) {
    elog(ERROR, "pgsenna2: index_info failed(%d)", rc);
  }

  elog(DEBUG1, "pgsenna2: nok=%d ign=%d", s->numberOfKeys, s->ignore_killed_tuples);

  if (key && s->numberOfKeys > 0) {
    memmove(s->keyData, key, s->numberOfKeys * sizeof(ScanKeyData));
  }
  r = sen_records_open(sen_rec_document, sen_rec_none, 0);
  if (!r) {
    elog(ERROR, "pgsenna2: sen_records_open failed");
  }

  r->ignore_deleted_records = s->ignore_killed_tuples ? 1 : 0;

  for (i = 0, op = sen_sel_or; i < s->numberOfKeys; i++, op = sen_sel_and) {
    char *c;
    elog(DEBUG1,
         "pgsenna2: sk_flags=%d,atn=%d,strategy=%d,subtype=%d,argument=%p",
         s->keyData[i].sk_flags,
         s->keyData[i].sk_attno,
         s->keyData[i].sk_strategy,
         s->keyData[i].sk_subtype,
         DatumGetPointer(s->keyData[i].sk_argument));
    if(s->keyData[i].sk_argument != 0) {
      c = text2cstr(DatumGetTextP(s->keyData[i].sk_argument));
      if ((q = sen_query_open(c, strlen(c), sen_sel_or,
                              PGS2_MAX_N_EXPRS, encoding))) {
        if (sen_query_rest(q, NULL)) {
          elog(WARNING, "pgsenna2: too many expressions (%d)",
               sen_query_rest(q, NULL));
        }
        rc = sen_query_exec(ii->index, q, r, op);
        sen_query_close(q);
      }
      pfree(c);
    }
    if (rc) {
      elog(ERROR, "pgsenna2: sen_query_exec failed(%d)", rc);
    }
  }
  option = GetConfigOption("ludia.max_n_sort_result");
  if (option) {
    max_n_sort_result = atoi(option);
    if (max_n_sort_result < -1) {
      elog(ERROR, "pgsenna2: value of max_n_sort_result is invalid: %d", max_n_sort_result);
    }
  } else {
      elog(DEBUG1, "pgsenna2: value of max_n_sort_result = %d", max_n_sort_result);
  }
  if (max_n_sort_result >= 0 && sen_records_nhits(r) > 0) {
    sen_records_sort(r, max_n_sort_result, NULL);
  }
  s->opaque = scan_stat_open(r, ii);
  ii->s = s;
  PG_RETURN_VOID();
}


Datum
pgs2endscan(PG_FUNCTION_ARGS)
{
  IndexScanDesc s = (IndexScanDesc) PG_GETARG_POINTER(0);
  scan_stat *ss = (scan_stat *) s->opaque;
  if (ss != NULL && ss->index != NULL) {
    ss->index->s = NULL;
  }
  scan_stat_close(ss);
  PG_RETURN_VOID();
}


Datum
pgs2markpos(PG_FUNCTION_ARGS)
{
  IndexScanDesc s = (IndexScanDesc) PG_GETARG_POINTER(0);
  scan_stat *ss = (scan_stat *) s->opaque;
  ss->mark = ss->records->curr_rec;
#ifndef POSTGRES83
  s->currentMarkData = s->currentItemData;
#endif
  PG_RETURN_VOID();
}


Datum
pgs2restrpos(PG_FUNCTION_ARGS)
{
  IndexScanDesc s = (IndexScanDesc) PG_GETARG_POINTER(0);
  scan_stat *ss = (scan_stat *) s->opaque;
  ss->records->curr_rec = ss->mark;
#ifndef POSTGRES83
  s->currentItemData = s->currentMarkData;
#endif
  PG_RETURN_VOID();
}


Datum 
pgs2costestimate(PG_FUNCTION_ARGS)
{
  PlannerInfo *root = (PlannerInfo *) PG_GETARG_POINTER(0);
  IndexOptInfo *index = (IndexOptInfo *) PG_GETARG_POINTER(1);
  List *indexQuals = (List *) PG_GETARG_POINTER(2);
#if defined(POSTGRES82) || defined(POSTGRES83)
  //  RelOptInfo *outer_rel = (RelOptInfo *) PG_GETARG_POINTER(3);
  Cost *indexStartupCost = (Cost *) PG_GETARG_POINTER(4);
  Cost *indexTotalCost = (Cost *) PG_GETARG_POINTER(5);
  Selectivity *indexSelectivity = (Selectivity *) PG_GETARG_POINTER(6);
  double *indexCorrelation = (double *) PG_GETARG_POINTER(7);
#else
  Cost *indexStartupCost = (Cost *) PG_GETARG_POINTER(3);
  Cost *indexTotalCost = (Cost *) PG_GETARG_POINTER(4);
  Selectivity *indexSelectivity = (Selectivity *) PG_GETARG_POINTER(5);
  double *indexCorrelation = (double *) PG_GETARG_POINTER(6);
#endif
  double numIndexTuples = 0.0;
  double numIndexPages = 0.0;
  QualCost index_qual_cost;
  double qual_op_cost;
  double qual_arg_cost;
  List *selectivityQuals;
  /*
    index_info *ii = index_info_open(index, 0, 0);
    unsigned int nrecords = sen_sym_size(ii->index->keys);
  */

  if (index->indpred != NIL)
    {
      List       *strippedQuals;
      List       *predExtraQuals;

      strippedQuals = get_actual_clauses(indexQuals);
      predExtraQuals = list_difference(index->indpred, strippedQuals);
      selectivityQuals = list_concat(predExtraQuals, indexQuals);
    }
  else
    selectivityQuals = indexQuals;

  *indexSelectivity = clauselist_selectivity(root, selectivityQuals,
                                             index->rel->relid,
                                             JOIN_INNER);
  if (*indexSelectivity <= 1.0) { *indexSelectivity = 0.0; }

  if (numIndexTuples <= 0.0)
    numIndexTuples = *indexSelectivity * index->rel->tuples;

  if (numIndexTuples > index->tuples)
    numIndexTuples = index->tuples;

  *indexTotalCost = numIndexPages;

#ifdef POSTGRES83
  cost_qual_eval(&index_qual_cost, indexQuals, NULL);
#else
  cost_qual_eval(&index_qual_cost, indexQuals);
#endif
  qual_op_cost = cpu_operator_cost * list_length(indexQuals);
  qual_arg_cost = index_qual_cost.startup +
    index_qual_cost.per_tuple - qual_op_cost;
  /* if (qual_arg_cost < nrecords) { qual_arg_cost = 0.0; } */
  qual_arg_cost = 0.0;
  *indexStartupCost = qual_arg_cost;
  *indexTotalCost += qual_arg_cost;
  *indexTotalCost += numIndexTuples * (cpu_index_tuple_cost + qual_op_cost);
#if defined(POSTGRES82) || defined(POSTGRES83)
  *indexTotalCost += -DEFAULT_RANDOM_PAGE_COST;
#endif
  *indexCorrelation = 1.0;
  elog(DEBUG1, "pgsenna2: cost=(%f,%f,%f)",
       *indexStartupCost, *indexTotalCost, *indexSelectivity);
  PG_RETURN_VOID();
}


Datum 
pgs2contain(PG_FUNCTION_ARGS)
{
  text *t = PG_GETARG_TEXT_P(0);
  text *q = PG_GETARG_TEXT_P(1);
  const char *option;
  char enable_seqscan[16];

  option = GetConfigOption("ludia.enable_seqscan");
  if (option) {
    strncpy(enable_seqscan, option, 16);
  } else {
    strncpy(enable_seqscan, ENABLE_SEQSCAN, 16);
  }
  if (!strcmp(enable_seqscan, "on")) {
    char *ct = text2cstr(t);
    char *cq = text2cstr(q);
    bool res = false;
    sen_encoding encoding;
    sen_query *sq;

    switch (GetDatabaseEncoding()) {
    case PG_UTF8:
      encoding = sen_enc_utf8;
      break;
    case PG_EUC_JP:
      encoding = sen_enc_euc_jp;
      break;
    case PG_SJIS:
      encoding = sen_enc_sjis;
      break;
    default:
      encoding = sen_enc_default;
    }
    if ((sq = sen_query_open(cq, strlen(cq), sen_sel_or,
                             PGS2_MAX_N_EXPRS, encoding))) {
      int found, score, seqscan_flags = SEQSCAN_FLAGS;
      unsigned int ct_strlen;
      const char *option;
      sen_rc rc;

      if (sen_query_rest(sq, NULL)) {
        elog(WARNING, "pgsenna2: too many expressions (%d)",
        sen_query_rest(sq, NULL));
      }
      option = GetConfigOption("ludia.seqscan_flags");
      if (option) {
        seqscan_flags = atoi(option);
      }
      if (seqscan_flags < 0 || 1 < seqscan_flags) {
        elog(ERROR, "pgsenna2: value of seqscan_flags is invalid: %d",
             seqscan_flags);
      }
      ct_strlen = strlen(ct);
      rc = sen_query_scan(sq, (const char **)&ct, &ct_strlen, 1,
                          seqscan_flags, &found, &score);
      if (rc != sen_success) {
        elog(ERROR, "pgsenna2: sen_query_scan failed (%d)", rc);
      }
      res = (bool) found;
      sen_query_close(sq);
    }
    pfree(ct);
    pfree(cq);
    PG_RETURN_BOOL(res);
  } else {
    ereport(ERROR, (errmsg("pgsenna2: sequencial scan disabled.")));
  }
  ereport(ERROR, (errmsg("pgsenna2: value of enable_seqscan is invalid.")));
  PG_RETURN_BOOL(false);
}


Datum 
pgs2nop(PG_FUNCTION_ARGS)
{
  PG_RETURN_POINTER(PG_GETARG_POINTER(0));
}


Datum
pgs2getscore(PG_FUNCTION_ARGS)
{
  int i, result;
  index_info *ii;
  scan_stat *ss = NULL;
  ItemPointer tid = (ItemPointer)PG_GETARG_POINTER(0);
  text *t = PG_GETARG_TEXT_P(1);
  char *indexname = text2cstr(t);
  for (i = max_n_index_cache, ii = index_cache; i; i--, ii++) {
    if (ii->index && !strcmp(ii->name, indexname)) { break; }
  }
  if (!i || !ii->s || !(ss = ii->s->opaque) || !ss->records) {
    elog(ERROR, "pgsenna2: index (%s) is not available", indexname);
  }
  result = sen_records_find(ss->records, (void *)tid);
  pfree(indexname);
  PG_RETURN_INT32(result);
}


Datum
pgs2getscore_simple(PG_FUNCTION_ARGS)
{
  int result = 0;
  ItemPointer tid = (ItemPointer)PG_GETARG_POINTER(0);
  if (curr_scan_stat) {
    result = sen_records_find(curr_scan_stat->records, (void *)tid);
  }
  PG_RETURN_INT32(result);
}


Datum
pgs2getnhits(PG_FUNCTION_ARGS)
{
  PG_RETURN_INT32(last_nhits);
}


#if defined(POSTGRES82) || defined(POSTGRES83)
Datum
pgs2options(PG_FUNCTION_ARGS)
{
  elog(ERROR, "pgsenna2: pgs2options is called");
  /*
  Datum      reloptions = PG_GETARG_DATUM(0);
  bool       validate = PG_GETARG_BOOL(1);
  bytea      *result;

  result = default_reloptions(reloptions, validate,
                              PGS2_MIN_FILLFACTOR,
                              PGS2_DEFAULT_FILLFACTOR);
  if (result)
    PG_RETURN_BYTEA_P(result);
  */
  PG_RETURN_NULL();
}
#endif


Datum
pgs2indexcache(PG_FUNCTION_ARGS)
{
  int n = 0;

  FuncCallContext     *funcctx;
  int                  call_cntr;
  int                  max_calls;
  TupleDesc            tupdesc;
  AttInMetadata       *attinmeta;
  index_info **last_used_cache_array;
  
  if (SRF_IS_FIRSTCALL()) {
    index_info *tmp_used_cache;
    MemoryContext   oldcontext;
    funcctx = SRF_FIRSTCALL_INIT();
    oldcontext = MemoryContextSwitchTo(funcctx->multi_call_memory_ctx);
    if (get_call_result_type(fcinfo, NULL, &tupdesc) != TYPEFUNC_COMPOSITE)
      ereport(ERROR,
              (errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
               errmsg("pgsenna2: function returning record called in context "
                      "that cannot accept type record")));
    attinmeta = TupleDescGetAttInMetadata(tupdesc);
    funcctx->attinmeta = attinmeta;

    last_used_cache_array = (index_info **)palloc(max_n_index_cache * sizeof(index_info *));
    tmp_used_cache = last_used_cache;
    while(tmp_used_cache) {
      last_used_cache_array[n] = tmp_used_cache;
      tmp_used_cache = tmp_used_cache->next;
      n++;
      if(n > max_n_index_cache) {
        elog(WARNING, "pgsenna2: warning happened at index cache");
        break;
      }
    }
    funcctx->user_fctx = (void *)last_used_cache_array;
    funcctx->max_calls = n;
    MemoryContextSwitchTo(oldcontext);
  }

  funcctx = SRF_PERCALL_SETUP();

  last_used_cache_array = (index_info **) funcctx->user_fctx;
  
  call_cntr = funcctx->call_cntr;
  max_calls = funcctx->max_calls;
  attinmeta = funcctx->attinmeta;
 
  if (call_cntr < max_calls) {
    int i;
    char       **values;
    HeapTuple    tuple;
    Datum        result;
    
    values = (char **) palloc(12 * sizeof(char *));
    for (i = 0; i < 6; i++) {
      values[i] = (char *) palloc(64 * sizeof(char));
    }
    last_used_cache_array = (index_info **) funcctx->user_fctx;
    snprintf(values[0], 64, "%d", last_used_cache_array[call_cntr]->relid);
    snprintf(values[1], 64, "%d", last_used_cache_array[call_cntr]->namespace);
    snprintf(values[2], 64, "%s", last_used_cache_array[call_cntr]->name);
    snprintf(values[3], 64, "%p", last_used_cache_array[call_cntr]->previous);
    snprintf(values[4], 64, "%p", last_used_cache_array[call_cntr]);
    snprintf(values[5], 64, "%p", last_used_cache_array[call_cntr]->next);
    tuple = BuildTupleFromCStrings(attinmeta, values);
    result = HeapTupleGetDatum(tuple);
    for (i = 0; i < 6; i++) {
      pfree(values[i]);
    }
    pfree(values);

    SRF_RETURN_NEXT(funcctx, result);
  } else {
    SRF_RETURN_DONE(funcctx);
  }
}


char* get_tuple_text(ExprContext *econtext,
                     HeapTuple tup,
                     int attnum,
                     ExprState *estate)
{
  Datum iDatum;
  text *t;
  char *c;
  bool isNull;
  ExecStoreTuple(tup, econtext->ecxt_scantuple, InvalidBuffer, false);
  if (!estate)
    iDatum = slot_getattr(econtext->ecxt_scantuple, attnum, &isNull);
  else
    iDatum = ExecEvalExprSwitchContext(estate, econtext, &isNull, NULL);
  if (isNull)
    {
      c = NULL;
    }
  else
    {
      t = DatumGetTextP(iDatum);
      c = text2cstr(t);
      if (t != (text *) iDatum)
        pfree(t);
    }
  ExecClearTuple(econtext->ecxt_scantuple);
  return c;
}


unsigned int update_index_with_tuple(IndexInfo* indexInfo,
                                     index_info *ii,
                                     EState *estate,
                                     HeapTuple oldtup,
                                     HeapTuple newtup)
{
  ExprContext *econtext = GetPerTupleExprContext(estate);
  sen_rc rc = sen_success;
  int j;
  unsigned int num_records = 0;
  ListCell *indexpr_item = list_head(indexInfo->ii_ExpressionsState);
          
  for (j = 0; j < indexInfo->ii_NumIndexAttrs; j++)
    {
      int keycol = indexInfo->ii_KeyAttrNumbers[j];
      char *oldc = NULL;
      char *newc = NULL;

      if (keycol != 0)
        { /* plain index */
          if (oldtup)
            oldc = get_tuple_text(econtext, oldtup, keycol, NULL);
          if (newtup)
            newc = get_tuple_text(econtext, newtup, keycol, NULL);
        }
      else 
        { /* index on expressions */
          if (indexpr_item == NULL)
            elog(ERROR, "pgsenna2: wrong number of index expressions");
          if (oldtup)
            oldc = get_tuple_text(econtext, oldtup, 0, (ExprState *)lfirst(indexpr_item));
          if (newtup)
            newc = get_tuple_text(econtext, newtup, 0, (ExprState *)lfirst(indexpr_item));
          indexpr_item = lnext(indexpr_item);
        }
      if (oldc || newc)
        {
          LOCKTAG locktag;
          LockAcquireResult res = LOCKACQUIRE_OK;

          SET_LOCKTAG_RELATION(locktag, MyDatabaseId, ii->relid);
#if defined(POSTGRES82) || defined(POSTGRES83)
          res = LockAcquire(&locktag, ShareUpdateExclusiveLock, 0, 0);
#else
          /*
            res = LockAcquire(DEFAULT_LOCKMETHOD, &locktag, r->rd_istemp,
            ShareUpdateExclusiveLock, 0, 0);
          */
#endif
          if(res == LOCKACQUIRE_OK)
            {
              if (oldc)
                {
                  rc = sen_index_upd(ii->index,
                                     (void *)(&(oldtup->t_self)),
                                     oldc, strlen(oldc), NULL, 0);
                  pfree(oldc);
                  if (rc != sen_success)
                    elog(ERROR, "pgsenna2: sen_index_upd failed while bulkdelete (%d)", rc);
                }
              if (newc)
                {
                  rc = sen_index_upd(ii->index,
                                     (void *)(&(newtup->t_self)),
                                     NULL, 0, newc, strlen(newc));
                  elog(DEBUG1, "pgsenna2: relid: %d, newc: %s", ii->relid, newc);
                  pfree(newc);
                  if (rc != sen_success)
                    elog(ERROR, "pgsenna2: sen_index_upd failed while bulkdelete (%d)", rc);
                }
            }
          else
            elog(ERROR, "pgsenna2: cannot LockAcquire while bulkdelete (%d)", res);

          num_records += 1;
#if defined(POSTGRES82) || defined(POSTGRES83)
          LockRelease(&locktag, ShareUpdateExclusiveLock, 0);
#else
          /*
            LockRelease(DEFAULT_LOCKMETHOD, &locktag, 
            ShareUpdateExclusiveLock, 0);
          */
#endif
        }
    }
  return num_records;
}
